# -*- encoding: utf-8 -*-

"""
Allgemeine Utilities
"""

import time, datetime, smtplib, random, re
from decimal import Decimal
from fields import PopQuerySet
from email.Utils import formatdate
from heapq import heapify, heappop, heappush, heapreplace

from django.db.models.fields import prep_for_like_query
from django.core.mail import SafeMIMEText, DNS_NAME
from django.conf import settings
from django.db import connection
from django.db.models.query import Q
from django.utils.encoding import smart_str
from kundebunt.popkern.fields import DescriptorField

_whitespace_translation = dict.fromkeys((ord(c) for c in u"\t\n\r\f\v"), u" ")

def whitespace(s):
    """Replaces all whitespace characters with spaces.
    If s is None, it returns None.

    # >>> whitespace(u'\t\n\r\x0cXY\x0b')
    # u'    XY '
    >>> whitespace(None) is None
    True
    """
    if s is None:
        return None
    else:
        return s.translate(_whitespace_translation)

def partition_dict(fn, list):
    """Erzeugt ein dict mit der Partionierung nach Äquivalenzklassen bezüglich fn.
    Die values des dicts sind Listen, die die relative Reihenfolge in `list` erhalten.

    >>> partition_dict(lambda x: x%3, range(5)) == {0: [0, 3], 1: [1, 4], 2: [2]}
    True
    """
    result = {}
    for x in list:
        key = fn(x)
        result.setdefault(key, []).append(x)
    return result


class _ConstWrapMeta(type):
    """meta class for ConstWrap, seals the class against modifications
    and rewrites the defined variables to be class instances.
    """
    __sealed = set()
    def __init__(cls, name, bases, dct):
        type.__init__(cls, name, bases, dct)
        cls._map_to_str = {}
        for (key, value) in dct.iteritems():
            if not key.startswith("_") and not callable(value):
                # build a reverse mapping dict
                cls._map_to_str[value] = key
                # make cls.XXXX return cls(XXXX), not XXXX
                setattr(cls, key, cls(value))
        # make __setattr__ prohibit further modifications in the class
        cls.__sealed.add(cls)

    def __setattr__(cls, name, value):
        """prohibit modifications in the class once sealed
        """
        if cls in cls.__sealed:
            raise TypeError("is readonly")
        else:
            type.__setattr__(cls, name, value)

class ConstWrap(object):
    """
    Special class to wrap collections of constants, e.g. enumerations.
    Use like this:

    >>> class Weekdays(ConstWrap, int):
    ...   Mon=0
    ...   Tue=1
    ...   Wed=2

    >>> Weekdays.Tue
    Weekdays.Tue
    >>> print Weekdays.Tue
    1
    >>> Weekdays.Tue.const_name()
    'Tue'
    >>> int(Weekdays.Tue)
    1
    >>> Weekdays.Tue + 1
    2
    >>> Weekdays.Tue + 1 == Weekdays.Wed
    True
    >>> Weekdays(Weekdays.Tue + 1)
    Weekdays.Wed

    # You can work with values out of range, but
    # calling const_name results in KeyErrors:

    >>> toohigh = Weekdays(Weekdays.Wed + 1)
    >>> int(toohigh)
    3
    >>> toohigh
    Weekdays(3)
    >>> toohigh.const_name()
    Traceback (most recent call last):
    ...
    KeyError: 3

    # modification of the class or the constants is not allowed.
    Weekdays.Mon=5
    Traceback (most recent call last):
    ...
    TypeError: is readonly
    """

    __metaclass__ = _ConstWrapMeta

    def __init__(self, value):
        self.__value = value

    def const_name(self):
        return self._map_to_str[self.__value]

    def __repr__(self):
        try:
            return "%s.%s" % (self.__class__.__name__, self.const_name())
        except KeyError:
            return "%s(%r)" % (self.__class__.__name__, self.__value)

    def __getattr__(self, name):
        return getattr(self.__value, name)

    def __setattr__(self, name, value):
        """
        prohibit modifications once the value has been set.
        delegate any other access to the proxied value.
        """
        if self.__dict__:
            raise TypeError("is readonly")
        else:
            object.__setattr__(self, name, value)


def domain_sort_key(domain):
    """
    gibt den Sortierschlüssel zu einer FQDN oder einer email-Adresse zurück
    - das ist die umgekehrte Label-Liste.
    Für None wird None zurückgegeben.

    >>> domain_sort_key("service.noris.de")
    ['de', 'noris', 'service']
    >>> domain_sort_key(".service.noris.de.")
    ['de', 'noris', 'service', '']
    >>> domain_sort_key(".")
    ['']
    >>> domain_sort_key("")
    ['']
    >>> domain_sort_key(None)
    >>> domain_sort_key("local.part@eine.domain")
    ['domain', 'eine', 'local.part']
    >>> domain_sort_key("local.part@eine.domain.")
    ['domain', 'eine', 'local.part']
    >>> domain_sort_key("@eine.domain.")
    ['domain', 'eine', '']
    >>> domain_sort_key("bla@ein@e.domain")
    ['domain', 'ein@e', 'bla']
    """
    if domain==None:
        return None
    res = domain.rstrip(".").split("@",1)
    res = res[:-1] + res[-1:][0].split(".")
    res.reverse()
    return res

class Domain(object):
    """Eine Domain, die mit den Kundendomains abgeglichen ist.

    __init__() Parameter:

    domain_string
        String mit einer Domainangabe, der auseinandergenommen wird

    accessible_domains
        [Kundedomain, ...] der Domains, die zur Verfügung stehen.
        Statt Kundedomain kann ein beliebiges Objekt stehen, das __str__() unterstützt.
        Kann None sein, dann wird `kunden` ausgewertet.

    kunden
        Wird nur benutzt, wenn `accessible_domains` None ist.
        Eine [Kunde, ...], aus der die zugänglichen Domains ermittelt werden.

    Attribute:

    * subdomain - string
    * parent_domain - string
    * domainkunde - Domainkunde (Hauptdomain)
    * foreign_domain - Domainkunde
    * obsolete_domain - string
    * accessible_domains - [Kundedomain, ...], zur Anzeige sortiert

    Nach folgendem Schema:

    * `subdomain` und `parent_domain` zusammen entsprechen immer der vorgegebenen Domain
    * Ist die Domain enthalten, enthalten `domainkunde` und `subdomain`
        die Domain.mbox.mailbox_
    * Gehört die Domain einem anderen Kunden, enthalten `foreign_domain` und `subdomain`
        die Domain
    * Ist die Domain gar nicht mehr aktiv, ist die Domain in `obsolete_domain`.

    >>> from kundebunt.popkern.models import Domainkunde, Kunde
    >>> d = Domain(u"blabla.noris.de",[Domainkunde(domain=u"noris.de")])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (u'blabla', u'noris.de', <Domainkunde: noris.de>, None, None)
    >>> d
    <Domain blabla . noris.de>
    >>> d.accessible_domains
    [<Domainkunde: noris.de>]
    >>> str(d)
    'blabla.noris.de'

    >>> d = Domain(u"blabla.wrzlbrnz.local", [])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (None, u'blabla.wrzlbrnz.local', None, None, u'blabla.wrzlbrnz.local')
    >>> d
    <Domain blabla.wrzlbrnz.local (obsolete)>
    >>> str(d)
    'blabla.wrzlbrnz.local'

    >>> d = Domain(u"blznkratz.checkts.net",[])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (u'blznkratz', u'checkts.net', None, <Domainkunde: checkts.net>, None)
    >>> d
    <Domain blznkratz . checkts.net (foreign)>
    >>> str(d)
    'blznkratz.checkts.net'

    >>> Domain(u"blabla.wrzlbrnz.local")
    Traceback (most recent call last):
        ...
    AssertionError: either `accessible_domains` or `kunde` must be provided

    >>> d = Domain(u"blabla.wrzlbrnz.local", kunden=Kunde.objects.filter(name=u"POP"))
    >>> d
    <Domain blabla.wrzlbrnz.local (obsolete)>

    >>> d = Domain(u"", [])
    >>> d
    <Domain None>
    >>> str(d)
    ''

    >>> d = Domain(u".noris.de", [Domainkunde(domain=u"noris.de")])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (u'*', u'noris.de', <Domainkunde: noris.de>, None, None)
    >>> d = Domain(u".x.noris.de", [Domainkunde(domain=u"noris.de")])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (u'*.x', u'noris.de', <Domainkunde: noris.de>, None, None)
    >>> d = Domain(u".wblwz.de", [])
    >>> (d.subdomain, d.parent_domain, d.domainkunde, d.foreign_domain, d.obsolete_domain)
    (u'*', u'wblwz.de', None, None, u'.wblwz.de')

    >>> Domain('xyconsors.de',['consors.de'])
    <Domain xyconsors.de (obsolete)>
    """

    def __init__(self, domain_string, accessible_domains=None, kunden=None):
        import models
        if accessible_domains==None:
            assert kunden!=None, "either `accessible_domains` or `kunde` must be provided"
            accessible_domains = list(PopQuerySet.list_union(models.Kunde, [kunde.domainkunde_set.active() for kunde in kunden]))
        else:
            accessible_domains = list(accessible_domains)  # copy as list
        self.accessible_domains = accessible_domains
        accessible_domains.sort(key=lambda d: -len(unicode(d)))

        for cand_dom in accessible_domains:
            cand_dom_str = unicode(cand_dom)
            if domain_string==cand_dom_str or domain_string.endswith(u'.'+cand_dom_str):
                self.subdomain = domain_string[:-len(cand_dom_str)-1]
                self.domainkunde = cand_dom
                self.parent_domain = unicode(cand_dom)
                self.foreign_domain = self.obsolete_domain = None
                break
        else:
            # not found. Try to find it in all active domains
            labels = domain_string.split(u".")
            for cand_sub, cand_dom in ((u".".join(labels[:i]), u".".join(labels[i:])) for i in range(len(labels)-1)):
                dom = models.Domainkunde.objects.active().filter(domain=cand_dom)
                if dom:
                    self.subdomain = cand_sub
                    self.foreign_domain = dom.get()
                    self.parent_domain = cand_dom
                    self.domainkunde = self.obsolete_domain = None
                    break
            else:
                # not found --> it's a obsolete domain
                self.obsolete_domain = domain_string
                self.subdomain = None
                self.parent_domain = domain_string
                self.domainkunde = self.foreign_domain = None
        # Nachbearbeiten für domains, die mit '.' beginnen.
        if domain_string and domain_string[0]==u'.':
            if self.subdomain:
                assert self.subdomain[0] == u'.'
                self.subdomain = u"*" + self.subdomain
            else:
                self.subdomain = u'*'
                if self.parent_domain[0] == u'.':
                    self.parent_domain = domain_string[1:]

    def __repr__(self):
        if self.domainkunde:
            dom_type = ""
            dom = str(self.domainkunde)
        elif self.foreign_domain:
            dom_type = " (foreign)"
            dom = str(self.foreign_domain)
        elif self.obsolete_domain:
            dom_type = " (obsolete)"
            dom = self.obsolete_domain
        else:
            return "<Domain None>"
        if self.subdomain:
            subdomain = "%s . " % str(self.subdomain)
        else:
            subdomain = ""
        return "<Domain %s%s%s>" % (smart_str(subdomain), smart_str(dom), smart_str(dom_type))

    def __str__(self):
        if self.subdomain:
            subdomain = "%s." % smart_str(self.subdomain)
        else:
            subdomain = ""
        return "%s%s" % (subdomain, smart_str(self.domainkunde or self.foreign_domain or self.obsolete_domain))

class EmailAddress(object):
    """Teilt eine email-Adresse in ihre Komponenten auf. Kann auch eine reine Domain sein.

    Folgende Attribute stehen zur Verfügung:

    * localpart - string,
    * separator - "" oder "@"
    * domain - string

    >>> e = EmailAddress(u"mir@noris.de", True)
    >>> e
    <EmailAddress mir @ noris.de>
    >>> e.check_domain([u"de"])
    >>> e.checked_domain
    <Domain noris . de>
    >>> (e.subdomain, e.domainkunde, e.foreign_domain, e.obsolete_domain)
    (u'noris', u'de', None, None)
    >>> e.display_domain()
    u'noris.de'

    >>> EmailAddress(u"mir", True)
    <EmailAddress  @ mir>
    >>> EmailAddress("mir", False)
    <EmailAddress mir @ >
    >>> EmailAddress(u"", False)
    <EmailAddress  @ >
    >>> EmailAddress(None, True)
    <EmailAddress  @ >

    >>> EmailAddress((u"mir", u"blxl", u"noris.net"))
    <EmailAddress mir @ blxl.noris.net>

    >>> e = EmailAddress(u".noris.de", True)
    >>> e.check_domain([u"noris.de"])
    >>> (e.subdomain, e.domainkunde, e.foreign_domain, e.obsolete_domain)
    (u'*', u'noris.de', None, None)
    >>> e.display_domain()
    u'*.noris.de'

    >>> e = EmailAddress(u".wblwz.de", True)
    >>> e.check_domain([u"noris.de"])
    >>> (e.subdomain, e.domainkunde, e.foreign_domain, e.obsolete_domain)
    (u'*', None, None, u'.wblwz.de')
    """
    def __init__(self, address, domain_required=True):
        """Konstruktor

        Parameter:

        address_string
            Die zu zerlegende Email-Adresse. Alternativ: Das Tupel (localpart, subdomain, domain)

        domain_required
            Wenn addr kein '@' enthält und ``domain_required`` True ist, wird ``addr`` als Domain betrachtet.
            Andernfalls, wenn ``addr`` kein '@' beinhaltet, wird es als reine
            localpart-Angabe ohne Domain betrachtet.
        """
        self.checked_domain = None
        if isinstance(address, tuple):
            localpart, subdomain, domain = address
            if subdomain:
                domain = u"%s.%s" % (subdomain,domain)
            if domain:
                domain = domain.lower()
            self.localpart = localpart
            self.domain = domain
            if localpart and domain:
                self.separator = u"@"
            else:
                self.separator = u""
        elif not address:
            self.localpart = self.separator = self.domain = u""
        else:
            address = address.lower()
            try:
                (self.localpart, self.domain) = address.split(u"@",1)
                self.separator = u"@"
            except ValueError:
                self.separator = u""
                if domain_required:
                    (self.localpart, self.domain) = (u"", address)
                else:
                    (self.localpart, self.domain) = (address, u"")

    def check_domain(self, accessible_domains=None, kunden=None):
        """Gleicht die Domain mit den Domainkunden ab.

        Darauf werden checked_doman (Domain) und die Attribute von class Domain zusätzlich zugänglich.
        Parameter wie im Konstruktor von class Domain.
        """
        self.checked_domain = Domain(self.domain, accessible_domains, kunden)

    def is_checked(self):
        return self.checked_domain!=None

    subdomain = property(lambda obj: obj.checked_domain.subdomain)
    parent_domain = property(lambda obj: obj.checked_domain.parent_domain)
    domainkunde = property(lambda obj: obj.checked_domain.domainkunde)
    foreign_domain = property(lambda obj: obj.checked_domain.foreign_domain)
    obsolete_domain = property(lambda obj: obj.checked_domain.obsolete_domain)

    def display_domain(self):
        """
        gibt die Domain für Anzeigezwecke zurück, d.h. es wird eventuell ein '*' vorangestellt.
        """
        if not self.domain or self.domain[0]==u'.':
            return u'*' + self.domain
        else:
            return self.domain


    def __str__(self):
        return "%s%s%s" % (self.localpart, self.separator, self.domain)

    def __repr__(self):
        return "<EmailAddress %s @ %s>" % (self.localpart, self.domain)

def assemble_address(localpart, subdomain, parent_domain):
    """setzt eine email-adresse aus localpart, subdomain und parent_domain zusammen.
    Rückgabe; String


    >>> assemble_address("bla","sub","domain")
    'bla@sub.domain'

    >>> assemble_address("","","domain.de")
    'domain.de'

    >>> assemble_address("bla", "", "essig.sauer")
    'bla@essig.sauer'

    >>> assemble_address("mir", "service.", "noris.net")
    'mir@service.noris.net'

    >>> assemble_address("", ".", "domain.net")
    '.domain.net'

    >>> assemble_address("", ".sub", "domain.net")
    '.sub.domain.net'

    >>> assemble_address("", "", ".domain.net")
    '.domain.net'

    >>> assemble_address("", "*", "domain.net")
    '.domain.net'

    >>> assemble_address("mir", "*.sub", "domain.net")
    'mir@.sub.domain.net'

    >>> assemble_address("", "", "*.domain.net")
    '.domain.net'

    >>> assemble_address("lonely", "", "")
    'lonely@'

    >>> assemble_address("", "", "")
    ''
    """
    if subdomain and subdomain.endswith("."):
        domain = subdomain + parent_domain
    elif subdomain:
        domain = "%s.%s" % (subdomain, parent_domain)
    else:
        domain = parent_domain
    if domain and domain[0] == '*':
        domain = domain[1:]
    if localpart:
        return "%s@%s" % (localpart, domain)
    else:
        return domain

def any(iterable):
    """Gibt zurück, ob einer der Werte in iterable als wahr gewertet wird.

    >>> any([False,True])
    True
    >>> any([False, False, False])
    False
    >>> any([])
    False
    """
    for x in iterable:
        if x:
            return True
    return False

def all(iterable):
    """Gibt zurück, ob alle Werte in iterable als wahr gewertet werden.

    >>> all([True, False])
    False

    >>> all([True, True])
    True

    >>> all([])
    True
    """
    for x in iterable:
        if not x:
            return False
    return True

def glob_to_like(value):
    """Erzeugt einen Wert für SQL-Abfragen mit LIKE aus einem
    einfachen Suchausdruck, der nur '*' und '?' als Jokerzeichen kennt.

    >>> glob_to_like(u'abc*d?ef')
    u'abc%d_ef'

    >>> glob_to_like(None)

    >>> glob_to_like(u'')
    u''

    >>> glob_to_like(u'*ab*cd*ef*')
    u'%ab%cd%ef%'

    >>> glob_to_like(u'%*_*x')
    u'\\\\%%\\\\_%x'

    >>> glob_to_like(u'*?%?*??_')
    u'%_\\\\%_%__\\\\_'

    >>> glob_to_like(u'?*%*?**_')
    u'_%\\\\%%_%%\\\\_'
    """
    if value == None:
        return None
    else:
        values = [u"%".join([prep_for_like_query(s) for s in p.split(u"*")]) for p in value.split(u'?')]
        return u"_".join(values)

def glob_to_regex(value):
    """Erzeugt einen (compilierten) regex aus einem einfachen Suchausdruck,
    der nur '*' und '?' als Jokerzeichen kennt.
    
    glob_to_regex(u'').pattern
    u'^$'

    >>> glob_to_regex(u'*ab*cd*ef*').pattern
    u'^.*ab.*cd.*ef.*$'

    >>> print glob_to_regex(u'.?*..*').pattern
    ^\...*\.\..*$
    """
    if value == None:
        return None
    else:
        values = [u".*".join([re.escape(s) for s in p.split(u"*")]) for p in value.split(u'?')]
        return re.compile(u"^%s$" % u".".join(values), re.IGNORECASE)


def simple_glob_filter(qset, data, field_names):
    qn = connection.ops.quote_name
    opts = qset.model._meta
    for name in field_names:
        query = data.get(name)
        field = opts.get_field(name)
        if query not in (None, '', '*'):
            if query == '-':
                if field.rel is None or isinstance(field, DescriptorField):
                    qset = qset.filter(Q(**{'%s__isnull' % name: True})|Q(**{'%s__exact' % name: ''}))
                else:
                    qset = qset.filter(**{'%s__isnull' % name: True})
            else:
                if field.rel is None or isinstance(field, DescriptorField):
                    qset = qset.filter(**{'%s__like' % name: glob_to_like(query)})
                else:
                    # related field, use default query of related models
                    rel = field.rel
                    rel_table_alias = "%s_%s" % (rel.to._meta.db_table, name)
                    qset = qset & rel.to.objects.default_query(
                            query, qset.model,
                            "%s.%s = %s.%s" % (qn(opts.db_table), qn(field.column),
                                                  qn(rel_table_alias), qn(rel.get_related_field().column)),
                            table_suffix="_" + name)

    return qset
    
#def simple_glob_query(data, *field_names):
    #def _query_to_Q(name):
        #for name, null_query in queries.iteritems():
            #field_value = data.get(name)
            #if field_value not in (None, '', '*'):
                #if field_value == '-':
                    #return Q(value__isnull=True) | Q(value__exact='')
                #else:
                    #return Q(**{'%s__like' % name: glob_to_like(field_value))
    #q = Q()
    #for name in field_names:
        #q = q & _query_to_Q(name)
    #return q

#def apply_default_queries(qset, data, model, **field_specs):
    #for name, spec in field_specs.iteritems():
        #if isinstance(spec, tuple):
            #rel_model, join_condition = tuple
        #else:
            #rel_mode, join_condition = (spec, None)
        #field_value = data.get(name)
        #if field_value not in (None, '', '*'):
            #if field_value == '-':
                #qset = qset.filter(**{'%s__isnull' % name: True})
            #else:
                #qset = rel_model.objects.default_query(field_value, model, join_condition)

# Uhrzeiten: In der Datenbank werden bisweilen integers verwendet,
# diese repräsentieren Sekunden seit epoch (1.1.1970) in UTC.
# Im Code wird dagegen datetime.datetime in lokaler Zeit (mit
# entsprechender Zeitzonenangabe).

now = datetime.datetime.now

def time_to_str(t, format='%Y-%m-%d %H:%M'):
    if t in (None,0,u''):
        return ''
    if isinstance(t, basestring):
        try:
            t = datetime.datetime(*time.strptime(t, '%Y-%m-%d %H:%M:%S')[:6])
        except ValueError:
            return t
    return t.strftime(format)

#def time_to_datetime(t):
    #"""Verwandelt long (seconds since epoch) in datetime.datetime.
    #Wenn es schon datetime.datetime ist, ist es auch in Ordnung.
    #"""
    #if t in (None,0):
        #return None
    #elif isinstance(t, datetime.datetime):
        #return t
    #else:
        #return datetime.datetime.fromtimestamp(t)

#def time_str_to_epoch(t, format='%Y-%m-%d %H:%M'):
    #if not t:
        #return None
    #else:
        #return time.mktime(time.strptime(t, format))

def is_active(beginn, ende):
    if beginn and isinstance(beginn, (int, long)):
        beginn = datetime.datetime.fromtimestamp(beginn)
    heute = datetime.date.today()
    if beginn and beginn.date() > heute:
        return False
    if ende and isinstance(ende, (int, long)):
        ende = datetime.datetime.fromtimestamp(ende)
    if ende and ende.date() < heute:
        return False
    return True

def moneyfmt(value, places=2, curr='', sep=',', dp='.',
             pos='', neg='-', trailneg=''):
    """Convert Decimal to a money formatted string.

    places:  required number of places after the decimal point
    curr:    optional currency symbol before the sign (may be blank)
    sep:     optional grouping separator (comma, period, space, or blank)
    dp:      decimal point indicator (comma or period)
             only specify as blank when places is zero
    pos:     optional sign for positive numbers: '+', space or blank
    neg:     optional sign for negative numbers: '-', '(', space or blank
    trailneg:optional trailing minus indicator:  '-', ')', space or blank

    >>> d = Decimal('-1234567.8901')
    >>> moneyfmt(d, curr='$')
    '-$1,234,567.89'
    >>> moneyfmt(d, places=0, sep='.', dp='', neg='', trailneg='-')
    '1.234.568-'
    >>> moneyfmt(d, curr='$', neg='(', trailneg=')')
    '($1,234,567.89)'
    >>> moneyfmt(Decimal(123456789), sep=' ')
    '123 456 789.00'
    >>> moneyfmt(Decimal('-0.02'), neg='<', trailneg='>')
    '<0.02>'

    (copied from documentation of the decimal module, modified to produce '0.02' instead of '.02')
    """
    q = Decimal((0, (1,), -places))    # 2 places --> '0.01'
    sign, digits, exp = value.quantize(q).as_tuple()
    assert exp == -places
    result = []
    digits = map(str, digits)
    build, next = result.append, digits.pop
    if sign:
        build(trailneg)
    for i in range(places):
        if digits:
            build(next())
        else:
            build('0')
    build(dp)
    i = 0
    if not digits:
        build('0')
    while digits:
        build(next())
        i += 1
        if i == 3 and digits:
            i = 0
            build(sep)
    build(curr)
    if sign:
        build(neg)
    else:
        build(pos)
    result.reverse()
    return ''.join(result)

def send_mail(subject, message, from_email, recipient_list, headers=None, fail_silently=False, auth_user=None, auth_password=None):
    """
    Easy wrapper for sending a single message to a recipient list. All members
    of the recipient list will see the other recipients in the 'To' field.

    If auth_user is None, the EMAIL_HOST_USER setting is used.
    If auth_password is None, the EMAIL_HOST_PASSWORD setting is used.
    """
    return send_mass_mail([[subject, message, from_email, recipient_list, headers]], fail_silently, auth_user, auth_password)


def send_mass_mail(datatuple, fail_silently=False, auth_user=None, auth_password=None):
    """
    Given a datatuple of (subject, message, from_email, recipient_list, headers), sends
    each message to each recipient list.
    `headers` is a dict with additional headers (e.g. for Reply-To)
    Returns the number of e-mails sent.

    If from_email is None, the DEFAULT_FROM_EMAIL setting is used.
    If auth_user and auth_password are set, they're used to log in.
    If auth_user is None, the EMAIL_HOST_USER setting is used.
    If auth_password is None, the EMAIL_HOST_PASSWORD setting is used.
    Copied from django.core.mail and modified to accept `headers`
    """
    if auth_user is None:
        auth_user = settings.EMAIL_HOST_USER
    if auth_password is None:
        auth_password = settings.EMAIL_HOST_PASSWORD
    try:
        server = smtplib.SMTP(settings.EMAIL_HOST, settings.EMAIL_PORT)
        if auth_user and auth_password:
            server.login(auth_user, auth_password)
    except:
        if fail_silently:
            return
        raise
    num_sent = 0
    for subject, message, from_email, recipient_list, headers in datatuple:
        if not recipient_list:
            continue
        from_email = from_email or settings.DEFAULT_FROM_EMAIL
        msg = SafeMIMEText(message.encode('utf-8'), 'plain', settings.DEFAULT_CHARSET)
        msg['Subject'] = subject
        msg['From'] = from_email
        msg['To'] = ', '.join(recipient_list)
        msg['Date'] = formatdate()
        random_bits = str(random.getrandbits(64))
        msg['Message-ID'] = "<%d.%s@%s>" % (time.time(), random_bits, DNS_NAME)
        if headers:
            for k,v in headers.iteritems():
                msg[k] = v
        try:
            server.sendmail(from_email, recipient_list, msg.as_string())
            num_sent += 1
        except:
            if not fail_silently:
                raise
    try:
        server.quit()
    except:
        if fail_silently:
            return
        raise
    return num_sent

def smart_repr(s):
    """same as repr(s), except for unicode strings: here, the leading 'u' is removed"""
    if isinstance(s,unicode):
        return repr(s)[1:]
    else:
        return repr(s)

def _dict_helper(desc, row):
    "Returns a dictionary for the given cursor.description and result row."
    return dict(zip([col[0] for col in desc], row))

def dictfetchone(cursor):
    "Returns a row from the cursor as a dict"
    row = cursor.fetchone()
    if not row:
        return None
    return _dict_helper(cursor.description, row)

def _test():
    import doctest
    doctest.testmod()

class _MergeNode(object):
    """helper class for merge_sort(), manages a single iterator with preview.

    Attributes:
      - it: the iterator
      - key: a key function, like for sorting
      - top: the next element (valid only if not at_end)
      - at_end: end reached?
    """
    
    def __init__(self, it, key):
        self.it = it
        self.key = key
        self.at_end = False
        self._fetch_next()

    def __iter__(self):
        return self
        
    def __le__(self, other):
        if not self.at_end:
            if not other.at_end:
                return self.key(self.top) <= other.key(other.top)
            else:
                return True
        else:
            return False

    def __ge__(self, other):
        return other.__lt__(self)
    
    def next(self):
        if self.at_end:
            raise StopIteration
        result = self.top
        self._fetch_next()
        return result

    def _fetch_next(self):
        try:
            self.top = self.it.next()
        except StopIteration:
            self.at_end = True
            
    def __repr__(self):
        if not self.at_end:
            return "<MergeNode next=%r>" % self.top
        else:
            return "<MergeNode (empty)>"

def identity(x):
    return x
        
def merge_sorted(iterators, key=identity):
    """chains a list of iterators that are already pre-sorted
       such that the order is retained in the result.

    >>> list(merge_sorted([iter(range(2,5)), iter(range(3,6)), iter(range(4))], lambda x:x))
    [0, 1, 2, 2, 3, 3, 3, 4, 4, 5]

    >>> list(merge_sorted([iter(range(0)), iter(range(0))], lambda x:x))
    []

    >>> list(merge_sorted([iter(range(0)), iter(range(4))], lambda x:x))
    [0, 1, 2, 3]
    """
    itq = [_MergeNode(it, key) for it in iterators]
    itq = [it for it in itq if not it.at_end]
    heapify(itq)

    while itq:
        head = itq[0]
        r = head.next()
        if not head.at_end:
            heapreplace(itq, head)
        else:
            heappop(itq)
        yield r

from itertools import tee

def unique(iterator, key=identity):
    """skips consecutive non-unique members of an iterator.

    >>> list(unique(iter([1,2,3,3,4]), lambda x:x))
    [1, 2, 3, 4]

    list(unique(iter(range(0)), lambda x:x))
    []
    """
    it1, it2 = tee(iterator)
    it2.next()
    while True:
        cand = it1.next()
        cand_key = key(cand)
        try:
            while key(it2.next())==cand_key:
                it1.next()
            yield cand
        except StopIteration:
            yield cand
            break

if __name__ == "__main__":
    _test()
