# -*- encoding: utf-8 -*-
"""Führen des Update-Logs in updatelog

Ein Model, dessen Änderungen in `updatelog` mitgeloggt werden soll, wird von
updatelog.UpdatelogMixin UND django.db.models abgeleitet. Zusätzlich wird
in einer inneren Klasse `Updatelog` Details zum Logging festgelegt, und zwar:

    * `log_index`: Welches Feld wird als Index geführt (Default: Primary Key)
    * `log_related`: Welche per ForeignKey verbundenen Tabellen bei INSERTS und DELETES mitgeloggt
      werden, und mit 'log_index', welches Feld als Indexfeld im Update-log
      bei allen Operationen (auch UPDATE) erscheint.
    * `log_related_as`: Welcher Tabellenname in den durch log_related spezifizierten
      Einträgen benutzt wird. Default: Der normale Tabellenname.
    * `log_always': Welche Felder zusätzlich immer mitgeloggt werden, auch wenn sie sich
      nicht geändert haben (z.B. 'quelle' bei Mailrule)

Ein Beispiel:

    >>> from django.db import models
    >>> from kundebunt.popkern.updatelog import UpdatelogMixin
    >>> class Blafasel(UpdatelogMixin, models.Model):
    ...     class Updatelog:
    ...         log_index="kunde"
    ...         log_related=["kunde"]
    ...         log_related_as="faselbla"
    ...         log_always=['foo']

Das Format in der Tabelle updatelog ist recht kreativ, und die Benennung der Spalten dieser
Tabelle ist irreführend. Neben der Tabelle werden die primary keys (in der Regel eh nur einer)
und die geänderten Attribute geloggt. `log_related` muss vorhanden sein, es kann natürlich eine
leere Liste sein, also [].

* Die Attributnamen der primary keys oder eben von insert_index stehen in `indexspalten`,
  die Werte in `wert`;

* die Attributnamen der anderen geänderten Daten stehen in `datenspalten`,
  die Werte in `dwert`

Näheres in kunde/kunde/log/update und kunde/POP-Datenbank.
Ein brauchbares Minimalbeispiel ist in den Tests unter
tests/popkern/test_updatelog.py.
"""



try:
    from threading import local
except ImportError:
    # import copy of _thread_local.py from python 2.4
    from django.utils._threading_local import local

from django.dispatch import dispatcher
from django.db.models import signals
from django.db.models.fields import FieldDoesNotExist
from django.db import connection, transaction
from django.db.models.fields import AutoField
from django.utils.encoding import smart_unicode
from kundebunt.popkern.fields import DescriptorField
from kundebunt.popkern.utils import dictfetchone, unique

# modes for update_log
UPDATE = "."
INSERT = "*"
DELETE = "-"

# thread local storage for accountable person.
_log_person = local()
_log_person.person = None
_log_person.request = None

def get_person():
    if _log_person.person is None:
        _log_person.person = _log_person.request.person
    return _log_person.person

def set_person(person):
    _log_person.person = person

class UpdatelogMiddleware:
    """Diese Middleware stellt die `handelnde Person` in _log_person.person zur Verfügung"""
    def process_request(self, request):
        assert hasattr(request, 'person'), "The kundebunt updatelog middleware requires authentication middleware to be installed. Edit your MIDDLEWARE_CLASSES setting to insert 'kundebunt.kunde_auth.middleware.AuthenticationMiddleware' before this middleware."
        _log_person.request = request
        _log_person.person = None


def log_update(mode, opts, index_dict, old_values, new_values, tablename=None, log_always=()):
    """Low-Level-Funktion zum Schreiben von Log-Einträgen in updatelog.

    Parameter:

w
      * mode: UPDATE|INSERT|DELETE, was mit dem Datensatz geschieht
      * opts: Die Metainformationen des Models (für Nutzung der Deskriptoren)
      * index_dict: Dict der zu loggenden index_spalten (im Regelfall nur einer), z.B. {'id': 1}
      * old_values: Dict mit den alten Werten
      * new_values: Dict mit den neuen Werten
      * tablename: Name der Datenbank-Tabellen, nur erlaubt wenn opts None ist.
      * log_always: Welche Felder auch mitgeloggt werden sollen, wenn sie sich nicht
                    verändert haben.
    """
    from kundebunt.popkern.models import DbTabelle, Updatelog, UpdatelogSpalten
    def make_log_entry(mode, person, db_tabelle_obj, index_spalten_obj, pks, log_daten_spalten, log_daten_werte):
        """Hilfsfunktion, die jeweils eine einzelne Zeile in updatelog schreibt.
        index_spalten, log_daten_spalten sind jeweils Instanz von UpdatelogSpalten.
        Wenn etwas zu lang ist, wird ValueError geworfen."""
        dwert = "|".join(log_daten_werte)
        if len(dwert) > 65535:
            raise ValueError
        fieldname_spec = "|".join(log_daten_spalten)
        if mode != UPDATE:
            fieldname_spec = "%s|%s" % (fieldname_spec, mode)
        if len(fieldname_spec) > 255:
            raise ValueError
        daten_spalten_obj, created = UpdatelogSpalten.objects.get_or_create(namen = fieldname_spec)
        log_obj = Updatelog(person=get_person(),
                            db_tabelle = db_tabelle_obj,
                            indexspalten = index_spalten_obj,
                            datenspalten = daten_spalten_obj,
                            wert = "|".join(pks),
                            dwert = "|".join(log_daten_werte)
                            )
        log_obj.save()

    if opts:
        assert tablename==None
        tablename = opts.db_table
    else:
        assert tablename
    db_tabelle_obj, created = DbTabelle.objects.get_or_create(name=tablename, defaults={'text': ''})
    index_spalten_obj, created = UpdatelogSpalten.objects.get_or_create(namen = "|".join(index_dict.keys()))
    pks = [str(pk) for pk in index_dict.values()]
    if mode==INSERT:
        data = new_values
    else:
        data = old_values
    keys = data.keys()
    keys.sort()
    # die logwürdigen Daten herausholen, die sich verändert haben.
    # die Reihenfolge der Spalten erhalten.
    log_daten_spalten = []
    log_daten_werte = []
    for key in keys:
        if key != "timestamp":
            if key in log_always or smart_unicode(old_values.get(key), strings_only=True) != new_values.get(key):
                log_daten_spalten.append(key)
                if opts == None:
                    field=None
                else:
                    for field in opts.fields:
                        if field.column==key:
                            break
                    else:
                        raise FieldDoesNotExist('%s has no field with column named %r' % (opts.object_name, key))
                if isinstance(field, DescriptorField):
                    descr = data[key]
                    if descr==None:
                        log_daten_werte.append(u'NULL')
                    else:
                        log_daten_werte.append(field.get_descriptor(data[key]).bla)
                else:
                    log_daten_werte.append(smart_unicode(data[key]))

    # geht alles zusammen?
    try:
        make_log_entry(mode, get_person(), db_tabelle_obj, index_spalten_obj, pks, log_daten_spalten, log_daten_werte)
    except ValueError:
        # Alles einzeln
        for spalte, wert in zip(log_daten_spalten, log_daten_werte):
            if len(wert)>65535:
                wert = wert[:65532]+'...'
            make_log_entry(mode, get_person(), db_tabelle_obj, index_spalten_obj, pks, [spalte], [wert])


def _get_log_index_dict(instance, data=None):
    custom_index = getattr(instance.Updatelog, "custom_index", None)
    if custom_index is not None:
        return custom_index(instance)
    else:
        opts = instance._meta
        field_name = getattr(instance.Updatelog, "log_index", None)
        if field_name is None:
            fields = (f for f in opts.fields if f.primary_key)
        else:
            fields = (opts.get_field(field_name),)
        if data is None:
            return dict((f.column, f.get_db_prep_save(f.pre_save(instance, False)))
                        for f in fields)
        else:
            return dict((f.column, data[f.column]) for f in fields)

class UpdatelogMixin(object):
    # Eventuell ist eine Lösung mit Signalen (pre_save, post_save, pre_delete)
    # günstiger. Das ganze ist aber ohnehin recht eng mit dem Metamodell von Django
    # verquickt, und es ist problematisch, INSERTs im post_save von UPDATEs zu trennen.
    # INSERTs können aber nicht im pre_save behandelt werden, weil dann die primary keys
    # noch nicht gesetzt sind.
    def save(self):
        qn = connection.ops.quote_name
        dispatcher.send(signal=signals.pre_save, sender=self.__class__, instance=self)

        non_pks = [f for f in self._meta.fields if not f.primary_key]
        pk_fields = [f for f in self._meta.fields if f.primary_key]
        cursor = connection.cursor()

        # First, try an UPDATE. If that doesn't update anything, do an INSERT.
        pk_val = self._get_pk_val()
        if not self._meta.has_composite_primary_key:
            pk_val = [pk_val]
        pk_set = True
        for val in pk_val:
            if val is None or smart_unicode(val) == u'':
                pk_set = False
                break
        record_exists = True

        if pk_set:
            where_clause = " AND ".join("%s=%%s" % qn(f.column)
                                        for f in pk_fields)
            pk_db_values = reduce(
                list.__add__, (f.get_db_prep_lookup('exact', val)
                               for f,val in zip(pk_fields, pk_val)))

            # Determine whether a record with the primary key already exists.
            cursor.execute(
                "SELECT %s FROM %s WHERE %s"
                  % (','.join([qn(f.column) for f in non_pks]),
                     qn(self._meta.db_table),
                     where_clause),
                pk_db_values)

            # If it does already exist, do an UPDATE.
            old_data = dictfetchone(cursor)
            if old_data:
                db_values = [f.get_db_prep_save(f.pre_save(self, False)) for f in non_pks]
                cursor.execute(
                        "UPDATE %s SET %s WHERE %s"
                        % (qn(self._meta.db_table),
                           ','.join(['%s=%s' % (qn(f.column), f.db_placeholder)
                                     for f in non_pks]),
                            where_clause),
                        db_values + pk_db_values)
                for f, v in zip((f for f in self._meta.fields if f.primary_key), pk_val):
                    old_data[f.column] = v
                old_index_dict = _get_log_index_dict(self, old_data)
                new_index_dict = _get_log_index_dict(self)
                for index_dict in unique((old_index_dict, new_index_dict)):
                    log_update(UPDATE, self._meta, index_dict,
                            old_data, dict(zip([f.column for f in non_pks], db_values)),
                            log_always=getattr(self.Updatelog, 'log_always', ()))
            else:
                record_exists = False
        if not pk_set or not record_exists:
            field_names = [qn(f.column) for f in self._meta.fields if not isinstance(f, AutoField)]
            placeholders = [f.db_placeholder for f in self._meta.fields if not isinstance(f, AutoField)]
            db_values = [f.get_db_prep_save(f.pre_save(self, True)) for f in self._meta.fields if not isinstance(f, AutoField)]
            # If the PK has been manually set, respect that.
            if pk_set:
                field_names += [f.column for f in self._meta.fields if isinstance(f, AutoField)]
                placeholders += [f.db_placeholder for f in self._meta.fields if isinstance(f, AutoField)]
                db_values += [f.get_db_prep_save(f.pre_save(self, True)) for f in self._meta.fields if isinstance(f, AutoField)]
            if self._meta.order_with_respect_to:
                field_names.append(qn('_order'))
                # TODO: This assumes the database supports subqueries.
                placeholders.append('(SELECT COUNT(*) FROM %s WHERE %s = %%s)' % \
                    (qn(self._meta.db_table), qn(self._meta.order_with_respect_to.column)))
                db_values.append(getattr(self, self._meta.order_with_respect_to.attname))
            if db_values:
                cursor.execute("INSERT INTO %s (%s) VALUES (%s)" % \
                    (qn(self._meta.db_table), ','.join(field_names),
                    ','.join(placeholders)), db_values)
                if self._meta.has_auto_field and not pk_set:
                    setattr(self, self._meta.pk.attname, connection.ops.last_insert_id(cursor, self._meta.db_table, self._meta.pk.column))
            else:
                # Create a new record with defaults for everything.
                cursor.execute("INSERT INTO %s (%s) VALUES (%s)" %
                    (qn(self._meta.db_table),
                     qn(self._meta.pk.column),
                     connection.ops.pk_default_value()))
                setattr(self, self._meta.pk.attname, connection.ops.last_insert_id(cursor, self._meta.db_table, self._meta.pk.column))
            new_values = dict(zip((f.column for f in self._meta.fields if not isinstance(f, AutoField)),
                                  db_values))
            if self._meta.has_auto_field and not pk_set or not db_values:
                new_values[self._meta.pk.column] = self._get_pk_val()
            log_update(INSERT, self._meta, index_dict=_get_log_index_dict(self),
                       old_values={}, new_values=new_values)
            for rel_fieldname in getattr(self.Updatelog, "log_related", []):
                rel_field = self._meta.get_field(rel_fieldname)
                rel_pk = self._get_pk_val()
                if type(rel_pk)==tuple:
                    rel_pk = u"|".join(unicode(x) for x in rel_pk)
                log_update(INSERT, opts=None,
                           index_dict={rel_field.rel.field_name: rel_field.get_db_prep_save(rel_field.pre_save(self, False))},
                           old_values={},
                           new_values={getattr(self.Updatelog, "log_related_as", self._meta.db_table): rel_pk},
                           tablename=rel_field.rel.to._meta.db_table)
        transaction.commit_unless_managed()

        # Run any post-save hooks.
        dispatcher.send(signal=signals.post_save, sender=self.__class__, instance=self)
    save.alters_data = True


# Anfang einer Lösung mit Signalen
#def pre_save_handler(instance):
    #if has_updatelog(instance):
        #print "catched pre_save"
        #non_pks = [f for f in instance._meta.fields if not f.primary_key]
        #cursor = connection.cursor()

        ## First, try an UPDATE. If that doesn't update anything, do an INSERT.
        #pk_val = instance._get_pk_val()
        #pk_set = bool(pk_val)
        #record_exists = True
        #if pk_set:
            ## Determine whether a record with the primary key already exists.
            #cursor.execute("SELECT %s FROM %s WHERE %s=%%s LIMIT 1" %
                            #( ','.join([backend.quote_name(f.column) for f in non_pks]),
                                #backend.quote_name(self._meta.db_table),
                                #backend.quote_name(self._meta.pk.column)),
                            #[pk_val])
            ## If it does already exist, do an UPDATE.
            #old_data = dictfetchone(cursor)
            #if old_data:
                #log_update(UPDATE, self._meta.db_table, {self._meta.pk.attname: pk_val},
                           #old_data,
                           #dict(zip([f.column for f in non_pks], db_values)))
                #return
        ## this is an insert, which is handled by the post_save_handler.
        #instance.__post_save_
        #dispatcher.connect(post_save_handler, signal=signals.post_save)



def pre_delete_handler(instance):
    """Erledigt das Loggen in Updatelog im Falle von DELETE"""
    if has_updatelog(instance):
        non_pks_data = dict([(f.column, f.get_db_prep_save(f.pre_save(instance, False)))
                            for f in instance._meta.fields if not f.primary_key])
        old_data = dict(non_pks_data)
        pk_val = instance.pk
        if not type(pk_val)==tuple:
            pk_val = (pk_val,)
        for f, v in zip((f for f in instance._meta.fields if f.primary_key), pk_val):
            old_data[f.column] = v
        log_update(DELETE, instance._meta, _get_log_index_dict(instance), old_data, {})
        for rel_fieldname in instance.Updatelog.log_related:
            rel_field = instance._meta.get_field(rel_fieldname)
            rel_pk = instance._get_pk_val()
            if type(rel_pk)==tuple:
                rel_pk = u"|".join(unicode(x) for x in rel_pk)
            log_update(DELETE, opts=None,
                        index_dict={rel_field.rel.field_name: rel_field.get_db_prep_save(rel_field.pre_save(instance, False))},
                        old_values={getattr(instance.Updatelog, "log_related_as", instance._meta.db_table): rel_pk},
                        new_values={},
                        tablename=rel_field.rel.to._meta.db_table)


def has_updatelog(instance):
    """Gibt zurück, ob die Instanz in updatelog erfasst wird.

    >>> from kundebunt.popkern import models
    >>> has_updatelog(models.Person.objects.all()[0])
    True
    >>> has_updatelog(models.DbTabelle.objects.all()[0])
    False
    """
    return UpdatelogMixin in instance.__class__.__bases__

dispatcher.connect(pre_delete_handler, signal=signals.pre_delete)


#def 

    #history = chain(display_logs(models.Updatelog.objects.filter(db_tabelle=db_kunde, wert__in=kunden), 11),
                    #display_logs(models.Updatelog.objects.filter(indexspalten=models.UpdatelogSpalten.objects.get(namen='kunde'),
                                                              #wert__in=kunden), 11))

    #history = reversed([list(l[1])[0] for l in groupby(sorted(history, key=_by_log), _by_log)][-11:])


