# -*- coding: UTF-8 -*-

import re
from itertools import chain
from django import template, oldforms
from django.conf import settings
from django.utils.html import conditional_escape
from django.utils.safestring import mark_safe, mark_for_escaping
from django.utils.encoding import force_unicode
from django.template.defaultfilters import stringfilter
from kundebunt.popkern.navigation import navi_menubar, navi_path, navi_info, navi_tag_exists, navimap

register = template.Library()
navilink_re = re.compile(r"(?:navilink|naviurl|navitext|icon)\s+([a-zA-Z0-9._]+)(?:\(([a-zA-Z0-9._]+)\))?")
messages_re = re.compile(r'messages ([a-zA-Z0-9._]+)\s*\(([^)]+)\)')

@register.filter(name="as_readonly_field")
@stringfilter
def as_readonly_field(val, name=None):
    from xml.dom.minidom import parseString
    from xml.parsers.expat import ExpatError
    # is `val` already a rendered field? Then just modify it.
    # hmmm ... there should be an easier and faster way than parsing `val` ... FIXME
    # and ... shouldn't the manipulator handle such stuff?
    try:
        element = parseString(val).documentElement
        if element.nodeName == u'input':
            element.setAttribute("readonly", "readonly")
            element.setAttribute("class","vReadonlyTextField")
            return element.toxml()
    except ExpatError:
        pass
    if name == None:
        name_attr = id_attr = u""
    else:
        name_attr = u'name="%s"' % force_unicode(name)
        id_attr = u'id="id_%s"' % name
    return mark_safe(u'<input type="text" %s %s class="vReadonlyTextField" name="name_display" size="30" value="%s" readonly="readonly" />'
            % (id_attr, name_attr, conditional_escape(val)))
as_readonly_field.is_safe = True

@register.filter
@stringfilter
def field_as_password_field(val, name=None):
    """Renders a password input field. This can either take different type of field (rendered)
    and modify its type, or it can take a plain string and render it as password field.
    """
    from xml.dom.minidom import parseString
    from xml.parsers.expat import ExpatError
    # is `val` already a rendered field? Then just modify it.
    # hmmm ... there should be an easier and faster way than parsing `val` ... FIXME
    # and ... shouldn't the manipulator handle such stuff?
    try:
        element = parseString(val).documentElement
        if element.nodeName == u'input':
            element.setAttribute(u"type",u"password")
            element.setAttribute(u"class",u"vPasswordField")
            return element.toxml()
    except ExpatError:
        pass

    # obviously `val` is a plain string. Render it as a password field.
    if name == None:
        name_attr = id_attr = u""
    else:
        name_attr = u'name="%s"' % force_unicode(name)
        id_attr = u'id="id_%s"' % name
    return mark_safe(u'<input type="password" %s %s class="vPasswordField" value="%s" />'
            % (id_attr, name_attr, conditional_escape(val)))
field_as_password_field.is_safe = True


@register.filter
@stringfilter
def as_hidden_field(val, name=None):
    """Renders a hidden input field. This can either take different type of field (rendered)
    and modify its type, or it can take a plain string and render it as password field.
    """
    from xml.dom.minidom import parseString
    from xml.parsers.expat import ExpatError
    # is `val` already a rendered field? Then just modify it.
    # hmmm ... there should be an easier and faster way than parsing `val` ... FIXME
    # and ... shouldn't the manipulator handle such stuff?
    try:
        element = parseString(val).documentElement
        if element.nodeName == u'input':
            element.setAttribute(u"type",u"hidden")
            element.setAttribute(u"class",u"vHiddenField")
            return element.toxml()
    except ExpatError:
        pass

    # obviously `val` is a plain string. Render it as a hidden field.
    if name == None:
        name_attr = id_attr = u""
    else:
        name_attr = u'name="%s"' % force_unicode(name)
        id_attr = u'id="id_%s"' % name
    return mark_safe(u'<input type="hidden" %s %s class="vHiddenField" value="%s" />'
            % (id_attr, name_attr, conditional_escape(val)))
as_hidden_field.is_safe = True

@stringfilter
def _render_field_help(text):
    return mark_safe(u'<div class="field_help"><a href="#"><span class="htrigger">?</span><span class="htext">%s</span></a></div>' % conditional_escape(text))

@register.filter
def with_label(field, label=None, no_label=False):
    """Transforms a field into a table row with label and field.
    Das Ergebnis sieht etwa so aus:
    <div id="div_blabla" class="formrow">
      <div class="val_error">fehler</div>
      <div class="val_error">fehler2</div>
      <div class="advisory>problem</div>
      <div class="advisory>problem2</div>
      <div class="inner">
        <label for="id_blabla" id="lab_blabla">Label:</label>
        <input id="id_blabla" ...></input>
        <div class="field_help"><a href="#">...kompliziertes Zeug...</a></div>
      </div>
    </div>

    """
    if hasattr(field.formfield, "id_prefix"):
        id_prefix = force_unicode(field.formfield.id_prefix) + u"_"
    else:
        id_prefix = u""
    if not field:
        return u""
    try:
        if not label:
            label = force_unicode(field.field_name)
        else:
            label = force_unicode(label)
        id = u"%sid_%s" % (id_prefix, field.field_name)
    except AttributeError:
        id = u"%sid_%s" % (id_prefix, label)
    if no_label:
        rendered = u'<div class="inner">%s</div>' % force_unicode(field)
    else:
        if hasattr(field.formfield, 'help_text') and field.formfield.help_text:
            helpdiv = _render_field_help(field.formfield.help_text)
        else:
            helpdiv = u''
        if isinstance(field.formfield, oldforms.CheckboxField):
            colon= u''
        else:
            colon = u':'
        if field.formfield.is_required:
            required_class = u' required'
        else:
            required_class = u''
        rendered = u'<div class="inner"><label class="%sLabel%s" for="%s" id="%slab_%s">%s%s </label>%s%s</div>' % (field.formfield.css_class(), required_class, id, id_prefix, force_unicode(field.field_name),
         label, colon, force_unicode(field), helpdiv)
    try:
        # Fehlermeldungen, die nicht escaped werden wollen, müssen mit mark_safe markiert werden. Siehe z.B.  mailadmin.set_mailflag.
        errors = [u'<div class="val_error">%s</div>' % force_unicode(conditional_escape(msg)) for msg in field.errors()]
    except AttributeError:
        errors = []
    if errors:
        return mark_safe(u'<div class="formrow" id="%sdiv_%s">\n%s\n%s</div>\n' % (id_prefix, force_unicode(field.field_name), u"\n".join(errors), rendered))
    else:
        try:
            advisories = u"\n".join([u'<div class="advisory">%s</div>' % force_unicode(conditional_escape(msg)) for msg in field.advisories()])
        except AttributeError:
            advisories = u''
        return mark_safe(u'<div class="formrow" id="%sdiv_%s">%s%s</div>' % (id_prefix, force_unicode(field.field_name), advisories, rendered))
with_label.is_safe = True

@register.filter
def with_errors(field):
    return with_label(field, u'-', True)
with_errors.is_safe = True


class FieldHelpNode(template.Node):
    def __init__(self, nodelist_block):
        self.nodelist_block = nodelist_block

    def __repr__(self):
        return "<FieldHelpNode>"

    def __iter__(self):
        for node in self.nodelist_block:
            yield node

    def get_nodes_by_type(self, nodetype):
        nodes = []
        if isinstance(self, nodetype):
            nodes.append(self)
        nodes.extend(self.nodelist_block.get_nodes_by_type(nodetype))
        return nodes

    def render(self, context):
        nodelist = template.NodeList()
        for node in self.nodelist_block:
            nodelist.append(node.render(context))
        return _render_field_help(nodelist.render(context))

@register.tag
def fieldhelp(parser, token):
    """{% fieldhelp %}...{% endfieldhelp %} - renders text as field help
    """
    nodelist_block = parser.parse((u'endfieldhelp',))
    parser.delete_first_token()
    return FieldHelpNode(nodelist_block)
fieldhelp.is_safe = True

class MessagesNode(template.Node):
    def __init__(self, form_expr, fieldname_expr):
        self.form_expr = form_expr
        self.fieldname_expr = fieldname_expr

    def __repr__(self):
        return u"<MessagesNode: %s(%s)>" % (self.form_expr, self.fieldname_expr)

    def render(self, context):
        form = self.form_expr.resolve(context)
        field_name = self.fieldname_expr.resolve(context)
        errors = form.error_dict.get(field_name, [])
        if errors:
            return mark_safe(u'\n'.join([u'<div class="val_error">%s</div>' % force_unicode(conditional_escape(msg)) for msg in errors]))
        else:
            advisories = form.advisory_dict.get(field_name, [])
            return mark_safe(u'\n'.join([u'<div class="advisory">%s</div>' % force_unicode(conditional_escape(msg)) for msg in advisories]))

@register.tag
def messages(parser,token):
    match = messages_re.match(token.contents)
    if not match:
        raise template.TemplateSyntaxError(u"%r doesn't match definition, usage: messages form-expr(fieldname-expr)")
    form_expr, fieldname_expr = match.groups()
    return MessagesNode(parser.compile_filter(form_expr),
                        parser.compile_filter(fieldname_expr))
messages.is_safe = True

def _render_messages(errors, advisories):
    if errors:
        return mark_safe(u'\n'.join([u'<div class="val_error">%s</div>' % force_unicode(conditional_escape(msg)) for msg in errors]))
    else:
        return mark_safe(u'\n'.join([u'<div class="advisory">%s</div>' % force_unicode(conditional_escape(msg)) for msg in advisories]))

@register.filter
def escape(html):
    """
    The original version of escape() produces the string 'None' for None.
    This is overwritten here to produce the empty string.
    For anything else, returns the given HTML with ampersands, quotes and carets encoded
    """
    if html is None:
        return ''
    else:
        return mark_for_escaping(force_unicode(html))

@register.filter
def length(list):
    """
    returns len(list)
    """
    return len(list)

class NavibarNode(template.Node):
    """generates the list items for the navigation bar--without <ul>
    """
    include_submenus = False

    def render(self, context):
        path = self.get_top_items(context)
        lines = []
        for el in path[1:-1]:
            url, descr, icon, menu_list = el
            if url:
                lines.append('<li><a href="%s">%s</a></li>' % (url, force_unicode(conditional_escape(descr))))
            else:
                lines.append('<li>%s</li>' % force_unicode(conditional_escape(descr)))
        url, descr, icon, menu_list = path[-1]
        lines.append('<li class="last_navi"><a href="%s">%s</a></li>' % (url, force_unicode(conditional_escape(descr))))
        return mark_safe(u"\n".join(lines))

    def get_top_items(self, context):
        return navi_path(context)


@register.tag
def navibar(parser, token):
    return NavibarNode()
navibar.is_safe = True

class MenubarNode(template.Node):
    
    include_submenus = True

    def render_submenu(self, context, info, css_class):
        url, descr, icon, menu_list = info
        if descr == u'':
            return
        if url is None:
            yield mark_safe(u'<li class="%s">%s' % (css_class, force_unicode(conditional_escape(descr))))
        else:
            yield mark_safe(u'<li class="%s"><a href="%s">%s</a>' % (css_class, url, force_unicode(conditional_escape(descr))))
        if self.include_submenus and menu_list:
            yield mark_safe(u'<div class="yuimenu"><div class="bd">')
            if isinstance(menu_list[0], str):
                menu_list = [menu_list]
            for sublist in menu_list:
                yield mark_safe(u'<ul>')
                add_class = u' first-of-type'
                for el in sublist:
                    while True:
                        try:
                            sub_info = navi_info(el, context, add_view_context=True, for_menu=True)
                            break
                        except KeyError:
                            node = navimap.get_valid_node(el, context)
                            if node.alt_parent_tag:
                                el = node.alt_parent_tag
                            else:
                                sub_info = None
                                break
                    if sub_info is not None:
                        for line in self.render_submenu(context, sub_info, u'yuimenuitem' + add_class):
                            yield line
                        add_class = u''
                yield mark_safe(u'</ul>')
            yield mark_safe(u'</div></div>')
        yield mark_safe(u'</li>')

    def render(self, context):
        path = self.get_top_items(context)
        add_css = u" first-of-type"
        lines = []
        for el in path[:-1]:
            lines.extend(self.render_submenu(context, el, u'yuimenubaritem' + add_css))
            add_css = u''
        lines.extend(self.render_submenu(context, path[-1], u'yuimenubaritem last' + add_css))
        return mark_safe(u"\n".join(lines))

    def get_top_items(self, context):
        return navi_menubar(context)

@register.tag
def menubar(parser, token):
    return MenubarNode()
menubar.is_safe = True


class NaviTitleNode(template.Node):
    def render(self, context):
        """The title of the current page as given by the last element of navi_path"""
        s = navi_path(context)[-1][1]
        return s

@register.tag
def navititle(parser, tag):
    return NaviTitleNode()

class NaviLinkNode(template.Node):
    def __init__(self, view_tag, add_context):
        self.view_tag = view_tag
        self.add_context = add_context

    #def _navi_text(self, context):
        #if self.add_context:
            #try:
                #add_context = [self.add_context.resolve(context).navi_context()]
            #except AttributeError:
                #raise template.VariableDoesNotExist, u"'%s' needs to resolve into a model object with navi_context() method. Instead, it resolves to %r" % (self.add_context, self.add_context.resolve(context))
        #else:
            #add_context = []
        #node = navimap.get_valid_node(self.view_tag)
        #context_proxy = NaviContextProxy([context] + add_context + [_navi_view.kwargs], error_cls=KeyError)
        #try:
            #return 

    def _navi_info(self, context, error_cls=KeyError):
        if self.add_context:
            try:
                add_context = self.add_context.resolve(context).navi_context()
            except AttributeError:
                raise template.VariableDoesNotExist, u"'%s' needs to resolve into a model object with navi_context() method. Instead, it resolves to %r" % (self.add_context, self.add_context.resolve(context))
            context.update(add_context)
        try:
            return navi_info(self.view_tag, context, add_view_context=True, error_cls=error_cls)
        finally:
            if self.add_context:
                context.pop()

    def render(self, context):
        info = self._navi_info(context)
        url = info[0]
        return mark_safe(u'<a href="%s">%s</a>' % (url, force_unicode(conditional_escape(info[1]))))

@register.tag
def navilink(parser, token):
    match = navilink_re.match(token.contents)
    if not match:
        raise template.TemplateSyntaxError(u"%r doesn't match definition, usage: navilink appname.viewname[(contextobj)]")
    view_tag, add_context = match.groups()
    if not navi_tag_exists(view_tag):
        raise template.TemplateSyntaxError('"view_tag %r does not exist"' % force_unicode(view_tag))
    if add_context:
        add_context = parser.compile_filter(add_context)
    return NaviLinkNode(view_tag, add_context)
navilink.is_safe = True

class NaviUrlNode(NaviLinkNode):
    def render(self, context):
        return force_unicode(self._navi_info(context)[0])

@register.tag
def naviurl(parser, token):
    match = navilink_re.match(token.contents)
    if not match:
        raise template.TemplateSyntaxError, "%r doesn't match definition, usage: naviurl appname.viewname[(contextobj)]"
    view_tag, add_context = match.groups()
    if not navi_tag_exists(view_tag):
        raise template.TemplateSyntaxError, '"view_tag %r does not exist"' % view_tag
    if add_context:
        add_context = parser.compile_filter(add_context)
    return NaviUrlNode(view_tag, add_context)
naviurl.is_safe = True

class IconNode(NaviLinkNode):
    def render(self, context):
        target, text, icon, menu_list = self._navi_info(context, error_cls=None)
        return mark_safe('<a href="%s"><img src="%s/%s" alt="%s" title="%s" /></a>' % (target, settings.ICONS_ROOTURL, icon, conditional_escape(text), conditional_escape(text)))

@register.tag
def icon(parser, token):
    match = navilink_re.match(token.contents)
    if not match:
        raise template.TemplateSyntaxError, "%r doesn't match definition, usage: icon appname.viewname[(contextobj)]"
    view_tag, add_context = match.groups()
    if not navi_tag_exists(view_tag):
        raise template.TemplateSyntaxError, '"view_tag %r does not exist"' % view_tag
    if add_context:
        add_context = parser.compile_filter(add_context)
    return IconNode(view_tag, add_context)
icon.is_safe = True

class NaviTextNode(NaviLinkNode):
    def render(self, context):
        return conditional_escape(self._navi_info(context, error_cls=None)[1])


@register.tag
def navitext(parser, token):
    match = navilink_re.match(token.contents)
    if not match:
        raise template.TemplateSyntaxError(u"%r doesn't match definition, usage: navitext appname.viewname[(contextobj)]")
    view_tag, add_context = match.groups()
    if not navi_tag_exists(view_tag):
        raise template.TemplateSyntaxError(u'"view_tag %r does not exist"' % view_tag)
    if add_context:
        add_context = parser.compile_filter(add_context)
    return NaviTextNode(view_tag, add_context)
navitext.is_safe = True

class CleverIconNode(template.Node):
    icon_dict = {
        'accounts': {
            'buchen': 'businessman_preferences',
            'callback': 'phone_recall',
            'callingback': 'phone_recall',
            'dialin': 'phone_hang_up',
            'dialout': 'phone_pick_up',
            'evn': 'document_notebook',
            'ftp': 'upload',   #??
            'ipass': 'earth_network',
            'login': 'console',
            'mail': 'mailbox_full',
            'mysql': 'data_table',
            'pop': 'worker2',
            'ppp': 'modem',
            'readonly': 'lock_edit', # tsp: document_forbidden
            'rechnung': 'cashier',
            'rt_admin': 'wrench',
            'sms1': 'mobilephone3',
            'sms10': 'mobilephone3',
            'verwaltung': 'paperclip', # tsp: devil
            'www': 'spider', # tsp: server_earth'
            'nostac': 'compress2_red',
            'vovjhc': 'compress2_red',
            'perso': 'safe',
            'ok-no-domains': 'pin_green',
            'dienste': 'businessmen',
            'isdn': 'telephone',
            'ze_textfield': 'row',
            'pppoe': 'environment_network',
            'tdsl-zisp': 'environment_network',
            'sonstige': 'hat_gray',
        },

        'dienste': {
            'general': 'dude5',
            'nic': 'pontifex',
            'tech-c': 'hat_red', # 'dude1',
            'admin-c': 'hat_green', #sunglasses',
            'bill-c': 'hat_white',
            'zone-c': 'hat_gray',
            'dns-contact': 'signpost',
            'contact': 'receiver2', # oder doch 'businessman'?
            '--': 'lemon',
            'service': 'screwdriver',
            'monitor_contact': 'replace',
            'rtweb': 'table_sql_information',
            'monitor_voice': 'message',
            'monitor_sms': 'mobilephone1',
            'monitor_access': 'find',
            'monitor_mail': 'mail_warning',
            'mon-standort': 'oszillograph',
            'mitarbeiter': 'office_chair',
            'trouble': 'ticket_red',
            'standort': 'courthouse',
            'flags': 'flag_blue',
            'vip': 'star_yellow',
        },

        'files': {
            'up': 'arrow_up_blue',
            'chart': 'chart'
        },
    }

    path_pattern = '/cleverbridge-icons/%sx%s/shadow/%s.png'

    def __init__(self, typ_expr, name_expr, size):
        self.typ_expr = typ_expr
        self.name_expr = name_expr
        self.size = size
        
    def render(self, context):
        name = self.name_expr.resolve(context)
        basename = self.icon_dict[self.typ_expr.resolve(context)].get(name)
        size = self.size
        if basename:
            return mark_safe('<img src="%s" alt="%s" title="%s" />' %
                        (self.path_pattern % (size, size, basename), conditional_escape(name), conditional_escape(name)))
        else:
            return conditional_escape(name)

    def __repr__(self):
        return '%s(%s,%s)' % (self.__class__.__name__, self.typ_expr, self.name_expr)

@register.tag
def clevericon(parser, token):
    bits = token.contents.split()
    if len(bits) not in (3,4):
        raise template.TemplateSyntaxError, "%r doesn't match definition, usage: clevericon typ name [size]"
    typ_expr = parser.compile_filter(bits[1])
    name_expr = parser.compile_filter(bits[2])
    if len(bits)==4:
        size = parser.compile_filter(bits[3])
    else:
        size = 24
    return CleverIconNode(typ_expr, name_expr, size)
clevericon.is_safe = True

class IndexNode(template.Node):
    def __init__(self, map_expr, index_expr, newvar, nodelist_block):
        self.map_expr = map_expr
        self.index_expr = index_expr
        self.newvar = newvar
        self.nodelist_block = nodelist_block

    def __repr__(self):
        return u"<IndexNode: index %s by %s as %s>" % (
            self.map_expr, self.index_expr, self.newvar)

    def __iter__(self):
        for node in self.nodelist_block:
            yield node

    def get_nodes_by_type(self, nodetype):
        nodes = []
        if isinstance(self, nodetype):
            nodes.append(self)
        nodes.extend(self.nodelist_block.get_nodes_by_type(nodetype))
        return nodes

    def render(self, context):
        nodelist = template.NodeList()
        context.push()
        try:
            index_value = self.index_expr.resolve(context)
            value = self.map_expr.resolve(context)[index_value]
        except KeyError:
            value = None
        context[self.newvar] = value
        for node in self.nodelist_block:
            nodelist.append(node.render(context))
        context.pop()
        return nodelist.render(context)

@register.tag
def index(parser, token):
    bits = token.contents.split()
    if len(bits) != 6:
        raise template.TemplateSyntaxError(u"%r doesn't match definition, usage: index var by indexvar as name")
    map_expr = parser.compile_filter(bits[1])
    index_expr = parser.compile_filter(bits[3])
    newvar = bits[5]
    nodelist_block = parser.parse((u'endindex',))
    parser.delete_first_token()
    return IndexNode(map_expr, index_expr, newvar, nodelist_block)
