#!/usr/bin/python2.4
# -*- coding: UTF-8 -*-

"""
Aufruf: ``%s <POP-Datenbank-Pfad> [-t <tabelle>, ...]``

Beschreibung
------------

Ohne -t:
    
Gleicht die in kundebunt erstellten Datenbank-Modelle mit der Spezifikation in
POP_Datenbank ab. Dabei wird zuerst POP_Datenbank mit den docstrings der kundebunt-Modelle
verglichen, danach wird POP-Datenbank mit den Metainformationen der Datenbank verglichen ("mod").

Mit -t:

Gibt Django-Modell-Klassen für die angegebenen Tabellen aus.

Ausgabe
-------

Meldungen (ohne -t) haben folgendes Format::

    <Phase> <Prio> <Typ> <Tabelle.Feld> <Beschreibung>

Phase
    ``doc`` oder ``mod``, je nach Phase.

Prio
    ``ERROR``, ``WARNING``, ``HINT``

Typ
    Wie bei diffs: ``-``, ``+``, ``!``


Exit Value
----------

* 0: Keine Fehlermeldungen
* 1: Fehlermeldungen
* 2: Falscher Aufruf auf der Kommandozeile
* 3: Syntaxfehler in POP_Datenbank
* 4: POP_Datenbank kann nicht geöffnet oder gelesen werden

Wichtige python-Funktionen
--------------------------

* ``validate()`` führt die obengenannte Funktionalität durch, aber detaillierter steuerbar
* ``print_spec()`` erzeugt aus einem Django-Modell eine Spezifikation wie in POP_Datenbank.
* ``read_popspec()`` liest die Spezifikationen von POP-Datenbank ein und gibt ein dict zurück
* ``popspec.django_model_spec()`` gibt ein Django-Modell für eine POP-Datenbank-Spezifikation zurück.
"""

from django.db.models import get_models, get_apps
_dummy = get_apps()    # andernfalls ermittelt get_apps nur einen Teil der Module, warum-auch-immer (TM)

import re, types, locale, sys
from itertools import chain
from django.conf import settings
from django.db.models import fields as djfields
# from django.db.models.related import related as djfields.related
#from kundebunt.popkern import models
from kundebunt.popkern.utils import any, ConstWrap
from kundebunt.popkern.fields import *

__all__ = ["validate", "print_spec" ]

_tabelle_re = re.compile(r'Tabelle:\s+(\w+)$')
_index_re = re.compile(r'Index:\s+(.*)$')
_verwendung_re = re.compile(r'Verwendung:\s+(\S.*)$')
_flag_split_re = re.compile(r"!\([^()]+\)|![^!()]*")

_typ_conv = {'AutoField': 'uint', 'ForeignKey': 'uint', 'IntegerField': 'int', 'DescriptorSet': 'int',
             'CharField': 'char', 'DateTimeField': 'datetime', 'TextField': 'text', 'DescriptorField': 'int',
             'PositiveSmallIntegerField': 'uint', 'PositiveIntegerField': 'uint', 'SmallIntegerField': 'int',
             'CharDescriptorField': 'char', 'UnixDateTimeField': 'uint',
             }

_test_only_modules = ['kundebunt.validator.models']

class DIFF_TYPE(ConstWrap, str):
    """Art der Änderung. Für PopTableDiff und PopFieldDiff."""
    INS = '+'
    DEL = '-'
    MOD = '!'

class PRIO(ConstWrap, int):
    """Priorität der Abweichung."""
    IGNORE = 0
    HINT = 1
    WARNING = 2
    ERROR = 3

class PHASE(ConstWrap, str):
    """In welcher Testphase: Docstrings, oder Models?"""
    DOC = 'doc'
    MOD = 'mod'


    
def validate_model(cls):
    """vergleicht ein Django-Modell mit seinem eigenen Docstring.
    Mehr zum Testen gedacht.
    """
    m =  PopTable.from_model(cls._meta)
    d = PopTable.from_doc(cls)
    diffs = d.validate(m, None)
    for diff in diffs:
        print(diff)
    return d, m, diffs

def read_popspec(popspec_path):
    f = open(popspec_path, "r")
    specs = {}
    try:
        for table_spec in split_popfile(f):
            spec = PopTable.from_text(table_spec)
            specs[spec.tablename] = spec
        f.close()
    except SyntaxError:
        f.close()
        print "FATAL: syntax error in pop db specification in %r" % popspec_path
        raise
        sys.exit(3)
    except IOError:
        f.close()
        print "FATAL: cannot read pop db specification at %r" % popspec_path
        sys.exit(4)
    return specs

def validate(popspec_path, min_prio_docstrings=PRIO.HINT, min_prio_models=PRIO.WARNING):
    """Vergleicht POP-Datenbank mit den docstrings und
    mit den Metadaten der Django-Modelle.

    popspec_path <string>: Pfad zu POP-Datenbank
    min_prio_docstrings <PRIO>: Meldungen unter dieser Priorität werden beim docstring-Lauf übersprungen.
                         Wenn None, wird dieser Test komplett übersprungen.
    min_prio_models <PRIO>: Meldungen unter dieser Priorität werden beim Modell-Lauf übersprungen.
                         Wenn None, wird dieser Test komplett übersprungen.
    Rückgabe <int>: 0 wenn alles OK und bei Warnungen,
                    1 bei Fehlern (PRIO.ERROR),
                    3 bei Syntaxfehlern in POP-Datenbank,
                    4 wenn POP-Datenbank nicht gelesen werden kann.
    """
    # POP-Datenbank einlesen -> dict specs
    def print_diffs(diffs, min_prio, phase):
        """Hilfsfunktion, Ausgabe einer Liste von PopTableDiff|PopFieldDiff, mit Setzen der angegebenen phase.
        """
        for diff in diffs:
            if diff.prio >= min_prio:
                diff.phase = phase
                print diff

    specs = read_popspec(popspec_path)

    # Docstrings der kundebunt-Modelle vergleichen
    err_count = 0
    if not min_prio_docstrings is None:
        for app in get_apps():
            if app.__name__.startswith('kundebunt.') and not app.__name__ in _test_only_modules:
                for mod in get_models(app):
                    try:
                        docspec = PopTable.from_doc(mod)
                        if not docspec.tablename in specs:
                            diffs = [PopTableDiff(prio=PRIO.ERROR, tablename=docspec.tablename, attname='table', diff_type=DIFF_TYPE.INS,
                                                text="%r has no specification for table '%s'" % (popspec_path, docspec.tablename))]
                        else:
                            diffs = specs[docspec.tablename].validate(docspec, None)
                    except (SyntaxError,), err:
                        diffs = [PopTableDiff(prio=PRIO.ERROR, tablename=mod._meta.db_table, attname='table', diff_type=DIFF_TYPE.DEL,
                                            text="Syntax error in docstring: %s" % UString(err.message))]
                    print_diffs(diffs, min_prio_docstrings, PHASE.DOC)
                    err_count += len([diff for diff in diffs if diff.prio >= PRIO.ERROR])

    # Django-Modelle mit POP-Datenbank-Specs vergleichen
    known_tables = {}
    table_list = []
    if not min_prio_models is None:
        for app in get_apps():
            if app.__name__.startswith('kundebunt.') and not app.__name__ in _test_only_modules:
                for mod in get_models(app):
                    modspec = PopTable.from_model(mod._meta)
                    known_tables[modspec.tablename] = modspec
                    table_list.append(modspec)
        for modspec in table_list:
            if not modspec.tablename in specs:
                diffs = [PopTableDiff(prio=PRIO.ERROR, tablename=modspec.tablename, attname='table', diff_type=DIFF_TYPE.INS, phase=PHASE.MOD,
                                        text="%r has no specification for table '%s'" % (popspec_path, modspec.tablename))]
            else:
                diffs = specs[modspec.tablename].validate(modspec, known_tables)
            print_diffs(diffs, min_prio_models, PHASE.MOD)
            err_count += len([diff for diff in diffs if diff.prio >= PRIO.ERROR])

    if err_count == 0:
        return 0
    else:
        print "%d errors encountered." % err_count
        return 1

def print_django_specs(popspec_path, tables):
    specs = read_popspec(popspec_path)
    for table in tables:
        spec = specs.get(table)
        if not spec:
            print "Tabelle %r gibt es nicht."
        else:
            print spec.django_model_spec()
    return 0

def print_spec(django_model):
    """Gibt die Spezifikation eines Django-Models auf stdout aus."""
    spec = PopTable.from_model(django_model._meta)
    print spec


class SyntaxError(RuntimeError):
    def __init__(self, message):
        self.message = message
        RuntimeError.__init__(self)
    def __unicode__(self):
        return u'Syntax Error in db specs: %s' % self.message
    def __str__(self):
        return 'Syntax Error in db specs: %s' % self.message

class UnicodeStr(object):
    """Mixin zur gescheiten Umkodierung von __unixcode__ zu __str__."""
    def __str__(self):
        return unicode(self).encode(locale.getpreferredencoding())

class UString(UnicodeStr, unicode):
    pass

class PopTableDiff(UnicodeStr):
    """Abweichung zwischen zwei Tabellenspezifikationen."""
    def __init__(self, prio, tablename, attname, diff_type, text, phase=None):
        self.prio = prio
        self.phase = phase
        self.tablename = tablename
        self.attname = attname
        self.diff_type = diff_type
        self.text = text

    def __unicode__(self):
        return u"%s %-8s %s %-20s %s" % (self.phase, self.prio.const_name(), self.diff_type, self.tablename, self.text)

    def __repr__(self):
        return "PopTableDiff(prio=%r, tablename=%r, attname=%r, diff_type=%s, text=%r, phase=%r)" % (self.prio, self.tablename, self.attname, self.diff_type, self.text, self.phase)

class PopFieldDiff(UnicodeStr):
    """Abweichung zwischen zwei Feldspezifikationen"""
    def __init__(self, prio, tablename, field1, field2, attname, diff_type, text, phase=None):
        self.prio = prio
        self.tablename = tablename
        self.field1 = field1
        self.field2 = field2
        self.attname = attname
        self.diff_type = diff_type
        self.text = text
        self.phase = phase

    def __unicode__(self):
        return u"%s %-8s %s %-20s %s" % (self.phase, self.prio.const_name(), self.diff_type, "%s.%s" % (self.tablename, self.field1.fieldname), self.text)

    def __repr__(self):
        return "PopFieldDiff(prio=%r, tablename=%r, field1=%r, field2=%r, attname=%r, diff_type=%s, text=%r, phase=%r)" % (self.prio, self.tablename, self.field1, self.field2, self.attname, self.diff_type, self.text, self.phase)
    

def line_is_irrelevant(line):
    """Für den Parser: ist die Zeile wichtig, oder ist das Kommentar/Leerzeile/unwichtige Angabe?"""
    return line.startswith("#") or line=='' or line.startswith('ID-Liste:') or line.startswith('TODO:') or line.startswith('Hash:') or line.startswith("Acct:")

def split_popfile(file):
    """
    Generiert Arrays aus Strings, die jeweils eine Tabelle definieren.
    Jedes Arrayelement entspricht einer Zeile.
    """
    collected = []
    for line in file:
        line = unicode(line.rstrip(), 'utf-8')
        if line_is_irrelevant(line):
            continue
        elif line == '*** ENDE ***':
            break
        elif line.startswith(' ') or line.startswith('\t') and collected:
            collected[-1] += (" %s" % line.lstrip())
        elif line.startswith('Tabelle:'):
            if collected:
                yield collected
            collected = [line]
        else:
            collected.append(line)
    if collected:
        yield collected

def split_flags(text):
    """
    Gibt ein Array aus den Substrings von text, die jeweils ein Flag definieren
    """
    return _flag_split_re.findall(text)

def check_fkey_to(djfield, table, column):
    """checks that the given django field is a foreign key
    that relates to the given table and column (both strings)
    raises AssertError if not.
    """
    assert djfield.__class__ == djfields.ForeignKey
    assert djfield.rel.to._meta.db_table == table
    assert djfield.rel.field_name == column
        
def check_fkey_to_primary_key(djfield, table):
    """checks that the given django field is a foreign key
    that relates to the given table and its primary key)
    raises AssertError if not.
    """
    pkey = djfield.rel.to._meta.pk.attname
    check_fkey_to(djfield, table, pkey)

def check_unique(djfield):
    """checks that the given django field has a unique constraint
    raises AssertError if not.
    """
    assert djfield.unique


def check_not_unique(djfield):
    """checks that the given django field does not have a unique constraint.
    raises AssertError if it does.
    """
    assert not djfield.unique

#def get_flag_classes():
    #for cand in PopFlag.__module__.__dict__.itervalues():
        #if type(cand)==type and issubclass(cand, PopFlag) and not cand is PopFlag:
            #yield cand

def _split_commasep(text):
    """
    splits an string with comma separated items, blanks optional

    >>> _split_commasep("abc, def,ghi , j")
    ['abc', 'def', 'ghi', 'j']
    """
    return [s.strip() for s in text.split(",")]

def lines_from_docstring(cls):
    """
    Generates text lines like in a PopDB definition from a django model with docstring
    """
    yield u"Tabelle: %s" % cls._meta.db_table
    it = ((unicode(_l, "UTF-8") for _l in cls.__doc__.splitlines()))
    line = it.next()
    while not line.strip():
        line = it.next()
    # Anhand der ersten Zeile den Indentation level bestimmen
    indent = line.index(line.strip()[0])
    del it

    # Und nochmal von vorne über den normalen line_filter
    buffer = ''
    for line in unicode(cls.__doc__, "UTF-8").splitlines():
        line = line[indent:].rstrip()
        if line_is_irrelevant(line):
            continue
        elif line.startswith(' ') or line.startswith('\t') and buffer:
            buffer += (" %s" % line.lstrip())
        else:
            if buffer:
                yield buffer
            buffer = line
    if buffer:
        yield buffer


            
class PopTable(UnicodeStr):
    """
    Definition einer Tabelle a la POP-Datenbank in geparster Form.
    """
    def __init__(self, tablename, unique_index, index, comment, fields):
        self.tablename = tablename
        self.unique_index = unique_index
        self.index = index
        self.comment = comment
        self.fields = fields

    @staticmethod
    def from_doc(cls):
        """
        Erzeugt ein PopTabke-Objekt aus dem doc-string eines Django models.
        Aufbau eines docstrings:
        -----
        Verwendung: ...
        ...
        <Leerzeile>
        Index: ... (Rest wie in POP-Datenbank, allerdings eingerückt.)
        """
        return PopTable.from_text(lines_from_docstring(cls))

    @staticmethod
    def from_text(lines):
        """
        Erzeugt ein PopDB-Objekt aus einem Stringarray im Format wie POP-Datenbank.
        lines beschreiben exakt eine Tabelle. Akzeptiert auch einen String direkt.
        """
        if isinstance(lines, types.StringTypes):
            lines = lines.splitlines()
        try:
            it = iter(lines)
            unique_index = []
            index = []
            comment = []
            fields = []
            tablename = '?'
            line = '*not a single line found*'
            try:
                line = it.next()
                match = _tabelle_re.match(line)
                if not match:
                    raise SyntaxError('%r should start with "Tabelle:"')
                tablename = match.group(1)
                line = it.next()
        
                match = _verwendung_re.match(line)
                if match:
                    comment.append(match.group(1))
                    line = it.next()
    
                match = _index_re.match(line)
                if match:
                    unique_index = _split_commasep(match.group(1))
                    line = it.next()
                    match = _index_re.match(line)
                    if match:
                        index = _split_commasep(match.group(1))
                        line = it.next()
            except StopIteration:
                raise SyntaxError('premature end in table %r, missing section "Aufbau:" at line %r' % (tablename, line))
                    
            if not line.startswith('Aufbau:'):
                raise SyntaxError('%r should start with "Aufbau:"' % line)
            for line in it:
                fields.append(PopField.from_text(line))
        except (SyntaxError,), err:
            err.message = ('Error in specification of database table %r'
                           ' within line: %r'
                           ' %s' % (tablename, line, err.message))
            raise err
        return PopTable(tablename=tablename, unique_index=unique_index, index=index,
                        comment=comment, fields=fields)

    @staticmethod
    def from_model(opts):
        """
        Erzeugt ein PopDB-Objekt aus den Metainformationen eines Django-Models
        """
        fields = []
        primary_index = []
        index = []
        unique_index = [opts.pk.column]
        for djfield in opts.fields:
            field, is_primary, is_unique, is_index = PopField.from_django(djfield)
            fields.append(field)
            if is_primary:
                primary_index.append(field.fieldname)
            elif is_unique:
                unique_index.append(field.fieldname)
            elif is_index:
                index.append(field.fieldname)
        if opts.verbose_name:
            comment = [opts.verbose_name.decode('utf-8')]
        else:
            comment = 'converted from django model'
        return PopTable(tablename=opts.db_table, unique_index=[" ".join(primary_index)] + unique_index, index=index, comment=comment, fields=fields)

    def __unicode__(self):
        """
        Erzeugt einen String im Format für POP-Datenbank
        """
        lines = []
        lines.append(u'Tabelle:\t%s' % self.tablename)
        if self.comment:
            lines.append(u'Verwendung:\t%s' % '\n            '.join(self.comment))
        lines.append(u'Index:\t%s' % ", ".join(self.unique_index))
        if self.index:
            lines.append(u'Index:\t%s' % u", ".join(self.index))
        lines.append(u'Aufbau:')
        lines.extend((unicode(field) for field in self.fields))
        return u'\n'.join(lines)

    def __repr__(self):
        return '%s.from_text(%r)' % (self.__class__.__name__, unicode(self))

    def __eq__(self, other):
        return (
            self.tablename == other.tablename 
            and self.unique_index == other.unique_index
            and self.index == other.index
            and self.fields == other.fields
            )

    def __ne__(self, other):
        return not self==other

    def primary_keys(self):
        """Gibt die Namen der primary keys zurück, als ['column_name', ...]
        """
        return self.unique_index[0].split()

    def validate(self, other, known_tables):
        """
        Gibt zurück, ob other auf das in self definierte Modell passt. Rückgabe ist
        eine Liste aus [PopTableDiff|PopFieldDiff]

        `other` <PopTable>: Eine Tabellenbeschreibung, die "ungenauer" als self ist.
        `known_tables` <{'tablename': <PopTable>, ...}>; Definierte Tabellenspezifikationen auf der other-Seite
            (ausschlaggebend für die Priorität von ForeignKey-Constraints)
            Kann ``None`` sein, das bedeutet dann, dass alle Tabellen als vorhanden vorausgesetzt werden.
        """
        diffs = []
        if self.tablename != other.tablename:
            diffs.append(PopTableDiff(prio=PRIO.ERROR, tablename=self.tablename, attname="tablename", diff_type=DIFF_TYPE.MOD, text=u"Die Tabellennamen sind unterschiedlich."))
        if self.primary_keys() != other.primary_keys():
            diffs.append(PopTableDiff(prio=PRIO.ERROR, tablename=self.tablename, attname="unique_index", diff_type=DIFF_TYPE.MOD, text=u"Die primary keys sind unterschiedlich: %r vs. %r" % (self.primary_keys(), other.primary_keys())))
        for attname, my_index, other_index in (('unique_index', self.unique_index[1:], other.unique_index[1:]),
                                               ('index', self.index, other.index)):
            if sorted(my_index) != sorted(other_index):
                for i in my_index:
                    if i not in other_index:
                        diffs.append(PopTableDiff(prio=PRIO.HINT, tablename=self.tablename, attname=attname, diff_type=DIFF_TYPE.DEL, text=u"Der %s %r ist nicht enthalten." % (attname, i)))
                for i in other_index:
                    if i not in my_index:
                        diffs.append(PopTableDiff(prio=PRIO.HINT, tablename=self.tablename, attname=attname, diff_type=DIFF_TYPE.INS, text=u"Der %s %r wurde hinzugefügt." % (attname, i)))
        if other.comment and self.comment and self.comment != other.comment:
            diffs.append(PopTableDiff(prio=PRIO.HINT, tablename=self.tablename, attname='comment', diff_type=DIFF_TYPE.MOD, text=u"Die Kommentare sind unterschiedlich"))
            
        other_fields = dict(((f.fieldname,f) for f in other.fields))
        for f in self.fields:
            if f.fieldname in other_fields:
                diffs.extend(f.validate(other_fields[f.fieldname], self.tablename, known_tables))
                del other_fields[f.fieldname]
            else:
                if f.has_flag(ObsoleteFlag):
                    pass
                elif f.has_flag(SoonObsoleteFlag):
                        diffs.append(PopTableDiff(prio=PRIO.HINT, tablename=self.tablename, attname="fields", diff_type=DIFF_TYPE.DEL, text=u"Bald obsoletes Feld '%s' bereits entfernt." % f.fieldname))
                else:
                    diffs.append(PopTableDiff(prio=PRIO.WARNING, tablename=self.tablename, attname="fields", diff_type=DIFF_TYPE.DEL, text=u"Feld '%s' fehlt." % f.fieldname))
        for f in other_fields.itervalues():
            diffs.append(PopTableDiff(prio=PRIO.ERROR, tablename=self.tablename, attname="fields", diff_type=DIFF_TYPE.INS, text=u"Feld %r wurde hinzugefügt" % f.fieldname))
        return diffs

    def get_field(self, fieldname):
        for f in self.fields:
            if f.fieldname == fieldname:
                return f
        else:
            raise KeyError('No field named %r' % fieldname)

    def django_model_spec(self):
        """Gibt einen unicode-String mit der
        Definition eines äquivalenten Django-Modells zurück."""
        def _gen():
            yield '"""'
            for l in unicode(self).split('\n'):
                yield l
            yield '"""'
            for f in self.fields:
                yield f.django_model_spec()
        return "\n".join(chain(
            ["class %s(models.Model):" % self.tablename.capitalize()],
            ("    %s" % l for l in _gen())))

class PopField(UnicodeStr):
    """Spezifikation eines Feldes, entsprechend einer Zeile in POP-Datenbank."""
    def __init__(self, fieldname, typ, flags, comment):
        self.fieldname = fieldname
        self.typ = typ
        self.flags = flags
        self.comment = comment
        if comment.startswith("!"):
            raise SyntaxError(u"Der Kommentar im Feld '%s' beginnt mit einem '!': '%s'" % (self.fieldname, UString(unicode(comment))))

    @staticmethod
    def from_text(text):
        """Erzeugen aus einem string (wie eine Zeile in POP-Datenbank)"""
        comment = flags = ''
        l = text.split(None,3)
        if len(l)==2:
            fieldname, typ = l
        elif len(l)==3:
            if l[2].startswith("!"):
                fieldname, typ, flags = l
            else:
                fieldname, typ, comment = l
        elif len(l)==4:
            fieldname, typ, flags, comment = l
            if flags=='-':
                flags = ''
        else:
            err = 'cannot parse field definiton: %r' % text
            print err
            raise SyntaxError(err)
        return PopField(fieldname, PopTyp.from_text(typ), [PopFlag.from_text(flag) for flag in split_flags(flags)], comment)
    
    @staticmethod
    def from_django(djfield):
        """Erzeugen aus einem django-Datenbankfeld
        """
        cls_name = djfield.__class__.__name__
        fieldname=djfield.column
        precision = djfield.max_length
        flags = []

        if cls_name=='AutoField':
            precision = 4
            flags = [AutoIncrementFlag()]
        elif cls_name=='ForeignKey':
            precision = 4
            rel_opts = djfield.rel.to._meta
            rel_field = rel_opts.get_field(djfield.rel.field_name)
            if rel_field.primary_key:
                flags = [FKeyToPrimaryFlag([rel_opts.db_table])]
            else:
                flags = [FKeyFlag([rel_opts.db_table, rel_field.column])]
        elif cls_name in ('IntegerField', 'PositiveIntegerField'):
            precision = 4
        elif cls_name in ('SmallIntegerField', 'PositiveSmallIntegerField'):
            precision = 2
        elif cls_name=='DescriptorSet':
            flags = [DescriptorSetFlag([djfield.descr_name])]
            precision = 8
        elif cls_name in ('DescriptorField', 'CharDescriptorField'):
            if not precision:
                if cls_name == 'DescriptorField':
                    precision = 2
                else:
                    precision = 1
            if djfield.default != djfields.NOT_PROVIDED:
                flags = [DefaultDescrFlag([djfield.descr_name, djfield.default])]
            else:
                flags = [DescrFlag([djfield.descr_name])]
        elif cls_name=='TextField':
            if precision:
                for max_bytes, new_size in [(1,255), (2,65535)]:
                    if precision <= max_bytes:
                        precision = new_size
                        break
                else:
                    precision = 3
            else:
                precision = 2
        elif cls_name=='UnixDateTimeField':
            precision = 4
            flags = [DisplayDateTimeFlag()]
        if djfield.null:
            flags.append(NullAllowedFlag())
        base_type = _typ_conv[cls_name]
        typ = PopTyp(base_type, precision)
        comment = djfield.verbose_name
        if fieldname == comment.lower():
            comment = ''
        return (PopField(fieldname=fieldname, typ=typ, flags=flags, comment=comment), djfield.primary_key, djfield.unique, djfield.db_index)

    def __unicode__(self):
        if self.flags:
            flag_str = u"".join([unicode(flag) for flag in self.flags])
        elif self.comment:
            flag_str = '-'
        else:
            flag_str = ''
        return u"%s\t%s\t%s\t%s" % (self.fieldname, self.typ, flag_str, self.comment)
        
    def __repr__(self):
        return '%s.from_text(%r)' % (self.__class__.__name__, unicode(self))
    
    def __eq__(self, other):
        return (
            self.fieldname == other.fieldname
            and self.typ == other.typ
            and self.flags == other.flags
            )

    def __ne__(self, other):
        return not self == other

    def has_flag(self, flag_classes):
        return any((isinstance(fl, flag_classes) for fl in self.flags))

    def validate(self, other, tablename, known_tables):
        """
        Gibt zurück, ob other auf das in self definierte Feld passt. Rückgabe ist
        eine Liste aus [PopFieldDiff]

        other <PopTable>: Eine Tabellenbeschreibung, die "ungenauer" als self ist.
        tablename <string>: Name der zugehörigen Tabelle (für die PopFieldDiff-Objekte)
        `known_tables` <{'tablename': <PopTable>, ...}>; Definierte Tabellenspezifikationen auf der other-Seite
            (ausschlaggebend für die Priorität von ForeignKey-Constraints)
            Kann ``None`` sein, das bedeutet dann, dass alle Tabellen als vorhanden vorausgesetzt werden.
        """
        diffs = []
        if self.fieldname != other.fieldname:
            diffs.append(PopFieldDiff(prio=PRIO.ERROR, tablename=tablename, field1=self, field2=other, attname="fieldname", diff_type=DIFF_TYPE.MOD, text=u"Die Feldnamen sind unterschiedlich (%r != %r)." % (self.fieldname, other.fieldname)))
        if self.typ != other.typ:
            if (self.typ.base_type != other.typ.base_type
                and not (self.typ.base_type=="fchar" and other.typ.base_type=="char")
                and not (
                    self.typ.base_type=="int" and other.typ.base_type=="uint"
                    and any(isinstance(fl, (AutoIncrementFlag, FKeyFlag, FKeyToPrimaryFlag, FKeyGroupFlag))
                            for fl in self.get_valid_flags()))
               ):
                flags = self.get_valid_flags()
                diffs.append(PopFieldDiff(prio=PRIO.ERROR, tablename=tablename, field1=self, field2=other, attname="typ", diff_type=DIFF_TYPE.MOD, text=u"Die Basistypen im Feld %r sind unterschiedlich." % str(self.fieldname)))
            elif self.typ.precision != other.typ.precision:
                if self.typ.base_type in ("int, uint") and other.typ.precision > self.typ.precision:
                    prio = PRIO.HINT
                else:
                    prio = PRIO.WARNING
                diffs.append(PopFieldDiff(prio=prio, tablename=tablename, field1=self, field2=other, attname="typ", diff_type=DIFF_TYPE.MOD, text=u"Die Feldlängen im Feld %r sind unterschiedlich." % str(self.fieldname)))
        other_flags = dict(((str(fl), fl) for fl in other.get_valid_flags()))
        for fl in self.get_valid_flags():
            if str(fl) not in other_flags:
                compatible_flags = [flcomp for flcomp in fl.compatible_flags() if str(flcomp) in other_flags]
                if compatible_flags:
                    diffs.append(PopFieldDiff(prio=PRIO.HINT, tablename=tablename, field1=self, field2=other, attname="flags", diff_type=DIFF_TYPE.MOD, text=u"Das Flag %s fehlt im Feld %r (%s) (welches aber %s enthält)." % (unicode(fl), str(self.fieldname), str(self.typ), str(compatible_flags[0]))))
                    del other_flags[str(compatible_flags[0])]
                else:
                    prio = fl.prio(known_tables)
                    diffs.append(PopFieldDiff(prio=prio, tablename=tablename, field1=self, field2=other, attname="flags", diff_type=DIFF_TYPE.MOD, text=u"Das Flag '%s' (%s) fehlt im Feld %r (%s)" % (unicode(fl), fl.__class__.__name__, str(self.fieldname), str(self.typ))))
            else:
                del other_flags[str(fl)]
        for fl in other_flags.itervalues():
            prio = fl.prio(known_tables)
            diffs.append(PopFieldDiff(prio=prio, tablename=tablename, field1=self, field2=other, attname="flags", diff_type=DIFF_TYPE.MOD, text=u"Das Flag '%s' (%s) wurde im Feld %r hinzugefügt" % (unicode(fl), fl.__class__.__name__, str(other.fieldname))))
        return diffs

    def get_valid_flags(self):
        """Zählt nur die Flags auf, die nicht durch andere Flags aufgehoben werden. Gibt u.U. auch eine Liste zurück.
        """
        cancelled = ()
        if self.has_flag(DescrLie):
            cancelled += (DescrFlag,)
        if self.has_flag(DescrFlag):
            cancelled += (AutoIncrementFlag,)
        if cancelled:
            return (fl for fl in self.flags if not isinstance(fl, cancelled))
        else:
            return self.flags


    def django_model_spec(self):
        """Gibt einen unicode-String mit der
        Definition eines äquivalenten Django-Modells zurück."""
        def _uni2str(s):
            if isinstance(s, unicode):
                return s.encode('utf-8')
            else:
                return s
            
        params = {}
        for flag_cls in (
                AutoIncrementFlag, FKeyToPrimaryFlag, FKeyFlag, DescriptorSet,
                DescriptorField, CharDescriptorField, DefaultDescrFlag,
                DescrFlag, DisplayDateTimeFlag):
            if self.has_flag(flag_cls):
                dj_class = flag_cls.dj_class
                break
        else:
            base_type = self.typ.base_type
            if base_type in ('char', 'fchar'):
                dj_class = djfields.CharField
                if self.typ.precision:
                    params['max_length'] = self.typ.precision
            elif base_type == 'int':
                dj_class = djfields.IntegerField
            elif base_type == 'uint':
                dj_class = djfields.PositiveIntegerField
            elif base_type == 'text':
                dj_class = djfields.TextField
            elif base_type == 'datetime':
                dj_class = djfields.DateTimeField
            else:
                print "undefiniert: ", self.typ.base_type
                raise SyntaxError("undefiniert: %s" % self.typ.base_type)
        for flag in self.flags:
            params.update(flag.django_model_params(self))
        if dj_class.__module__.startswith('django'):
            class_prefix = 'models.'
        else:
            class_prefix = ''
        #print self.fieldname
        return "%s = %s%s(%s)" % (self.fieldname, class_prefix, dj_class.__name__,
                                  ", ".join(["%s=%r" % (k,_uni2str(v)) for k,v in params.iteritems()]))

class PopTyp(UnicodeStr):
    """Ein Datenfeld-Typ."""
    parse_re = re.compile(r'([a-z]+)(\d+)?$')
    djtyp = {  # maps base_type -> [compatible_django_field_class, ...]
            'char': [djfields.CharField, djfields.EmailField, djfields.URLField],
            'fchar': [djfields.CharField],
            'int': [djfields.IntegerField, djfields.SmallIntegerField, djfields.related.ForeignKey],
            'uint': [djfields.PositiveIntegerField, djfields.PositiveSmallIntegerField, djfields.related.ForeignKey, djfields.AutoField, djfields.AutoField],
            'text': [djfields.CharField],
            'bin': [djfields.ImageField],
            'datetime': [djfields.DateTimeField],
            }
    def __init__(self, base_type, precision):
        if precision==None and base_type in ("fchar", "char"):
            precision = 1
        self.base_type = base_type
        self.precision = precision

    @staticmethod
    def from_text(text):
        match = PopTyp.parse_re.match(text)
        if not match:
            raise SyntaxError("%r is not parseable as field type" % text)
        base_type, precision = match.groups()
        return PopTyp(base_type, precision and int(precision))

    def __unicode__(self):
        if self.precision:
            return u"%s%d" % (self.base_type, self.precision)
        else:
            return self.base_type

    def __eq__(self, other):
        return (
            self.base_type == other.base_type
            and self.precision == other.precision
            )

    def __ne__(self, other):
        return not self == other

    def __repr__(self):
        return 'PopTyp(%r,%r)' % (self.base_type, self.precision)

    def django_model_params(self, field):
        return {}


class PopFlagMeta(type):
    """Meta class for all PopFlag classes, to get a list of all subclasses."""
    subclasses = []
    def __init__(cls, name, bases, dct):
        super(PopFlagMeta, cls).__init__(cls, name, bases, dct)
        PopFlagMeta.subclasses.append(cls)
        
class PopFlag(UnicodeStr):
    """Abstrakte Oberklasse für Feld-Flags."""
    __metaclass__ = PopFlagMeta
    important = False
    
    def __init__(self, attrs=()):
        self.attrs = tuple(attrs)

    @staticmethod
    def from_text(text):
        for cls in PopFlagMeta.subclasses:
            if cls != PopFlag:
                match = cls.regex.match(text)
                if match:
                    return cls(match.groups())
        else:
            raise SyntaxError('%r does not define a flag.' % text)

    def __unicode__(self):
        return self.format % self.attrs

    def __eq__(self, other):
        return (
            self.__class__ == other.__class__
            and self.attrs == other.attrs
            )

    def prio(self, known_tables):
        """
        Gibt die Bedeutung dieses Pop-Flags zurück (als PRIO).
        `known_tables` <{'tablename': <PopTable>, ...}>; Definierte Tabellenspezifikationen auf der other-Seite
            (ausschlaggebend für die Priorität von ForeignKey-Constraints)
            Kann ``None`` sein, das bedeutet dann, dass alle Tabellen als vorhanden vorausgesetzt werden.
        """
        if self.important:
            return PRIO.ERROR
        else:
            return PRIO.HINT

    def compatible_flags(self):
        """Gibt mit diesem Flag kompatible Flags (im Sinne von validate) zurück.
        """
        return ()
    
    def null_allowed(self):
        return False

    def __ne__(self, other):
        return not self == other

    def django_model_params(self, field):
        return {}

class FKeyFlag(PopFlag):
    """
    !>bla.fasel Der Eintrag verweist auf das Datenfeld 'fasel' in der
    Tabelle 'bla'
    """
    regex = re.compile(r'!>(\w+)\.(\w+)$')
    format = '!>%s.%s'
    dj_class = djfields.related.ForeignKey
    important = True

    def prio(self, known_tables):
        if known_tables != None and self.attrs[0] in known_tables:
            return PRIO.ERROR
        else:
            return PRIO.HINT

    def django_model_params(self, field):
        return {'to': self.attrs[0].capitalize(), 'to_field': self.attrs[1], 'db_column': field.fieldname}

class FKeyToPrimaryFlag(PopFlag):
    """
    !>bla Der Eintrag verweist auf den primary key in der Tabelle 'bla'.
    """
    regex = re.compile(r'!>(\w+)$')
    format = '!>%s'
    dj_class = djfields.related.ForeignKey
    important = True
    
    def prio(self, known_tables):
        if known_tables != None and self.attrs[0] in known_tables:
            return PRIO.ERROR
        else:
            return PRIO.HINT

    def django_model_params(self, field):
        return {'to': self.attrs[0].capitalize(),  'db_column': field.fieldname}

class FKeyGroupFlag(PopFlag):
    """
    !>bla/laber siehe !bla/laber (Altlast: Dienst-Deskriptor)
    # !bla/F,G,^H begrenze die Anzeige auf Werte in Gruppen F oder G aber außer H
    #             (Default: ^hide, wenn '!<Name>_ident.hide' bekannt ist)
    """
    regex = re.compile(r'!>(\w+)/(\w+)$')
    format = '!>%s/%s'
    dj_class = djfields.related.ForeignKey
    important = True

    def compatible_flags(self):
        return (FKeyToPrimaryFlag(self.attrs[:1]),)

    def django_model_params(self, field):
        return {'to': self.attrs[0].capitalize(), 'db_column': field.fieldname}


class FkeyCascadingFlag(PopFlag):
    """
    !>>bla: wie !>bla, aber mit ON DELETE CASCADE
    """
    regex = re.compile(r'!>>(\w+)$')
    format = '!>>%s'
    dj_class = djfields.related.ForeignKey
    important = True

    def compatible_flags(self):
        return (FKeyToPrimaryFlag(self.attrs[:1]),)

    def django_model_params(self, field):
        return {'to': self.attrs[0].capitalize(), 'db_column': field.fieldname}

class DescrFlag(PopFlag):
    """
    !bla        Der Eintrag verweist auf den Deskriptor 'bla' in der
    Deskriptorentabelle; NULLs darf es nicht geben
    """
    regex = re.compile(r'!([\w][-\w]*)$')
    format = "!%s"
    dj_class = DescriptorField
    important = True

    def django_model_params(self, field):
        return dict(descr_model='Descr', descr_name=self.attrs[0], db_column=field.fieldname)

class DefaultDescrFlag(PopFlag):
    """
    !bla.X      dito, Defaultwert bei NULL ist X
    """
    regex = re.compile(r'!(\w+)\.(\w+)$')
    format = "!%s.%s"
    dj_class = DescriptorField
    important = True

    def django_model_params(self, field):
        return dict(descr_model='Descr', descr_name=self.attrs[0], default=self.attrs[1], db_column=field.fieldname)

class DescrGroupFlag(PopFlag):
    """
    !bla/F,G,^H begrenze die Anzeige auf Werte aus F oder G aber außer H
    (Default: ^hide, wenn '!<Name>_ident.hide' bekannt ist)
    """
    regex = re.compile(r'!\((?!TT/)(\w+)/([\w]+(?:,\^?\w+)*)\)$|!(\w+)/([\w]+(?:,\^?\w+)*)$')
    format = None
    dj_class = DescriptorField
    important = True

    def __init__(self, attrs):
        if attrs[0] == None:
            attrs = attrs[2:]
        self.attrs = [attrs[0]] + _split_commasep(attrs[1])
        
    def __unicode__(self):
        return u"!(%s/%s)" % (self.attrs[0], u",".join(self.attrs[1:]))

    def compatible_flags(self):
        return (DescrFlag(self.attrs[:1]),)

    def django_model_params(self, field):
        return dict(descr_model='Descr', descr_name=self.attrs[0], db_column=field.fieldname)

class DescriptorSetFlag(PopFlag):
    """
    !*bla       Der Eintrag enthält eine Bitmap von Deskriptoren 'bla',
    der Wertebereich ist dabei notwendigerweise auf 1..63 beschränkt
    """
    regex = re.compile(r'!\*([\w]+)$')
    format = "!*%s"
    dj_class = DescriptorSet
    important = True

    def django_model_params(self, field):
        return dict(descr_model='Descr', descr_name=self.attrs[0], db_column=field.fieldname)

class IsDescriptorGroupFlag(PopFlag):
    """
    !*<name>_ident  Für descr.gruppe, Supersonderfall, das ist ein IntegerField.
    """
    regex = re.compile(r'!\*<name>_ident$')
    format = '!*<name>_ident'
    dj_class = djfields.IntegerField
    important = False

class NextIdFlag(PopFlag):
    """
    !?bla       der nächste freie Wert steht in nextid->'bla'
    """
    regex = re.compile(r'!\?(\w+)$')
    format = "!?%s"
    dj_class = djfields.IntegerField

class AutoIncrementFlag(PopFlag):
    """
    !?+         der nächste freie Wert wird via autoincrement zugeteilt
    """
    regex = re.compile(r'!\?\+$')
    format = "!?+"
    dj_class = djfields.AutoField
    important = True


class MonetaryFlag(PopFlag):
    """
    !$          Geldbetrag (in Cent)
    """
    regex = re.compile(r'!\$')
    format = "!$"

class DisplayDateTimeFlag(PopFlag):
    """
    !/          der Eintrag ist eine Datums+Zeitangabe (Unix-Sekunden, wenn uint4)
    """
    regex = re.compile(r'!/$')
    format = "!/"
    dj_class = djfields.UnixDateTimeField
    important = True

class TimestampFlag(PopFlag):
    """
    !/+         der Eintrag ist ein Zeitstempel (automatisch geupdatet)
    """
    regex = re.compile(r'!/\+$')
    format = '!/+'
    important = False

    def django_model_params(self, field):
        return {'auto_now': True}

class DisplayTimePeriodFlag(PopFlag):
    """
    !//         der Eintrag ist ein Zeitraum (Sekunden)
    """
    regex = re.compile(r'!//$')
    format = "!//"

class RangeFlag(PopFlag):
    """
    !(A..Z)      Beschränkung des Wertebereichs auf A bis Z inklusive
    können Zahlen oder andere Felder sein
    """
    regex = re.compile(r'!\(([\w.]+)-([\w.]+)\)$')
    format = "!(%s-%s)"

#class RangeFlag2(PopFlag):
    #"""
    #!(A-Z)      Beschränkung des Wertebereichs auf A bis Z inklusive
    #können Zahlen oder andere Felder sein
    #"""
    #regex = re.compile(r'!\(([\w.]+)-([\w.]+)\)')
    #format = "!(%s-%s)"

class YearMonthFlag(PopFlag):
    """
    !(JJJJMM)   Feld enthält Jahr+Monat
    """
    regex = re.compile(r'!\(JJJJMM\)$')
    format = "!(JJJJMM)"

class DayFlag(PopFlag):
    """
    !(TT/X)     Feld enthält Tag; Jahr+Monat steht in Feld X, zwecks Prüfung
    """
    regex = re.compile(r'!\(TT/(\w+)\)$')
    format = "!(TT/%s)"

class DescrOrFKeyFlag(PopFlag):
    """
    !(dienst!ziel) Suche in der durch das Feld 'dienst' bezeichneten
    Deskriptortabelle (also doppelt indirekt), oder in Tabelle
    'ziel' wenn es eine solche nicht gibt
    """
    regex = re.compile(r'!\((\w+)!(\w+)\)$')
    format = "!(%s!%s)"

class DisplayFlag(PopFlag):
    """
    !^          Display: dieses Feld wird angezeigt, wenn ein Index auf
    die Tabelle verweist. Das können mehrere sein =>
    untereinander, eingerückt, und mit Rekursionsbegrenzung.
    """
    regex = re.compile(r'!\^([^!]*)$')
    format = "!^%s"

    def prio(self, known_tables):
        return PRIO.IGNORE

class NullAllowedFlag(PopFlag):
    """
    !-          Feld darf NULL sein
    """
    regex = re.compile(r'!-$')
    format = "!-"
    important = True

    def null_allowed(self):
        return True

    def django_model_params(self, field):
        return dict(null=True, blank=True)

class ObsoleteFlag(PopFlag):
    """
    !##         Feld ist obsolet
    """
    regex = re.compile(r'!##$')
    format = "!##"
    important = True

class SoonObsoleteFlag(PopFlag):
    """
    !#          Feld ist bald obsolet
    """
    regex = re.compile(r'!#$')
    format = "!#"

    def prio(self, known_tables):
        return PRIO.WARNING

class SoonNotNull(PopFlag):
    """
    !-#         Feld ist bald nicht mehr NULLbar
    """
    regex = re.compile('!-#$')
    format = '!-#'
    important = True
    

class SetDefaultFlag(PopFlag):
    """
    !=bla       Setz einen Defaultwert von bla für dieses Feld
    """
    regex = re.compile(r'!=(\w+)$')
    format = '!=%s'

    def django_model_params(self, field):
        return dict(default=self.attrs[0])

class SpecialTreatment(PopFlag):
    """
    # !%XXX       Spezialbehandlung notwendig (Constraint: XXX)
    """

    regex = re.compile(r'!%(?!null|ref)([^!]*)$')
    format = '!%%%s'

class DescrLie(PopFlag):
    """
    # !%ref       die Referenz (NUR bei Deskriptoren!) darf ungültig sein
    """

    regex = re.compile(r'!%ref$')
    format = '!%%ref'
    

class NullOrZero(PopFlag):
    """
    # !%null      das Feld kann sowohl 0 als auch NULL enthalten
    """

    regex = re.compile(r'!%null$')
    format = '!%%null'

    def null_allowed(self):
        return True

    def django_model_params(self, field):
        return dict(null=True, blank=True)

if __name__ == '__main__':
    if len(sys.argv) > 3 and sys.argv[2] == '-t':
        sys.exit(print_django_specs(sys.argv[1], sys.argv[3:]))
    elif len(sys.argv) != 2:
        print __doc__ % sys.argv
        sys.exit(2)
    else:
        sys.exit(validate(sys.argv[1]))
    
