# -*- encoding: UTF-8 -*-

"""Save the name of the view function into thread local store.
The context_processor brings relevant url settings into the context.
There it's picked up by the navibar template tag and controls the
display of the navigation bar.

>>> import kundebunt.mailadmin.views
>>> NavigationMiddleware().process_view(None, kundebunt.mailadmin.views.create_mailbox, [], {"kunden_name":"ilias"})
>>> len(navi_path(navigation_context_processor(None)))==0
False

# If parts of the path are cannot be created because of missing context variables,
# these parts are skipped.
>>> NavigationMiddleware().process_view(None, kundebunt.mailadmin.views.edit_mailbox, [],{})
>>> len(navi_path(navigation_context_processor(None)))==0
False
"""

try:
    from threading import local
except ImportError:
    # import copy of _thread_local.py from python 2.4
    from django.utils._threading_local import local
from django.conf import settings
from django.utils.translation import gettext_lazy, gettext, ugettext_lazy, ngettext, ugettext as _
from django.utils.encoding import smart_unicode
from django.utils.safestring import mark_safe
from django.utils.html import conditional_escape

_navi_view = local()
_navi_view.view_name = _navi_view.module =_navi_view.kwargs = None


class NaviContextProxy(object):
    """Proxies a list of contexts for use with % substitution.
    The contexts are searched for the first match. If none is found,
    it raises ``error_cls`` if provided, or it returns None

    If you set ``add_default_context`` to True, a default context will
    be appended as last resort.

    If you set allow_none to False, __getitem__ will treat a value of None
    like a miss, i.e. returning '' or raising KeyError.

    Also, __getitem__ will htmlescape the string, except when it's been marked
    with mark_safe().

    >>> c=NaviContextProxy([{"a": '1'}, {"b": '2'}])
    >>> c
    <NaviContextProxy [{'a': '1'}, {'b': '2'}]>
    >>> c["a"]
    u'1'
    >>> c["b"]
    u'2'
    >>> c["c"]
    u''
    >>> c=NaviContextProxy([{'a': 1}, {'b': 2}, {'c': None}], False, KeyError, allow_none=False)
    >>> c["c"]
    Traceback (most recent call last):
        ...
    KeyError: "'c' not in any context"

    >>> c=NaviContextProxy([{'a': '<&>', 'b': mark_safe(u'<bla/>')}])
    >>> c["a"]
    u'&lt;&amp;&gt;'
    >>> c["b"]
    u'<bla/>'
    """

    def __init__(self, contexts, add_default_context=False, error_cls=None, allow_none=True):
        self.contexts = contexts
        if add_default_context:
            self.contexts.append(default_navigation_context)
        self.error_cls = error_cls
        self.allow_none = allow_none

    def __getitem__(self, key):
        for context in self.contexts:
            try:
                value = context[key]
                if self.allow_none or value!=None:
                    return conditional_escape(value)
            except KeyError:
                pass
        if self.error_cls:
            raise self.error_cls, "%r not in any context" % key
        else:
            return u''

    def __repr__(self):
        return "<NaviContextProxy %r>" % self.contexts



class NavigationMiddleware(object):
    def process_view(self, request, view_func, view_args, view_kwargs):
        _navi_view.view_name = view_func.__name__
        _navi_view.module = view_func.__module__
        _navi_view.kwargs = view_kwargs
        return None


default_navigation_context = {
        "MAILADMIN_ROOTURL": settings.MAILADMIN_ROOTURL,
        "AUTH_ROOTURL": settings.AUTH_ROOTURL,
        "RELEASENOTES_ROOTURL": settings.RELEASENOTES_ROOTURL,
        "POPKERN_ROOTURL": settings.POPKERN_ROOTURL,
        "HARDWARE_ROOTURL": settings.HARDWARE_ROOTURL,
        "KUNDE_ROOTURL": settings.KUNDE_ROOTURL,
        "ADD_SERVICE_LINKS": settings.ADD_SERVICE_LINKS,
        "BASE_URL": settings.BASE_URL,
        "SERVICE_URL": settings.SERVICE_URL,
        "HISTORY_ROOTURL": settings.HISTORY_ROOTURL,
        "SERVERSTATS_ROOTURL": settings.SERVERSTATS_ROOTURL,
        }

def navigation_context_processor(request):
    """adds variables for url setup"""
    return default_navigation_context.copy()

#def view_tag(view_fn):
    #"""returns the tag for the currently active view
    #import kundebunt.mailadmin.views
    #>>> view_tag(kundebunt.mailadmin.views.create_mailbox)
    #'mailadmin.create_mailbox'
    #>>> view_tag('mailadmin.create_mailbox')
    #'mailadmin.create_mailbox'
    #"""


class NaviMap(dict):
    def get_valid_node(self, tag, context):
        for node in self[tag]:
            if node.is_valid(context):
                return node
        raise KeyError

navimap = NaviMap()

class NaviNode(object):
    """A NaviNode describes an URL with its label and relations to other URLs"""

    def __init__(self, tag, label, url=None, sub_tags=None, parent_tag=None, icon_url=None, alt_parent_tag=None, menu_label=None):
        self.tag = smart_unicode(tag, strings_only=True)
        self.label = smart_unicode(label, strings_only=True)
        self.menu_label = smart_unicode(menu_label, strings_only=True)
        self.url = smart_unicode(url, strings_only=True)
        self.sub_tags = sub_tags
        self.parent_tag = smart_unicode(parent_tag, strings_only=True)
        self.icon_url = smart_unicode(icon_url, strings_only=True)
        self.alt_parent_tag = alt_parent_tag
        self.tuple = (self.label, self.url, self.parent_tag, self.icon_url, self.sub_tags)
        navimap.setdefault(tag, []).append(self)

    def is_valid(self, context):
        return True

class InPanelNaviNode(NaviNode):
    def is_valid(self, context):
        return context.get(u'in_panel')

class StaffNaviNode(NaviNode):
    def is_valid(self, context):
        person = context.get(u'person')
        return person and person.is_staff()

class MitKundeNaviNode(NaviNode):
    def is_valid(self, context):
        return context.get("kunden_name")

class MitKundeStaffNaviNode(StaffNaviNode):
    def is_valid(self, context):
        return super(MitKundeStaffNaviNode, self).is_valid(context) and context.get("kunden_name")

def add_old_navimap(m):
    for (tag, (label, url, parent_tag, icon_url, sub_tags)) in m.iteritems():
        NaviNode(tag=tag, label=label, url=url, parent_tag=parent_tag, icon_url=icon_url, sub_tags=sub_tags)

if settings.ADD_SERVICE_LINKS:
    service_links = ['troubletickets', 'mailadmin', 'rz_zutritt', 'powercycle', 'accounting', 'capacity', 'monitoring', 'hardware', 'download']
else:
    service_links = ['mailadmin', 'hardware']

StaffNaviNode(tag="service", label=ugettext_lazy(u"Menue"), url=settings.BASE_URL)

StaffNaviNode(tag="menubar", label=u"", sub_tags=["kunde.menu", "mailadmin.menu", "hardware.menu", "service.menu"])

NaviNode(tag="menubar", label=u"", sub_tags=["mailadmin.menu", "service.menu"])
NaviNode(tag="kunde.menu", label="%(kunden_name)s", url="%(KUNDE_ROOTURL)s/%(kunden_name)s/", parent_tag="kunde",
         sub_tags=[("kunde.overview", ),
                   ("kunde.unterkunden", "kunde.personen", "kunde.create_person", "kunde.accounts", "kunde.domains", "kunde.ips", "kunde.wiki", "kunde.monitoring"),
                   ("history.kunde",),
                   ("kunde.kunden_suche", ),
                  ])
NaviNode(tag="popkern.hardware_data", label=ugettext_lazy(u"Hardware-Daten"), parent_tag='service', url="%(POPKERN_ROOTURL)s/data/hardware/")
NaviNode(tag="service.menu", label=ugettext_lazy(u"Weitere Dienste"), sub_tags=['troubletickets', 'rz_zutritt', 'powercycle', 'accounting', 'capacity', 'monitoring', 'download'], parent_tag="service")
StaffNaviNode(tag="mailadmin.menu", label=ugettext_lazy(u"E-Mail-Konfiguration"), url="%(MAILADMIN_ROOTURL)s/portal/%(kunden_name)s",
         parent_tag="mailadmin.menu.ohnekunde",
         sub_tags=[('mailadmin.portal',),
                   ('mailadmin.mailbox_qsearch.kunde', 'mailadmin.create_mailbox'),
                   ('mailadmin.mailrule_qsearch.kunde', 'mailadmin.mailrule_search.kunde',  'mailadmin.create_mailrule'),
                   ])
NaviNode(tag="mailadmin.menu", label=ugettext_lazy(u"E-Mail-Konfiguration"), url="%(MAILADMIN_ROOTURL)s/portal/%(kunden_name)s/",
         parent_tag="mailadmin.menu.ohnekunde",
         sub_tags=[('mailadmin',),
                   ('mailadmin.mailbox_qsearch.kunde', 'mailadmin.create_mailbox'),
                   ('mailadmin.mailrule_qsearch.kunde', 'mailadmin.mailrule_search.kunde',  'mailadmin.create_mailrule'),
                   ])
StaffNaviNode(tag="mailadmin.menu.ohnekunde", label=ugettext_lazy(u"E-Mail-Konfiguration"),
         sub_tags=[('mailadmin.mailbox_qsearch',),
                   ('mailadmin.mailrule_qsearch', 'mailadmin.mailrule_search')])
NaviNode(tag="mailadmin.menu.ohnekunde", label=ugettext_lazy(u"E-Mail-Konfiguration"),
         sub_tags=[('mailadmin',),
                   ('mailadmin.mailbox_qsearch',),
                   ('mailadmin.mailrule_qsearch', 'mailadmin.mailrule_search'),
                   ])

add_old_navimap ({
    "service": (ugettext_lazy("Menue"), settings.BASE_URL, None, None, service_links),
    "mailadmin.create_mailbox": (ugettext_lazy("Mailbox anlegen"), "%(MAILADMIN_ROOTURL)s/create-mailbox/%(kunden_name)s/", "mailadmin.portal", "box.png", None),
    "mailadmin.delete_mailbox": (ugettext_lazy("Mailbox loeschen"), "%(MAILADMIN_ROOTURL)s/delete-mailbox/%(mailbox_name)s/", "mailadmin.edit_mailbox", None, None),
    "mailadmin.mailbox_qsearch": (ugettext_lazy("Mailbox suchen"), "%(MAILADMIN_ROOTURL)s/mailbox-qsearch/", "mailadmin.portal", None, None),
    'mailadmin.mailbox_qsearch.kunde': (ugettext_lazy("Mailbox suchen"), "%(MAILADMIN_ROOTURL)s/mailbox-qsearch/?kunde=%(kunden_name)s;noquery", "mailadmin.portal", None, None),
    "mailadmin.edit_mailrule": (ugettext_lazy("Weiterleitungsregel bearbeiten"), "%(MAILADMIN_ROOTURL)s/mailrule/%(mailrule_id)s/", "mailadmin.portal", None, ['mailadmin.delete_mailrule']),
    "mailadmin.edit_mailrule_type": (ugettext_lazy("Typ aendern"), "%(MAILADMIN_ROOTURL)s/mailrule-type/%(mailrule_id)s/", "mailadmin.edit_mailrule", None, None),
    "mailadmin.delete_mailrule": (ugettext_lazy("Weiterleitungsregel loeschen"), "%(MAILADMIN_ROOTURL)s/delete-mailrule/%(mailrule_id)s/", "mailadmin.edit_mailrule", None, None),
    "mailadmin.create_mailrule": (ugettext_lazy("Weiterleitungsregel anlegen"), "%(MAILADMIN_ROOTURL)s/create-mailrule/%(kunden_name)s/", "mailadmin.portal", "arrow_switch.png", None),
    "mailadmin.mailrule_search": (ugettext_lazy("Weiterleitungsregeln-Expertensuche"), "%(MAILADMIN_ROOTURL)s/mailrule-search/", "mailadmin.portal", None, None),
    "mailadmin.mailrule_search.kunde": (ugettext_lazy("Weiterleitungsregeln-Expertensuche"), "%(MAILADMIN_ROOTURL)s/mailrule-search/?kunde=%(kunden_name)s;noquery", "mailadmin.portal", None, None),
    "mailadmin.mailrule_qsearch": (ugettext_lazy("Weiterleitungsregeln suchen"), "%(MAILADMIN_ROOTURL)s/mailrule-qsearch/", "mailadmin.portal", None, None),
    "mailadmin.mailrule_qsearch.kunde": (ugettext_lazy("Weiterleitungsregeln suchen"), "%(MAILADMIN_ROOTURL)s/mailrule-qsearch/?kunde=%(kunden_name)s;noquery", "mailadmin.portal", None, None),
    "mailadmin.kunde_qsearch": (ugettext_lazy("Kunde suchen"), "%(MAILADMIN_ROOTURL)s/customer-qsearch/", "mailadmin", None, None),
    "mailadmin.set_mailflag": (ugettext_lazy("Mailflag setzen"), "%(MAILADMIN_ROOTURL)s/set-mailflag/%(mailbox_name)s", "mailadmin.portal", None, None),
    "kunde_auth.logout": (ugettext_lazy("Ausloggen"), "%(AUTH_ROOTURL)s/logout/", "service", None, None),
    "kunde_auth.login": (ugettext_lazy("Einloggen"), "%(AUTH_ROOTURL)s/login/", "service", None, None),
    "kunde_auth.admin_login": (ugettext_lazy("Administrativ Einloggen"), "%(AUTH_ROOTURL)s/admin-login", "service", None, None),
    "kunde_auth.forbidden": (ugettext_lazy("Verboten"), "%(AUTH_ROOTURL)s/forbidden/", None, None, None),
    "releasenotes.index": (ugettext_lazy("Versionshinweise"), "%(RELEASENOTES_ROOTURL)s/", "service", None, ['releasenotes.create_note', 'releasenotes.close_bugs']),
    "releasenotes.edit_note": (ugettext_lazy("Hinweis bearbeiten"), "%(RELEASENOTES_ROOTURL)s/note/%(ticket)s/", "releasenotes.index", None, None),
    "releasenotes.create_note": (ugettext_lazy("Hinweis anlegen"), "%(RELEASENOTES_ROOTURL)s/create-note/", "releasenotes.index", None, None),
    "releasenotes.close_bugs": (ugettext_lazy("Meldungen abschliessen"), "%(RELEASENOTES_ROOTURL)s/close-bugs/", "releasenotes.index", None, None),
    "releasenotes.admin": (ugettext_lazy("Admin-Interface"), "%(RELEASENOTES_ROOTURL)s/admin/note/", "releasenotes.index", None, None),
    
    "hardware.hw_search": (ugettext_lazy("Hardware suchen"), "%(HARDWARE_ROOTURL)s/search/", "hardware", None, None),
    "hardware.hw_copy": (ugettext_lazy("Hardware kopieren"), "%(HARDWARE_ROOTURL)s/copy/%(id)s/", "hardware.hw_edit", None, None),
    "hardware.hw_copy_panel_prefix": (ugettext_lazy("Hardware kopieren"), "%(HARDWARE_ROOTURL)s/panel/copy/", "hardware.hw_edit", None, None),
    "hardware.hw_create": (ugettext_lazy("Hardware anlegen"), "%(HARDWARE_ROOTURL)s/create/", "hardware", None, None),
    "hardware.hw_create_panel": ("", "%(HARDWARE_ROOTURL)s/panel/create/", "hardware", None, None),
    "hardware.hw_edit": (ugettext_lazy("Hardware bearbeiten"), "%(HARDWARE_ROOTURL)s/%(id)s/", "hardware", None, ['housingdb.edit', "hardware.hw_wartungsvertrag_create"]),
    "hardware.hw_edit_panel": ("", "%(HARDWARE_ROOTURL)s/panel/%(id)s/", None, None, None),
    "hardware.hw_edit_panel_prefix": ("", "%(HARDWARE_ROOTURL)s/panel/", None, None, None),
    "housingdb.edit": (ugettext_lazy("Housing-DB"), settings.HOUSING_DB_URL + "?action=edit_hw;hw=%(id)s", "hardware.hw_edit", None, None),
    "housingdb.rack": (ugettext_lazy("Rack-Ansicht"), settings.HOUSING_DB_URL + "?action=show_rack;rack=%(rack_id)s", "hardware.hw_edit", None, None),
})

StaffNaviNode(tag="mailadmin", label=ugettext_lazy("E-Mail-Konfiguration"), url="%(MAILADMIN_ROOTURL)s/mailbox-qsearch/",
         parent_tag="service", sub_tags=['mailadmin.mailbox_qsearch', 'mailadmin.mailrule_qsearch', 'mailadmin.mailrule_search'])
NaviNode(tag="mailadmin", label=ugettext_lazy("E-Mail-Konfiguration"),
         menu_label=ugettext_lazy(u"E-Mail-Übersicht"),
         url="%(MAILADMIN_ROOTURL)s/portal/",
         parent_tag="service")
StaffNaviNode(tag="mailadmin.portal", label=ugettext_lazy(u"E-Mail-Übersicht"),
              url="%(MAILADMIN_ROOTURL)s/portal/%(kunden_name)s/",
              parent_tag="kunde.overview", alt_parent_tag="mailadmin")
NaviNode(tag="mailadmin.portal", label="%(kunden_name)s", menu_label=ugettext_lazy(u"E-Mail-Übersicht"), url="%(MAILADMIN_ROOTURL)s/portal/%(kunden_name)s/",
         parent_tag="mailadmin")
NaviNode(tag="mailadmin.edit_mailbox", label=ugettext_lazy(u"Postfach '%(mailbox_name)s'"),
         url="%(MAILADMIN_ROOTURL)s/mailbox/%(mailbox_name)s/",
         parent_tag="mailadmin.portal")
NaviNode(tag="mailadmin.open_xchange", label=ugettext_lazy(u"Open XChange auswählen"),
        url="%(MAILADMIN_ROOTURL)s/open-xchange/select/%(kunden_name)s/",
        parent_tag="mailadmin.portal", alt_parent_tag="mailadmin.open_xchange.ohnekunde")
NaviNode(tag="mailadmin.open_xchange.ohnekunde", label=ugettext_lazy(u"Open XChange auswählen"),
        url="%(MAILADMIN_ROOTURL)s/open-xchange/select/",
        parent_tag="mailadmin.portal")
NaviNode(tag="mailadmin.open_xchange_bestellung", label=ugettext_lazy(u"Open XChange bestellen"),
        url="%(MAILADMIN_ROOTURL)s/open-xchange/order/%(kunden_name)s/",
        parent_tag="mailadmin.open_xchange", alt_parent_tag="mailadmin.open_xchange_bestellung.ohnekunde")
NaviNode(tag="mailadmin.open_xchange_bestellung.ohnekunde", label=ugettext_lazy(u"Open XChange bestellen"),
        url="%(MAILADMIN_ROOTURL)s/open-xchange/order/",
        parent_tag="mailadmin.open_xchange.ohnekunde")

InPanelNaviNode(tag="hardware", label=ugettext_lazy("Hardware"), url="%(HARDWARE_ROOTURL)s/", parent_tag="service")

NaviNode(tag="troubletickets", label=ugettext_lazy("Trouble-Tickets"),
         url="%(SERVICE_URL)s/trouble-ticket/", parent_tag="service")
NaviNode(tag="rz_zutritt", label=ugettext_lazy("RZ-Zutritt"),
         url="%(SERVICE_URL)s/rzaccess/", parent_tag="service")
NaviNode(tag="powercycle", label=ugettext_lazy("Powercycle"),
         url="%(SERVICE_URL)s/powercycle/", parent_tag="service")
NaviNode(tag="accounting", label=ugettext_lazy("Accounting"), parent_tag="service",
         sub_tags=["accounting.ip", "accounting.radius", "accounting.traveller"])
NaviNode(tag="accounting.ip", label=ugettext_lazy("IP"), parent_tag="accounting", url="%(SERVICE_URL)s/ipacct/")
NaviNode(tag="accounting.radius", label=ugettext_lazy("Radius"), parent_tag="accounting", url="%(SERVICE_URL)s/radacct/")
NaviNode(tag="accounting.traveller", label=ugettext_lazy("Traveller"), parent_tag="accounting", url="%(SERVICE_URL)s/ipassacct/")
NaviNode(tag="capacity", label=ugettext_lazy("Capacity Management"), parent_tag="service.menu",
         sub_tags=["capacity.network", "serverstats.overview"])
NaviNode(tag="capacity.network", label=ugettext_lazy("Netzwerk"), parent_tag="capacity", url="%(SERVICE_URL)s/rrd/")
MitKundeNaviNode(tag="serverstats.overview", label=ugettext_lazy(u"Server"), parent_tag="capacity", url="%(SERVERSTATS_ROOTURL)s/%(kunden_name)s/")
NaviNode(tag="serverstats.overview", label=ugettext_lazy(u"Server"), parent_tag="capacity", url="%(SERVERSTATS_ROOTURL)s/")
NaviNode(tag="monitoring", label=ugettext_lazy("Monitoring"), parent_tag="service",
         sub_tags=["monitoring.end_to_end"])
NaviNode(tag="monitoring.end_to_end", label=ugettext_lazy("End-To-End"), parent_tag="monitoring", url="%(SERVICE_URL)s/end2end/")

NaviNode(tag="hardware.hw_create_panel_js", label=ugettext_lazy("Hardware anlegen"),
         url='javascript:ResultGridLoader.showCreatePanel()', parent_tag="hardware")
NaviNode(tag="hardware.hw_wartungsvertrag_create_panel_js", label=ugettext_lazy("Wartungsvertrag anlegen"),
         url="javascript:ResultGridLoader.showCreateWartung(%(id)s, '%(HARDWARE_ROOTURL)s/panel/%(id)s/wartungsvertrag/create/')", parent_tag="hardware")
NaviNode(tag="hardware", label=ugettext_lazy("Hardware-Suche"), url="%(HARDWARE_ROOTURL)s/", parent_tag="service")
NaviNode(tag="hardware.hw_search_new", label=ugettext_lazy("Neue Suche"), url="%(HARDWARE_ROOTURL)s/search/", parent_tag="hardware")
NaviNode(tag="hardware.hw_search_panel", label=ugettext_lazy("Hardware suchen"), url="%(HARDWARE_ROOTURL)s/panel/search/", parent_tag="hardware")
NaviNode(tag="hardware.hw_wartungsvertrag_create", label=ugettext_lazy("Wartungsvertrag anlegen"),
         url="%(HARDWARE_ROOTURL)s/%(id)s/wartungsvertrag/create/", parent_tag="hardware.hw_edit")
NaviNode(tag="hardware.hw_wartungsvertrag_create_panel", label=ugettext_lazy("Wartungsvertrag anlegen"),
         url="%(HARDWARE_ROOTURL)s/panel/%(id)s/wartungsvertrag/create/", parent_tag="hardware.hw_edit_panel")
NaviNode(tag="hardware.hw_wartungsvertrag_edit", label=ugettext_lazy("Wartungsvertrag bearbeiten"),
         url="%(HARDWARE_ROOTURL)s/%(id)s/wartungsvertrag/%(wv_id)s/", parent_tag="hardware.hw_edit")
NaviNode(tag="hardware.hw_wartungsvertrag_edit_panel", label=ugettext_lazy("Wartungsvertrag bearbeiten"),
         url="%(HARDWARE_ROOTURL)s/panel/%(id)s/wartungsvertrag/%(wv_id)s/", parent_tag="hardware.hw_edit_panel")
NaviNode(tag="hardware.hw_wartungsvertrag_copy", label=ugettext_lazy("Wartungsvertrag kopieren"),
         url="%(HARDWARE_ROOTURL)s/%(id)s/wartungsvertrag/copy/%(wv_id)s/", parent_tag="hardware.hw_wartungsvertrag_edit")
NaviNode(tag="hardware.hw_wartungsvertrag_copy_panel", label=ugettext_lazy("Wartungsvertrag kopieren"),
         url="%(HARDWARE_ROOTURL)s/panel/%(id)s/wartungsvertrag/copy/%(wv_id)s/", parent_tag="hardware.hw_edit_panel")
InPanelNaviNode(tag="hardware.menu", label=ugettext_lazy(u"Hardware"),
                sub_tags = ["hardware.hw_search_new", "hardware.hw_create_panel_js"])
NaviNode(tag="hardware.menu", label=ugettext_lazy(u"Hardware"), sub_tags=["kunde.hardware", "hardware.hw_create"])


NaviNode(tag="kunde", label=ugettext_lazy("Kundenauswahl"), url="%(KUNDE_ROOTURL)s/", parent_tag="service", sub_tags=["kunde.kunden_suche"])
NaviNode(tag="kunde.kunden_suche", label=ugettext_lazy("Kunden-Suche"), url="%(KUNDE_ROOTURL)s/", parent_tag="kunde")
NaviNode(tag="kunde.overview", label=u"%(kunden_name)s", menu_label=ugettext_lazy(u"Kunden-Übersicht"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/", parent_tag="kunde")
NaviNode(tag="kunde.unterkunden", label=ugettext_lazy("Unterkunden"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/unterkunden/", parent_tag="kunde.overview")
NaviNode(tag="kunde.overview_by_id", label=ugettext_lazy("Kundenuebersicht"), url="%(KUNDE_ROOTURL)s/%(kunden_id)s/id/",
parent_tag="kunde")
NaviNode(tag="kunde.personen", label=ugettext_lazy("Personenliste"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/personen/", parent_tag="kunde.overview")
NaviNode(tag="kunde.accounts", label=ugettext_lazy("Benutzerkonten"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/accounts/", parent_tag="kunde.overview")
NaviNode(tag="kunde.person", label=ugettext_lazy("Person"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/person/%(person_id)s/", parent_tag="kunde.personen")
NaviNode(tag="kunde.create_person", label=ugettext_lazy("Person anlegen"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/create-person/", parent_tag="kunde.personen")
NaviNode(tag="kunde.domains", label=ugettext_lazy("Domainliste"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/domains/", parent_tag="kunde.overview")
NaviNode(tag="kunde.ips", label=ugettext_lazy("IP-Liste"), url="%(KUNDE_ROOTURL)s/%(kunden_name)s/ips/", parent_tag="kunde.overview")
NaviNode(tag="kunde.hardware", label=ugettext_lazy("Hardwareliste"), url='%(KUNDE_ROOTURL)s/%(kunden_name)s/hardware/', alt_parent_tag="hardware", parent_tag="kunde.overview")
NaviNode(tag="kunde.wiki", label=ugettext_lazy("Wiki-Eintrag"),
         url=settings.WIKI_URL + '/Kunde:%(kunden_name)s', parent_tag="kunde")
NaviNode(tag="kunde.monitoring", label=ugettext_lazy("Monitoring"),
         url=settings.NAGIOS_URL + '/status.cgi?hostgroup=%(kunden_name)s-bf&amp;style=detail', parent_tag="kunde")
NaviNode(tag="tickets", label=ugettext_lazy("Tickets"), url=settings.RT_URL)
NaviNode(tag="kunde.mailadmin", label=ugettext_lazy("E-Mail-Konfiguration"),
         url="%(MAILADMIN_ROOTURL)s/portal/%(kunden_name)s/", parent_tag="kunde.overview",
         sub_tags=[('mailadmin.mailbox_qsearch.kunde', 'mailadmin.create_mailbox'),
         ('mailadmin.mailrule_qsearch.kunde', 'mailadmin.mailrule_search.kunde',  'mailadmin.create_mailrule')])


NaviNode(tag="secure_login", label=ugettext_lazy("Kundebunt via OTP"), url=settings.SECURE_LOGIN)
NaviNode(tag="external_login", label=ugettext_lazy("externes Einloggen"), url="https://service.noris.net/auth/login/")
NaviNode(tag="internal_login", label=ugettext_lazy("internes Einloggen"), url="https://intra.office.noris.de/kundebunt/auth/login/")

MitKundeStaffNaviNode(tag="history.menu", label=ugettext_lazy(u"Historie"), sub_tags=["history.kunde", "history.hardware"])
NaviNode(tag="history.kunde", label=ugettext_lazy(u"Kunden-Historie"),
         url="%(HISTORY_ROOTURL)s/kunde/%(kunden_name)s",
         parent_tag="kunde.overview")
NaviNode(tag="history.hardware", label=ugettext_lazy(u"Hardware-Historie"),
         url="%(HISTORY_ROOTURL)s/hardware/%(hardware_db_id)s",
         parent_tag="hardware.hw_edit")
NaviNode(tag="history.ipkunde", label=ugettext_lazy(u"Ipkunde-Historie"), parent_tag="history.kunde",
         url="%(HISTORY_ROOTURL)s/ipkunde/%(id)s/")
NaviNode(tag="history.person", label=ugettext_lazy(u"Personen-Historie"), parent_tag="kunde.person",
         url="%(HISTORY_ROOTURL)s/person/%(id)s/")
NaviNode(tag="history.mailrule", label=ugettext_lazy(u"Mailrule-Historie"), parent_tag="mailadmin.edit_mailrule",
         url="%(HISTORY_ROOTURL)s/mailrules/%(id)s/")
NaviNode(tag="history.generic", label=ugettext_lazy(u"%(tabelle)s-Historie"), parent_tag="kunde",
         url="%(HISTORY_ROOTURL)s/%(tabelle)s/%(id)s/")

NaviNode(tag="user.download_header", label=ugettext_lazy(u"Download-Bereich"), parent_tag="service",
         url="/download/")
NaviNode(tag="download", label=ugettext_lazy(u"Download-Bereich"), parent_tag="service", url="%(SERVICE_URL)s/download/")

NaviNode(tag="otrs.login", label=ugettext_lazy(u"OTRS-Login"), 
         parent_tag="service", url="/otrsdata/login/")
NaviNode(tag="otrs.logout", label=ugettext_lazy(u"OTRS-Logout"),
         parent_tag="service", url="/otrsdata/logout/")

# data sources

def navi_path(context):
    """Returns a list of navigation elemens that describe a hierarchical navigation path.
    Each path info element is a tuple (url, descriptive_text, menu_list <[string, string, ...]>)
    """
    path = []
    current = "%s.%s" % (_navi_view.module.split(".")[-2], _navi_view.view_name,)
    context_proxy = NaviContextProxy([context, _navi_view.kwargs], False, KeyError, allow_none=False)
    while current != None:
        while True:
            try:
                node = navimap.get_valid_node(current, context)
                text, url, current, icons, menu = node.tuple
            except KeyError:
                text, url, current, icons, menu = navimap.get_valid_node("service", context).tuple
            try:
                text = text % context_proxy
                if url is not None:
                    url = url % context_proxy
                path.append((url, text, icons, menu))
                break
            except KeyError:
                if node.alt_parent_tag is not None:
                    current = node.alt_parent_tag
                else:
                    break
    path.reverse()
    return path

def navi_menubar(context):
    context_proxy = NaviContextProxy([context, _navi_view.kwargs], False, KeyError, allow_none=False)
    result = []
    for tag in navimap.get_valid_node("menubar", context).sub_tags:
        try:
            node = navimap.get_valid_node(tag, context)
        except KeyError:
            node = None
        while node is not None:
            try:
                descr = (node.menu_label or node.label) % context_proxy
                url = node.url
                if url is not None:
                    url = url % context_proxy
                break
            except KeyError:
                if node.alt_parent_tag is not None:
                    node = navimap.get_valid_node(node.alt_parent_tag, context)
                elif node.parent_tag is not None:
                    node = navimap.get_valid_node(node.parent_tag, context)
                else:
                    node = None
        if node:
            result.append((None, descr, node.icon_url, node.sub_tags))
    return result

def navi_info(view_tag, context, add_default_context=False, add_view_context=False, for_menu=False, error_cls=KeyError):
    """returns a tuple (url, descriptive_text, icon) for a given view tag (a string).

    >>> navi_info('kunde_auth.logout', navigation_context_processor(None))
    (u'/auth/logout/', u'Ausloggen', None, None)
    """
    node = navimap.get_valid_node(view_tag, context)
    text, url, parent, icon, menu_list = node.tuple
    if for_menu and node.menu_label:
        text = node.menu_label
    if add_view_context:
        context_proxy = NaviContextProxy([context, _navi_view.kwargs], add_default_context, error_cls=error_cls)
    else:
        context_proxy = NaviContextProxy([context], add_default_context, error_cls=KeyError)
    #try:
    if url is None:
        return(None, text % context_proxy, icon, menu_list)
    else:
        return (url % context_proxy, text % context_proxy, icon, menu_list)
    #except ValueError:
        #return navi_info(parent, context, add_default_context, add_view_context)

def navi_url(view_tag, context, add_default_context=False):
    return navi_info(view_tag, context, add_default_context)[0]

def navi_url_pattern(view_tag, context, add_default_context=False, pk_context="id"):
    """Returns a pattern for the url for passing to a generic create_view."""
    context.push()
    context[pk_context] = "%(id)s"
    url = navi_url(view_tag, context, add_default_context)
    context.pop()
    return url

def navi_tag_exists(view_tag):
    """Checks if the view_tag exists.

    >>> navi_tag_exists("mailadmin.portal")
    True
    >>> navi_tag_exists("mailadmin.krmlwml")
    False
    """
    return view_tag in navimap
