# -*- encoding: utf-8 -*-

"""
Modelle für alles, was wir aus der POP-Datenbank brauchen.
"""

# OBACHT: app_label ist fuer pychecker noetig,
# aber sycndb (und damit die Testsuite) gehen
# nur ohne app_label :-(
#app_label = "kundebunt.popkern"

import re, socket, time, datetime
from struct import pack, unpack
from socket import ntohl, htonl, inet_aton, inet_ntoa, inet_ntop, inet_pton, AF_INET6, AF_INET, error as socket_error
from itertools import chain
from types import GeneratorType

from django import db
from django.db import models
from django.core import validators
from django.conf import settings
from django.utils.translation import gettext, gettext_lazy, ngettext, ugettext_lazy
from django.utils.encoding import smart_str

from kundebunt.popkern.updatelog import UpdatelogMixin
from kundebunt.popkern.navigation import navi_url
from kundebunt.popkern.fields import *
from kundebunt.popkern.utils import glob_to_like, time_to_str, is_active, now, glob_to_regex, any, unique, merge_sorted


OPEN_XCHANGE_DISLAY = {
    'off': ugettext_lazy(u'deaktiviert'),
    'Webmail+': ugettext_lazy(u'Webmail+'),
    'PIM+': ugettext_lazy(u'Noris Groupware'),
    'Groupware+': ugettext_lazy(u'Noris Groupware+'),
    'Premium': ugettext_lazy(u'Noris Groupware Premium'),
    }

class FoundMatch(Exception):
    def __init__(self, matches=None):
        self.matches = matches

###########################################################
# Keller: Deskriptoren

class DescrTyp(models.Model):
    """
    Verwendung: Deskriptoren-Name

    Index:  id, name
    Index:  timestamp
    Aufbau:
    id  int4    !^!?+   eindeutige Nummer
    name    char30  !^  Name der Subtabelle
    info    char255 !-  kurze Erklärung
    timestamp   datetime    !/+  Zeitstempel
    """
    name = models.CharField(unique=True, max_length=30)
    info = models.CharField(max_length=255, blank=True, null=True)
    timestamp = models.DateTimeField(auto_now=True)
    class Meta:
        db_table = 'descr_typ'
    def __unicode__(self):
        return self.name

class Descr(models.Model):
    """
    Verwendung: Mappt numerische Werte auf lesbare Beschreibungen.

    Index:  id, typ descr, typ bla
    Index:  timestamp
    Aufbau:
    id  int4    !?+ ID
    typ int4    !>descr_typ Subtabelle
    descr   int4    -   numerischer Wert, oder ASCII des Zeichens
    bla char30 -   textuelle Beschreibung (kurz)
    idchar  char1   !-!%case_sens!%utf8 Kurzzeichen für die Übersicht
    gruppe  int4    !*<name>_ident  Untergruppen
    infotext    char255 !-  kurze Erklärung
    timestamp   datetime    !/+  Zeitstempel
    """
    descr = models.IntegerField(db_index=True)
    typ = models.ForeignKey(DescrTyp, db_column='typ')
    bla = models.CharField(max_length=30)
    idchar = models.CharField(max_length=1, null=True, blank=True)
    gruppe = models.IntegerField()
    infotext = models.CharField(max_length=255, blank=True, null=True)
    timestamp = models.DateTimeField(auto_now=True)
    class Meta:
        db_table = 'descr'
    def __unicode__(self):
        return self.bla

class DbTabelle(models.Model):
    """
    Verwendung: speicher für Tabellendefs

    Index:  id
    Index:  timestamp
    Aufbau:
    id  int4    !?+
    name    char30  !^  Tabellenname
    text    char100 -   Verwendungstext der Tabelle
    timestamp   datetime    !/+ Zeitstempel
    """
    name = models.CharField(max_length=30)
    text = models.CharField(max_length=100)
    timestamp = models.DateTimeField(auto_now = True)
    class Meta:
        db_table = 'db_tabelle'
    def __unicode__(self):
        return self.name

class Dienst(models.Model):
    """
    Verwendung: Speichert die Daten zu einem Dienst (war: tarifeq-Tabelle)

    Index:  id, name
    Index:  timestamp
    Aufbau:
    id  int2    !?+         ID
    timestamp   datetime    !/+  Zeitstempel
    name    char15  .   Name dieses Diensts
    info    char100 !-  Beschreibung, was da accountet wird
    rechnungstext   char100 !-  Bezeichnung des Dienstes auf Rechnungen (falls NULL, wird der dienst.name verwendet
    """
    timestamp = models.DateTimeField(auto_now=True)
    name = models.CharField(unique=True, max_length=15)
    info = models.CharField(blank=True, null=True, max_length=100)
    rechnungstext = models.CharField(blank=True, null=True, max_length=100)

    class Meta:
        db_table = 'dienst'

    def __unicode__(self):
        return self.name


class KundenQuerySet(PopQuerySet):
    """4
    QuerySet für Kunden; hat zusätzliche Methoden zur Verknüpfung Personen/Kunden über Kundemail.
    """
    def active(self):
        """Gibt ein QuerySet mit allen Kunden, die einen laufenden Vertrag haben, zurueck
        """
        return self.extra(
            where=["""(kunde.ende IS NULL
                       OR kunde.ende = 0
                       OR kunde.ende >= UNIX_TIMESTAMP(NOW()))"""])

    def managed_by(self, person_id, dienst_namen=('service',)):
        """Gibt ein QuerySet zurueck mit allen Kunden, die:
        - "active" sind (s.o.)
        - und zusätzlich in kundemail mit der angegebenen Person
          über mindestens einen der angegebenen Dienste verknüpft sind.
        """
        dienst_liste = u"(%s)" % u','.join([unicode(Kundemail.get_dienst_code(name)) 
                                            for name in dienst_namen])
        return self.active().extra(
            params=[person_id],
            tables=["kundemail"],
            where=["""kunde.id = kundemail.kunde
                    and kundemail.person = %s
                    and kundemail.dienst in """ + dienst_liste + """
                    and (kunde.ende IS NULL OR kunde.ende = 0
                        OR kunde.ende >= UNIX_TIMESTAMP(NOW()))"""]).distinct()



###########################################################
# Erdgeschoss: Kunden und Personen



class KundenManager(PopManager):

    _numeric_re = re.compile(r'\s*#?\s*([1-9][0-9]*)(\s.*)?$')
    _name_re = re.compile(r'\s*(\S*)(\s.*)?$')
    _re_domain_id = re.compile(ur'^domain#(\d+)$')
    _re_rechnung = re.compile(ur'^R(\d+)')
    #_re_ipkunde = re.compile(ur'^([\d*]+\.\d+\.\d+\.\d+)(?:/(\d+))?$')

    def get_query_set(self):
        return KundenQuerySet(self.model)

    def active(self):
        return self.get_query_set().active()

    def managed_by(self, admin_id, dienst_namen=('service',)):
        return self.get_query_set().managed_by(admin_id, dienst_namen=dienst_namen)

    def default_query(self, value, model = None, join_condition = None, queryset_required=True, table_suffix=""):
        if model:
            assert(join_condition)
            join_condition = [join_condition]
            join_tables = ["kunde kunde%s" % table_suffix]
            manager = model.objects
        else:
            manager = self
            join_condition = []
            join_tables = []
        if isinstance(value, Kunde):
            objs = [value]
        else:
            objs = KundenManager._get_objects(value)
        if manager==self and not queryset_required:
            return objs
        else:
            if objs:
                if isinstance(objs, (list, GeneratorType)):
                    ids = [p.id for p in objs]
                else:
                    ids = objs.valuelist("id")
                if not ids:
                    return EmptyQuerySet()
                return manager.extra(
                    where=[u"""(kunde%s.id in (%s))""" % (table_suffix, u",".join(unicode(id) for id in ids))] + join_condition,
                    tables = join_tables
                    )
            else:
                return EmptyQuerySet()


    @staticmethod
    def _get_objects(data):
        if data in (u"", u"*", None):
            return Kunde.objects.all()
        results = []
        exclude = False     # Suchergebnis negieren
        sub_too = False    # auch Unterkunden?
        if data.startswith(u"^"):
            exclude = not exclude
            data = data[1:]
        if data.endswith(u"/"):
            sub_too = True
            data = data[:-1]
        get_all_matches = u"*" in data or u"?" in data
        if get_all_matches:
            def handle_qset(qset, final=False):
                if qset:
                    results.append(qset)
                    if final:
                        raise FoundMatch
        else:
            def handle_qset(qset, final=False):
                if qset:
                    results.append(qset)
                    raise FoundMatch
        globbed = glob_to_like(data)
        kunden = Kunde.objects
        try:
            # Kunde-ID
            if data.isdigit():
                handle_qset(kunden.filter(id=int(data)), final=True)
            # Domain-ID
            match = KundenManager._re_domain_id.match(data)
            if match:
                try:
                    handle_qset([Domainkunde.objects.get(id=match.group(1)).kunde],
                                final=True)
                except Domainkunde.DoesNotExist:
                    pass
            # Rechnungs-Nr.
            match = KundenManager._re_rechnung.match(data)
            if match:
                try:
                    handle_qset([Rechnung.objects.get(rnr=match.group(1)).kunde],
                                final=True)
                except Rechnung.DoesNotExist:
                    pass
            # Kunden-Name
            handle_qset(kunden.filter(name__like=globbed).order_by("id"))
            # Kunden-Alias
            handle_qset(kunden.active().filter(uucpkunde__name__like=globbed).order_by("id"))
            # Domain-Name
            ids = Domainkunde.objects.active().filter(domain__like=globbed).valuelist("kunde")
            if ids:
                handle_qset(kunden.filter(id__in=ids).order_by("id"))
            # IP-Adresse
            if Ipkunde._ip4_wildcard_re.match(data) or Ipkunde._ip4_subnet_re.match(data) or ":" in data:
                ids = Ipkunde.objects.active().filter(Ipkunde.q_ip_glob(data)).valuelist("kunde")
                if ids:
                    handle_qset(kunden.filter(id__in=ids).order_by("id"))
            # Personen-Benutzername
            handle_qset(kunden.active().filter(person__user__like=globbed).order_by("id"))
            # Personen-Email
            handle_qset(kunden.active().filter(person__email__like=globbed).order_by("id"))
            # Beendete Objekte
            if data.startswith("!"):
                now = datetime.datetime.now()
                # Domain
                doms = Domainkunde.objects.filter(domain__like=globbed[1:], ende__gt=0, ende__lt=now).order_by("domain", "-beginn")
                doms = unique(doms, lambda dom: dom.domain)  # nur die zuletzt begonnene zu jedem Domain-Namen
                handle_qset(list(sorted((dom.kunde for dom in doms), key=lambda k: k.id)))
                # IP
                if Ipkunde._ip4_wildcard_re.match(data[1:]) or Ipkunde._ip4_subnet_re.match(data[1:]):
                    ips = (Ipkunde.objects.default_query(data[1:])
                            .filter(ende__gt=0, ende__lt=now).order_by("ip6", "bits", "-beginn"))
                    ips = unique(ips, lambda ip:(ip.ip6, ip.bits))  # nur die zuletzt begonnene ip
                    handle_qset(list(ordered((ip.kunde for ip in ips), key=lambda k: k.id)))
            # Hardware-ID
            ids = Hardware.objects.active().filter(hardware_id__like=globbed).valuelist("kunde")
            if ids:
                handle_qset(kunden.filter(id__in=ids).order_by("id"))
            # NIC-Handle
            handle_qset(kunden.filter(person__nic__handle__like=globbed).order_by("id"))
        except FoundMatch:
            pass
        if len(results) > 1:
            def get_iterator(collection):
                if isinstance(collection, QuerySet):
                    return collection.iterator()
                else:
                    return iter(collection)
            results = unique(merge_sorted((get_iterator(collection) for collection in results), lambda p:p.id), lambda p:p.id)
        elif len(results) == 1:
            results = results[0]
        if sub_too:
            # Unterkunden dazu
            results = list(results)
            seen = set(k.id for k in results)
            i = 0
            while i < len(results):
                for k in results[i].unterkunde.all():
                    if k.id not in seen:
                        results.append(k)
                        seen.add(k)
                i += 1
        if exclude:
            results = kunden.exclude(id__in=[k.id for k in results])
        return results


class Kunde(UpdatelogMixin, models.Model):
    """
    Verwendung: Kundendatenbank

    Index:  id, name
    Index:  timestamp, reseller, kunde, ende id
    Aufbau:
    id  int4    !?+ ID-Nummer des Kunden
    timestamp   datetime    !/+  Zeitstempel
    kunde   int4    !-!>kunde   Übergeordneter Kunde
    reseller    int4    !-!>kunde   Reseller, der für diesen Kunden aktiv sein darf
    name    char32  !^  Kurzname des Kunden (weitere -> uucpkunde)
    exttyp  char    !##!-!ziel  Typkennung für Traffic eines anderen Kunden an diesen Kunden
                (normalerweise "n"; POP ist "l", Xlink-intern "x").
    beginn  uint4    !/  Zeitpunkt Vertragsabschluß (Kunde)
    ende    uint4    !-!/    Kündigungstermin. Kunde:0 nie Kunde:1.
    zuletzt int4    !-!(JJJJMM) nächster zu berechnender Monat
    berechne    char    !intervall.m wie oft wird berechnet? 'm'onat 'q'uartal 'j'ahr 'x'=manuell
    kprio   char    !-!kprio    wie wichtig ist der Kunde?
    kontogr int2    !##!-!konten    Kontengruppe
    geaendert uint4    !-!/    Zeitpunkt der letzten Änderung an diesem Kunden;
                verwendet zum Rechnungssortieren
    flags   int4    !*kunde Flags dieses Kunden
    pwklasse    int2    !-!passwort Hint, welches Passwort für den Kunden verwendet wurde
    skip    char    !bool.n überspringe in diesem Rechnungslauf
    erloeskonto int4    !-  Buchungs-Kontonr für Rechnung
    steuernr    char20  !-  UST-ID-Nr
    zahlung int1    !-  Zahlungsziel (NULL == Default)
    sprache int2    !-!sprache  Default-Sprache, insb. für Postfächer (RT#224274)
    hauptperson int4    !-!>person  Adresse etc. des Kunden (ehemals :-0)
    billc   int4    !-!>person  Rechnungsanschrift des Kunden (ehemals billc-0)
    adminc  int4    !-!>person  Ansprechpartner für Domains (ehemals nic/adminc-0)
    ap_vertrieb int4    !-!>person  vertrieblicher Ansprechpartner, vgl. RT#273307
    ap_technik  int4    !-!>person  technischer Hauptansprechpartner, vgl. RT#273307
    """
    kunde = models.ForeignKey('self', related_name='unterkunde', null=True, db_column='kunde', blank=True, raw_id_admin=True)
    name = models.CharField(unique=True, max_length=32)
    beginn = models.UnixDateTimeField(default=0)
    ende = models.UnixDateTimeField(null=True, blank=True)
    zuletzt = models.IntegerField(null=True, blank=True)
    berechne = CharDescriptorField(Descr, "intervall", db_column="berechne", default="m", max_length=1)
    kprio = CharDescriptorField(Descr, "kprio", db_column="kprio", blank=True, null=True, max_length=1)
    geaendert = models.UnixDateTimeField(null=True, blank=True)
    flags = DescriptorSet(Descr, "kunde", default=0)
    pwklasse = DescriptorField(Descr, "passwort", null=True, blank=True, db_column="pwklasse")
    skip = CharDescriptorField(Descr, "bool", db_column="skip", default="n", max_length=1)
    timestamp = models.DateTimeField(auto_now=True)
    erloeskonto = models.IntegerField(null=True, blank=True)
    reseller_id = models.ForeignKey('self', related_name='kunde_by_reseller', null=True, db_column='reseller', blank=True, raw_id_admin=True)
    steuernr = models.CharField(blank=True, null=True, max_length=20)
    zahlung = models.IntegerField(blank=True, null=True, max_length=4)
    sprache = DescriptorField(Descr, "sprache", null=True, blank=True, db_column="sprache")
    hauptperson = models.ForeignKey("Person", related_name="hauptperson_for", blank=True, null=True, db_column="hauptperson")
    billc = models.ForeignKey("Person", blank=True, null=True, related_name="billc_for", db_column="billc")
    adminc = models.ForeignKey("Person", blank=True, null=True, related_name="adminc", db_column="adminc")
    ap_vertrieb = models.ForeignKey("Person", null=True, blank=True, verbose_name="Ansprechpartner Vertrieb", db_column="ap_vertrieb", related_name="kunde_by_ap_vertrieb")
    ap_technik = models.ForeignKey("Person", null=True, blank=True, verbose_name="Ansprechpartner Technik", db_column="ap_technik", related_name="kunde_by_ap_technik")
    mail_ip = models.PositiveIntegerField(null=True, blank=True)

    objects = KundenManager()

    class Meta:
        verbose_name = 'customer'
        db_table = 'kunde'
        ordering = ['name']
    def __unicode__(self):
        return self.name
    def mailbox_set(self):
        """all mailboxes of this Kunde (as Person instances)
        """
        return self.person_set.all().has_flag("pwuse","mail")
    def navi_context(self):
        """{% navilink %} benutzt das, um zusätzlichen Context zum Auffinden einer URL zu holen."""
        return {'id': self.id, 'kunden_name': self.name, 'kunde_id': self.id}

    def get_absolute_url(self):
        return navi_url('kunde.overview', self.navi_context(), True)

    def abgelaufen(self):
        """Wenn `ende` erreicht ist, hat es den Wert von `ende`. Andernfalls None."""
        if self.ende and self.ende < now():
            return self.ende
        else:
            return None

    def hat_open_xchange_entschieden(self):
        return OpenXChange.objects.filter(kunde=self).exists()

    def open_xchange_status_display(self):
        try:
           last_xchange = OpenXChange.objects.filter(kunde=self, beginn__lt=datetime.datetime.now()).latest("beginn")
        except OpenXChange.DoesNotExist:
            return ugettext(u"keine Angabe")
        return OPEN_XCHANGE_DISLAY.get(last_xchange.get_option_display())

    def betriebsteam(self):
        """Gibt das Betriebsteam für diesen Kunden zurück ("ITO", "MSM", "MSE")
        Siehe #10096664
        """
        team = None
        if self.ap_technik and self.ap_technik.abt:
            team = settings.BETRIEBSTEAM.get(self.ap_technik.abt.lower())
        if team is None:
            team = settings.BETRIEBSTEAM.get(u'*')
        return team


class Nextid(models.Model):
    """
    Verwendung: Speicherung des nächsten ID-Werts
        (nicht alle Datenbanken können atomare sequentielle Werte)

    Index:  name
    Index:  timestamp
    Aufbau:
    id  int4    Zähler
    name    char255 -   ID
    timestamp   datetime    !/+  Zeitstempel

    ID-Liste:   person ipkunde domain tarif kunde stunden
    """
    id = models.IntegerField()
    name = models.CharField(primary_key=True, max_length=255)
    timestamp = models.DateTimeField(auto_now=True)
    class Meta:
        db_table = 'nextid'

class UpdatelogSpalten(models.Model):
    """
    Verwendung: Spaltenreferenz für Änderungslog

    Index:  id, namen
    Index:  timestamp
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+  Zeitstempel
    namen   char255 -   Spaltennamen
    """
    timestamp = models.DateTimeField(auto_now=True)
    namen = models.CharField(unique=True, max_length=255)
    class Meta:
        db_table = 'updatelog_spalten'
    def __unicode__(self):
        return self.namen

class DbFeld(models.Model):
    """
    Verwendung: speichern der Felddefs

    Index:  id, db_tabelle name
    Index:  timestamp
    Aufbau:
    id  int4    !?+
    db_tabelle  int4    !>db_tabelle
    name    char30  !^  Feldname
    typ char8   -   Datentyp
    beschreib   char100 -   Beschreibung des Inhalts, der Maschienenverwertbar ist
    rem char200 -   Bemerkungen zum Feld und Beschreibung
    timestamp   datetime    !/+  Zeitstempel
    """
    db_tabelle = models.ForeignKey(DbTabelle, db_column='db_tabelle', raw_id_admin=True)
    name = models.CharField(unique=True, max_length=30)
    typ = models.CharField(max_length=8)
    beschreib = models.CharField(max_length=100)
    rem = models.CharField(max_length=200)
    timestamp = models.DateTimeField(auto_now=True)
    class Meta:
        db_table = 'db_feld'

MAILRULE_TYP_DISPLAY = {
    u'dev_null':             ugettext_lazy(u'Verwerfen'),
    u'domain_mapping':       ugettext_lazy(u'Domain umschreiben'),
    u'sms1':                 ugettext_lazy(u'SMS senden an'),
    u'sms10':                ugettext_lazy(u'SMS senden (evtl. mehrfach)'),
    u'smtp_route_a':         ugettext_lazy(u'SMTP-Zustellung an Ihren Mail-Server'),
    u'smtp_route_mx':        ugettext_lazy(u'Zustellung anhand des MX-RRsets'),
    u'virt':                 ugettext_lazy(u'Weiterleiten'),
    u'reject':               ugettext_lazy(u'Abweisen'),
    u'defer':                ugettext_lazy(u'Temp. abweisen')
}

MAILRULE_TYP_HELP = (
    (u'virt',           ugettext_lazy('Der Normalfall: Was an die Quell-Adresse geht, wird an die Ziel-Adresse weitergeleitet. '
                                      'Wird nur eine Domain angegeben, gilt das fuer alle Adressen der angegebenen Domain.')),
    (u'domain_mapping', ugettext_lazy('Der Domain-Teil der Adresse wird durch die Ziel-Domain ersetzt.')),
    (u'smtp_route_a',   ugettext_lazy('Die Mail wird an den Mailserver mit dem angegebenen Namen weitergeleitet.')),
    (u'smtp_route_mx',  ugettext_lazy('Die Mail wird an den Mailserver weitergeleitet, der durch das '
                                      'MX-RRset der angegebenen Ziel-Domain bestimmt ist.')),
    (u'reject',         ugettext_lazy('Die Mail wird abgewiesen und der Absender benachrichtigt.')),
    (u'defer',          ugettext_lazy(u'Die E-Mail wird zunächst nicht angenommen (SMTP-Status 451). Die Gegenstelle wird im Regelfall ca. eine Woche lang wiederholt versuchen, sie zuzustellen.'))
    )

def mailrule_typ_help():
    """Gibt den Hilfetext fuer mailrule.typ zurueck. Dieser ist abhängig von den Spracheinstellungen.
    """
    from django.utils.html import escape
    from django.utils.safestring import mark_safe
    return mark_safe(u"<span>%s</span>%s" % (
        ugettext_lazy(u"Die verschiedenen Weiterleitungsarten bedeuten:"),
        u"\n".join([u'<span class="dd">%s</span><span class="dt">%s</span>' % (MAILRULE_TYP_DISPLAY[typ], escape(helptext)) for typ, helptext in MAILRULE_TYP_HELP])))

def quelle_localpart_help():
    """Gibt den Hilfstext für localparts in Quellen zurück. Abhängig von Spracheinstellungen.
    """
    return _("Bei der E-Mail-Adresse wird nicht zwischen Gross- und Kleinschreibung unterschieden. Ist der Teil vor der Domain leer, wird die gesamte Domain weitergeleitet.")

class MailruleManager(PopManager):
    def for_mailbox_query(self, mboxes):
        """
        Gibt ein QuerySet mit allen Mailrules zu den angegebenen mboxes zurück.

        mboxes kann selber ein QuerySet oder ein iterable sein.
        """
        if isinstance(mboxes, QuerySet):
            ziele = [x.values()[0] for x in mboxes.values("user")]
        else:
            ziele = [x.user for x in mboxes]
        if ziele:
            ziele_regex = "|".join((re.escape(z) for z in ziele))
            return Mailrule.objects.active().descr_eq('typ', 'virt')\
            .extra(where=["(ziel regexp %s)"], params=[r"(^|,)(%s)(,|$)" % ziele_regex])
            raise Bla
        else:
            return self.filter(id__isnull=True)[:0]

class Mailrule(UpdatelogMixin, models.Model):
    """
    Verwendung: Mail-Zustellungsregeln (vgl. RT#124007)

    Index:  id, quelle
    Index:  typ, kunde typ, timestamp
    Aufbau:
    id  int4    !?+ ID
    kunde   int4    !>kunde assoziierter Kunde
    quelle  char255 !-  E-Mail-Adresse oder Domain, für die diese Regel gilt (NULL = "*"-Umleitung)
    typ int2    !mailrules  Typ der Regel
    ziel    text2   !-  Umleitungsziel (genaue Bedeutung je nach Typ unterschiedlich)
    timestamp   datetime    !/+  Zeitstempel
    """
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)   # FIXME: Warum mit db_column?
    quelle = models.CharField(unique=True, max_length=255, null=True, blank=False)
    typ = DescriptorField(Descr,"mailrules", db_column="typ")
    ziel = models.TextField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now=True)

    objects = MailruleManager()

    class Meta:
        db_table = 'mailrules'
        verbose_name = 'mail forwarding rule'
    class Admin:
        list_display=["quelle","ziel","timestamp"]
    class Updatelog:
        log_index = "kunde"
        log_related = ["kunde"]
        log_always = ["quelle"]

    def typ_display(self):
        """Gibt eine kundentaugliche Anzeige des Typs zurueck.
        """
        name = self.get_typ_display()
        return MAILRULE_TYP_DISPLAY.get(name, name)   # FIXME

    def typ_specific_help(self):
        """Gibt einen Hilfetext zu dem einen Typ zurück.
        """
        typ_display = self.get_typ_display()
        for (label, helptext) in MAILRULE_TYP_HELP:
            if label == typ_display:
                return helptext
        else:
            return ''

    def get_absolute_url(self):
        return "%s/mailrule/%d" % (settings.MAILADMIN_ROOTURL, self.id)

    def editable(self):            # FIXME: noch benoetigt?
        return True

    def __unicode__(self):
        return u"(%s) %s -> %s" % (self.get_typ_display(), self.quelle, self.ziel)

    def ziel_adressen(self):
        """Bei virt-Regel: Für jeden Empfänger ein Listelelement.
        Ansonsten besteht die Ergebnisliste nur aus einem Element.
        Die Liste besteht aus EmailAddress(en).

        Diese Funktion ist vorwiegend für Anzeigezwecke in Templates gedacht.
        Wenn es keine Ziele gibt, wird ein einziges leeres Zielelement erzeugt.
        """
        from kundebunt.popkern.utils import domain_sort_key, EmailAddress
        if self.typ.bla == 'reject':
            ziele = [self.ziel]
        elif self.ziel == None:
            ziele = []
        else:
            ziele = [EmailAddress(z, self.typ_id!=self.get_typ_code('virt')) for z in self.ziel.split(",")]
        if ziele==[]:
            ziele = [EmailAddress("", False)]
        ziele.sort(key=lambda addr:(domain_sort_key(getattr(addr, "domain", None)), getattr(addr, "localpart", None)))
        return ziele

    def quell_adresse(self):
        from kundebunt.popkern.utils import EmailAddress
        return EmailAddress(self.quelle, True)

    def navi_context(self):
        return {'mailrule_id': self.id, 'id': self.id}

    def matches_email(self, email, domains, rules=None):
        """Gibt zurueck, ob die Regel fuer die Email-Adresse 'email' passt.

        Parameter:
            'email': Die Email-Adresse, der Testkandidat
            'domains': QuerySet von Domainkunde oder Liste aller Domains als [string] des Kunden
            'rules': Nur für die doctests, None oder Liste aller Regelquellen als [string]. Normal ist None.

        >>> rule = Mailrule(quelle='localpart@domain.bla')
        >>> rule.matches_email('localpart@domain.bla', [], [])
        True
        >>> rule.matches_email('localpart.bla', [], [])
        False

        >>> rule = Mailrule(quelle='domain.bla')
        >>> rule.matches_email('localpart@domain.bla', [], [])
        True
        >>> rule.matches_email('localpart@domain', [], [])
        False
        >>> rule.matches_email('localpart@domain.bla', [], ['localpart@domain.bla'])
        False

        >>> rule = Mailrule(quelle='.domain.bla')
        >>> rule.matches_email('localpart@sub.domain.bla', [], ['.domain.bla'])
        True
        >>> rule.matches_email('localpart@domain.bla', [], ['.domain.bla'])
        False
        >>> rule.matches_email('localpart@sub.domain.bla', [], ['.domain.bla', 'sub.domain.bla'])
        False
        >>> rule.matches_email('localpart@sub.domain.bla', [], ['.domain.bla', 'domain.bla'])
        True
        >>> rule.matches_email('localpart@sub.domain.bla', [], ['.domain.bla', '.bla.sub.domain.bla'])
        True
        >>> rule.matches_email('localpart@sub.domain.bla', [], ['.domain.bla', '.sub.domain.bla'])
        True

        >>> rule = Mailrule(quelle=None)
        >>> rule.matches_email('localpart@domain.bla', ['domain.bla'], [])
        True
        >>> rule.matches_email('localpart@domain.bla', ['domain.bla'], ['domain.bla'])
        False
        >>> rule.matches_email('localpart@domain.bla', ['wasanderes.bla'], [])
        False
        >>> # Wildcard-Regeln passen nicht fuer Subdomains von Kundendomains!
        >>> rule.matches_email('localpart@sub.domain.bla', ['domain.bla'], [])
        False
        """
        from kundebunt.popkern.utils import EmailAddress, any
        def _exists_normal_mailrule(email, rules):
            if rules==None:
                return Mailrule.objects.filter(quelle=email).active().exists()
            else:
                return email in rules

        def _exists_domain_mailrule(domain, rules):
            if rules==None:
                return Mailrule.objects.filter(quelle=domain).active().exists()
            else:
                return domain in rules

        def _exists_subdomain_mailrule(domain, rules, min_labels=1):
            labels = domain.split('.')
            doms = ['.' + '.'.join(labels[i:]) for i in range(1, len(labels) - min_labels + 1)]
            if rules==None:
                return Mailrule.objects.active().filter(quelle__in=doms).exists()
            else:
                return any((r in doms for r in rules))

        if not "@" in email:
            # Es gibt keine Regeln, die Postfächer umleiten.
            return False
        if not self.quelle:
            # Wildcard-Regel, gilt fuer alle Domains des Kunden.
            # Passt denn die Domain?
            emailAddress = EmailAddress(email, domain_required=False)
            if isinstance(domains, QuerySet):
                if not domains.filter(domain=emailAddress.domain).exists():
                    # Kunde hat diese Domain nicht, Wildcard-Regel passt definitiv nicht.
                    return False
            else:
                if not emailAddress.domain in domains:
                    return False
            return (not _exists_domain_mailrule(emailAddress.domain, rules)
                and not _exists_subdomain_mailrule(emailAddress.domain, rules)
                and not _exists_normal_mailrule(email, rules)
                )
        elif "@" in self.quelle:
            return email==self.quelle
        elif self.quelle[0]=='.':
            # Dies ist eine Regel für Subdomains
            emailAddress = EmailAddress(email, domain_required=False)
            if not emailAddress.domain.endswith(self.quelle):
                return False  # Subdomain passt nicht
            return (not _exists_normal_mailrule(email, rules)
                and not _exists_domain_mailrule(emailAddress.domain, rules)
                and not _exists_subdomain_mailrule(emailAddress.domain, rules, self.quelle.count(".") + 1)
                )
        else:
            # Dies ist eine Regel für die gesamte Domain
            emailAddress = EmailAddress(email, domain_required=False)
            if not emailAddress.domain==self.quelle:
                return False  # Domain passt nicht
            return not _exists_normal_mailrule(email, rules)

    #def is_editable(self, user_is_staff):
        #"""Darf diese Mailregel geändert oder gelöscht werden?"""
        #from utils import EmailAddress
        #if user_is_staff
            #return True
        #kunden = Kunde.objects.managed_by(person.id)
        #if not self.kunde in kunden:
            #return False
        #addr = EmailAddress(self.quelle, True)
        #addr.check_domain(kunden=kunden)
        #return not addr.foreign_domain

    def is_wildcard_forward(self):
        """Gibt zurück, ob diese Regel eine Weiterleitung ist, die sich
        auf eine ganze Domain (oder mehrere Domains) bezieht.
        """
        return self.typ_id == self.get_typ_code('virt') and (self.quelle is None or not "@" in self.quelle)

    def ziel_ist_meldung(self):
        """Gibt zurück, ob bei dieser Regel die Zielangabe eine Meldung ist
        (und keine E-Mail-Adresse)
        """
        return self.typ_in_group("to_freetext")


class Land(UpdatelogMixin, models.Model):
    """
    Verwendung:     Speichert Daten über ein Land
    Index:  id
    Index:  timestamp
    Aufbau:
    id      int2    !?+     ID
    timestamp       datetime        !/+     Zeitstempel
    kfz     char3   -       ein- bis dreistellige Kennung
    iso2    char2   -       iso 2stellig
    iso3    char3   -       iso 3stellig
    name    char30  -       Name des Landes
    eu      char    !jbool.n    EU-Mitglied?
    """
    timestamp = models.DateTimeField(auto_now=True)
    kfz = models.CharField(max_length=3)
    iso2 = models.CharField(max_length=2)
    iso3 = models.CharField(max_length=3)
    name = models.CharField(max_length=30)
    name_en = models.CharField(max_length=30)
    eu = CharDescriptorField(Descr, 'jbool', db_column='eu', max_length=1, default="n")
    class Meta:
        verbose_name = 'Land'
        db_table = 'land'
        ordering = ["name"]
    class Updatelog:
        log_related = []
    def __unicode__(self):
        return self.name


class Adresse(UpdatelogMixin, models.Model):
    """
    Verwendung:     Speichert Adressdaten.
    Index:  id, kurz
    Index:  timestamp
    Aufbau:
    id      int4    !?+     ID
    timestamp       datetime        !/+     Zeitstempel
    kurz    char40  !-      Kurzname für diese Adresse (z.B. "RZ2")
    land    int2    !>land
    plz     char10  -       Postleitzahl
    ort     char100 -       Ort
    strasse char250 -       der ganze Rest; potenziell mehrzeilig
    """
    timestamp = models.DateTimeField(auto_now=True)
    kurz = models.CharField("Kurzname", max_length=40, null=True, blank=True, unique=True)
    land = models.ForeignKey(Land, db_column="land")
    plz = models.CharField("PLZ", max_length=10)
    ort = models.CharField("Ort", max_length=100)
    strasse = models.CharField("Straße", max_length=250)
    class Meta:
        verbose_name = 'Adresse'
        db_table = 'adresse'
    class Updatelog:
        log_related = []
    def __unicode__(self):
        return u", ".join((x for x in (self.kurz, self.strasse, self.plz, self.ort, self.land and self.land.name) if x))


class PersonManager(PopManager):

    _numeric_re = re.compile(ur'\s*#?\s*([1-9][0-9]*)(\s.*)?$')
    _user_re = re.compile(ur'\s*(\S*)(\s\(.*)?$')
    _uid_re = re.compile(ur'^uid=(\d+)$')
    _fon_re = re.compile(ur'^\+[0-9 *?][0-9 *?]+$')

    def default_query(self, value, model = None, join_condition = None, queryset_required=True, table_suffix=""):
        if model:
            assert(join_condition)
            join_condition = [join_condition]
            join_tables = ["person person%s" % table_suffix]
            manager = model.objects
        else:
            manager = self
            join_condition = []
            join_tables = []
        if isinstance(value, Person):
            personen = [value]
        else:
            personen = PersonManager._get_objects(value)
        if manager==self and not queryset_required:
            return personen
        else:
            if personen:
                if isinstance(personen, (list, GeneratorType)):
                    ids = [p.id for p in personen]
                else:
                    ids = personen.valuelist("id")
                if ids:
                    return manager.extra(
                        where=[u"""(person%s.id in (%s))""" % (table_suffix, u",".join(unicode(id) for id in ids))] + join_condition,
                        tables = join_tables
                        )
        return EmptyQuerySet()
        
    @staticmethod
    def _get_objects(data):
        if data in (u"", u"*", None):
            return Person.objects.all()
        results = []
        get_first_match = u"*" in data or u"?" in data
        if get_first_match:
            def handle_qset(qset):
                if qset:
                    results.append(qset)
        else:
            def handle_qset(qset):
                if qset:
                    raise FoundMatch(qset)
        globbed = glob_to_like(data)
        personen = Person.objects
        # id
        match = PersonManager._numeric_re.match(data)
        if match:
            try:
                return [personen.get(id=int(match.group(1)))]
            except Person.DoesNotExist:
                pass
        try:
            # Benutzername
            match = PersonManager._user_re.match(globbed)
            if match:
                handle_qset(personen.filter(user__like=match.group(1)).order_by("id"))
            # NIC-Handle
            handle_qset(personen.filter(nic__handle__like=globbed).order_by("id"))
            # uid
            match = PersonManager._uid_re.match(data)
            if match:
                handle_qset(personen.active().filter(uid=int(match.group(1))).order_by("id"))
            # name (vollständig aufgeführt)
            handle_qset(personen.active().filter(name__like=globbed).order_by("id"))
            # email
            handle_qset(personen.active().filter(email__like=globbed).order_by("id"))
            # fon/fax/pager
            match = PersonManager._fon_re.match(data)
            if match:
                handle_qset(personen.active().filter(
                        Q(fon__like=globbed)|Q(fax__like=globbed)|Q(pager__like=globbed)
                    ).order_by("id"))
            # name (unter Vortäuschung von Intelligenz)
            handle_qset([p for p in personen.active().filter(name__like=Person.name_like(globbed)).order_by("id")
                    if Person.vergleiche_namen(data, p.name)])
        except FoundMatch, err:
            return err.matches
        if len(results) == 1:
            return results[0]
        else:
            def get_iterator(collection):
                if isinstance(collection, QuerySet):
                    return collection.iterator()
                else:
                    return iter(collection)
            #return results
            return unique(merge_sorted((get_iterator(collection) for collection in results), lambda p:p.id), lambda p:p.id)
        



class Person(UpdatelogMixin, models.Model):
    """
    Verwendung: Speichert die für einen Mitarbeiter einer Firma relevanten Daten.
                Dient auch als Zusatz zur /etc/passwd bei noris. Die hier nicht
                aufgeführten Felder der /etc/passwd werden automagisch generiert.

    Index:  id, suchbegriff, user
    Index:  uid, email, mperson, timestamp, name, uremip, kunde
    Aufbau:
    infotext    char255 !-  (längliche) Beschreibung
    id  int4    !^!?+!%kpersinfo    ID-Nr des Menschen
    kunde   int4    !>kunde "Hauptkunde" des Eintrags
    name    char255 !-  Anrede/Vor/Zuname | Firmenname
    abt char255 !-  Abteilung
    email   char255 !-  Mailadresse
    fon char255 !-!%Fon    Telefonnummer
    fax char255 !-!%Fon Faxnummer
    pager   char255 !-!%Fons    Pager/Handynummer
    isdn    char255 !-!%Fons    ISDN-Nummern
    dest    char4   !-!mailart  Ziele für Nachrichten TODO !*mailart
    adr text2   !##!-  Adresse
    adresse int4 !-!>adresse
    ausweisnr   char255 !-  Personalausweisnummer
    zusatzinfo  char255 !-  Sonstiges
    suchbegriff char255 !-  Suchbegriff
    user    char32 !-  Username
    pass    char255 !-  Paßwort
    uid int4    !-!%UID ID des Users
    homedir char255 !-  Homeverzeichnis / WWW-Unterverzeichnis
    pwuse   int8    !*pwdomain  für was das PW gilt
    redirect    char255 !-  HTTP-Redirect-Ziel
    radiustemplate  int2    !-!radius-templates zu verwendendes Radius-Template, vgl. RT#200867
    maxconn int1    !-!(0-30)   max. Zahl gleichzeitiger Multilink-Verbindungen
    uremip  int4    !-!>ipkunde entfernte Adresse (Transitnetz?) für PPP etc.
    prefcall    int2    !-!mailart  welcher Dienst vorrangig zur Benachrichtigung
                verwendet wird -> descr.mailart
    tarif   int4    !##!-!>tarif.name   wie die Arbeitszeit normalerweise verrechnet wird
    tarifname   int4    !-!>tarifname   wie die Arbeitszeit normalerweise verrechnet wird
    mperson int4    !-!>person  übergeordneter Eintrag?
    gebtag  int2    !-  wann geboren?
    gebjahr int2    !-  wann geboren?
    job char255 !-  was macht der dort?
    timestamp   datetime    !/+  Zeitstempel
    """
    objects = PersonManager()
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    dest = CharDescriptorField(Descr, 'mailart', db_column='dest', blank=True, null=True, max_length=4)
    uid = models.IntegerField(null=True, blank=True)
    pwuse = DescriptorSet(Descr, "pwdomain", db_column="pwuse")
    redirect = models.CharField(blank=True, null=True, max_length=255)
    radiustemplate = DescriptorField(Descr, "radius-templates", db_column="radiustemplate", null=True, blank=True)
    maxconn = models.IntegerField(blank=True, null=True, max_length=4)
    uremip = models.ForeignKey("Ipkunde", db_column="uremip", null=True, blank=True)
    prefcall = DescriptorField(Descr, "mailart", db_column="prefcall", null=True, blank=True)
    mperson = models.ForeignKey('self', null=True, db_column='mperson', blank=True, raw_id_admin=True)
    gebtag = models.IntegerField(null=True, blank=True)
    gebjahr = models.IntegerField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now=True)
    infotext = models.CharField(blank=True, null=True, max_length=255)
    abt = models.CharField(blank=True, null=True, max_length=255)
    email = models.CharField(blank=True, null=True, max_length=255)
    fon = models.CharField(blank=True, null=True, max_length=255)
    fax = models.CharField(blank=True, null=True, max_length=255)
    pager = models.CharField(blank=True, null=True, max_length=255)
    isdn = models.CharField(blank=True, null=True, max_length=255)
    adresse = models.ForeignKey(Adresse, blank=True, null=True, db_column="adresse", raw_id_admin=True)
    ausweisnr = models.CharField(blank=True, null=True, max_length=255)
    zusatzinfo = models.CharField(blank=True, null=True, max_length=255)
    suchbegriff = models.CharField(blank=True, null=True, unique=True, max_length=255)
    user = models.CharField(blank=True, null=True, unique=True, max_length=32)
    pass_field = models.CharField(blank=True, null=True, db_column='pass', max_length=255) # Field renamed because it was a Python reserved word.
    homedir = models.CharField(blank=True, null=True, max_length=255)
    tarifname = models.IntegerField(null=True, db_column='tarifname', blank=True)  # foreign key to Tarifname
    job = models.CharField(blank=True, null=True, max_length=255)
    name = models.CharField(blank=False, null=True, max_length=255)
    nagiosconf = models.CharField(max_length=255, null=True, blank=True)
    class Meta:
        verbose_name = 'account'
        db_table = 'person'
        ordering = ["user"]
    class Admin:
        list_display = ["user","name","pwuse"]
    class Updatelog:
        log_related = ["kunde"]

    def __unicode__(self):
        if self.name:
            return self.name
        elif self.user:
            return self.user
        else:
            return "#%d" % self.id

    def __repr__(self):
        if self.user and self.name:
            return '<Person: %s, %s>' % (smart_str(self.user), smart_str(self.name))
        else:
            return '<Person: %s>' % str(self)

    def mailrules(self):
        """Gibt ein PopQuerySet mit allen Mailrules zurück, die direkt zu dieser Mailbox gehören (ohne composite_rules)"""
        return Mailrule.objects.filter(ziel=self.user).descr_eq('typ','virt')
    def composite_rules(self):
            """Gibt die Regeln zurück, die dieses Postfach als Ziel haben, allerdings mit anderen Postfächern zusammen.
            """
            return Mailrule.objects.active().descr_eq('typ', 'virt')\
                .extra(where=["(ziel regexp %s)"], params=["^%(user)s,|,%(user)s($|,)" % {'user': re.escape(self.user)}])
    def mailbox_quellen(self):
        """Für eine Person, die eine Mailbox repraesentiert:
        Gibt eine Liste der Mailbox-Quellen als EmailAddress zurück, die auf `virt`-Regeln beruhen.
        Sucht nur ueber virt rules
        """
        from kundebunt.popkern.utils import domain_sort_key
        result = [rule.quell_adresse() for rule in self.mailrules()]
        result.sort(key=lambda addr: (domain_sort_key(addr.domain),addr.localpart))
        return result

    def get_mailbox_url(self):
        return navi_url("mailadmin.edit_mailbox", self.navi_context(), True)
    def get_absolute_url(self):
        if self.has_pwuse_flag('mail'):
            return self.get_mailbox_url()
        else:
            return navi_url('kunde.person', self.navi_context(), True)
    def is_staff(self):
        """gehöre ich zum Personal?"""
        return self.kunde_id==1 and self.has_pwuse_flag("pop")
    def is_mailbox(self):
        """Bin ich (auch) ein Postfach?"""
        return self.has_pwuse_flag("mail")
    def navi_context(self):
        return {'mailbox_name': self.user, 'kunden_name': self.kunde.name, 'person_id': self.id, 'id': self.id}
    def managed_domainkunden(self):
        return PopQuerySet.list_union(Domainkunde,
                                      [k.domainkunde_set.active() for k in Kunde.objects.managed_by(self.id)])
    def managed_kunden(self, dienst_namen=('service',)):
        return Kunde.objects.managed_by(self.id, dienst_namen=dienst_namen)
    def auth_user(self):
        """Gibt den Django-User zu dieser Person zurück.
        Falls es den nicht gibt, gibt's eine User.DoesNotExist exception.
        """
        from django.contrib.auth.models import User
        return User.objects.get(id=self.id)

    _anreden_re = re.compile(ur'^(Herr|Frau|..+\.|Professor|Doktor)$')
    @staticmethod
    def zerlege_name(name, keep_hyphen=False):
        """Gibt ein Tupel aus Listen mit Anreden, Vornamen und Nachnamen zurück: ([Anrede], [Vorname], [Nachnamen]).
        Firmennamen werden evtl. falsch erkannt (stur nach der Methode "das erste muss wohl ein Vorname sein")

        >>> Person.zerlege_name(u"Herr Prof.  Dr. Klaus-Peter von Markenschein-Fallsinger")
        ([u'Herr', u'Prof.', u'Dr.'], [u'Klaus', u'Peter'], [u'von', u'Markenschein', u'Fallsinger'])

        >>> Person.zerlege_name(u"Herr Prof.  Dr. Klaus-Peter von Markenschein-Fallsinger", True)
        ([u'Herr', u'Prof.', u'Dr.'], [u'Klaus-Peter'], [u'von', u'Markenschein-Fallsinger'])

        >>> Person.zerlege_name(u"Behrends")
        ([], [], [u'Behrends'])

        >>> Person.zerlege_name("Motoren-Maier")
        ([], [], [u'Motoren', u'Maier'])

        >>> Person.zerlege_name(u'Motoren Maier')
        ([], [u'Motoren'], [u'Maier'])

        >>> Person.zerlege_name(u"")
        ([], [], [])

        >>> Person.zerlege_name(u"   ")
        ([], [], [])

        >>> Person.zerlege_name(None)
        ([], [], [])

        >>> Person.zerlege_name(u"Meier-Kleinfeld")
        ([], [], [u'Meier', u'Kleinfeld'])

        >>> Person.zerlege_name(u"Herr")
        ([], [], [u'Herr'])

        >>> Person.zerlege_name(u"B. H. Neuner")
        ([], [u'B.', u'H.'], [u'Neuner'])
        """
        if keep_hyphen:
            def _extend_or_append(target, s):
                target.append(s)
        else:
            def _extend_or_append(target, s):
                target.extend(s.split(u"-"))
        tokens = name and name.split()
        if not tokens or tokens == [u'']:
            return ([], [], [])
        anreden, vornamen, nachnamen = [], [], []
        try:
            token = tokens.pop(0)
            while tokens and Person._anreden_re.match(token):
                _extend_or_append(anreden, token)
                token = tokens.pop(0)
            while tokens and token not in (u"von", u"zu", u"und", u"bei", u"vom"):
                _extend_or_append(vornamen, token)
                token = tokens.pop(0)
            _extend_or_append(nachnamen, token)
            for token in tokens:
                _extend_or_append(nachnamen, token)
        except IndexError:
            pass
        return (anreden, vornamen, nachnamen)

    @staticmethod
    def vergleiche_namen(suche, kandidat):
        """Vergleicht zwei Namen mit globbing und dem Versuch, klug auszusehen.

        >>> Person.vergleiche_namen("Hintertupfing", "Herr Franz Hintertupfing")
        True

        >>> Person.vergleiche_namen("Motoren Maier", "Motoren Maier")
        True

        >>> Person.vergleiche_namen("Michael", "Herr Franz Michael")
        True
        
        >>> Person.vergleiche_namen("Michael", "Herr Michael Fischer")
        False
        
        >>> Person.vergleiche_namen("Michael *", "Frau Michaela Fischer")
        False
        """
        if not suche and kandidat:
            return False
        if glob_to_regex(suche).match(kandidat):
            return True
        such_teile = Person.zerlege_name(suche)
        kandidat_teile = Person.zerlege_name(kandidat)
        for such_tokens, kand_tokens in zip(Person.zerlege_name(suche), Person.zerlege_name(kandidat)):
            for token in such_tokens:
                token_re = glob_to_regex(token)
                if not any(token_re.match(kand_token) for kand_token in kand_tokens):
                    return False
        return True

    @staticmethod
    def name_like(like_string):
        """Verwandelt einen String zur Namens-Suche mit like um in einen String,
        der die Aufteilung von Namen in Betracht zieht. Das Ergebnis muss dann
        aber noch mit vergleiche_namen nachgefiltert werden.
        """
        zerlegung = Person.zerlege_name(like_string)
        return u"%" + u"%".join(chain(* zerlegung))

    #def default_filter(self, suche):
        #return glob_to_regex(suche).match(self.user) or Person.vergleiche_namen(suche, self.name)

class Kundemail(UpdatelogMixin, models.Model):
    """
    Verwendung: Welche Person gehört bei welchem Kunden zu was?

    Index:  kunde dienst person
    Index:  timestamp, person
    Aufbau:
    kunde   int4    !>kunde ID des Kunden
    dienst  int2    !^!dienst   Dienst
    dringend    int2    - Ordnungskriterium
    person  int4    !^!>person  Person
    timestamp   datetime    !/+  Zeitstempel
    """
    kunde = models.ForeignKey(Kunde, db_column='kunde', primary_key = True)
    dienst = DescriptorField(Descr, "dienst", db_column="dienst", primary_key = True)
    person = models.ForeignKey(Person, db_column='person', primary_key = True)
    dringend = models.SmallIntegerField()
    timestamp = models.DateTimeField(auto_now=True, db_index=True)
    class Meta:
        db_table = 'kundemail'
        has_composite_primary_key = True
    class Updatelog:
        log_index = "kunde"
        log_related = []
    def __unicode__(self):
        return u"Kunde:%s - Person:%s" % (self.kunde, self.person)

class Updatelog(models.Model):
    """
    Verwendung: Speichert Änderungen

    Index:  id
    Index:  timestamp, db_tabelle wert
    Aufbau:
    id  int4    !?+ ID
    person  int4    !^!>person  ändernder Mensch, <- person.id
    db_tabelle  int4    !^!>db_tabelle  Geänderte Tabelle
    indexspalten    int4    !^!>updatelog_spalten   Liste der Datenfelder
    wert    char100 -   Indexwerte, getrennt mit '|'. Sollte "indexinhalt" heißen.
    datenspalten    int4    !^!>updatelog_spalten   Liste der Datenfelder
    dwert   text2   !-  alte Datenwerte, getrennt mit '|'. Sollte "dateninhalt" heißen.
    timestamp   datetime    !/+  Zeitstempel
    """
    person = models.ForeignKey(Person, db_column='person', raw_id_admin=True)
    db_tabelle = models.ForeignKey(DbTabelle, db_column='db_tabelle', raw_id_admin=True)
    indexspalten = models.ForeignKey(UpdatelogSpalten, related_name='updatelog_by_indexspalten', db_column='indexspalten', raw_id_admin=True)
    wert = models.CharField(max_length=100)
    datenspalten = models.ForeignKey(UpdatelogSpalten, related_name='updatelog_by_datenspalten', db_column='datenspalten', raw_id_admin=True)
    dwert = models.TextField(null=True)
    timestamp = models.DateTimeField(auto_now = True)
    class Meta:
        db_table = 'updatelog'
        ordering = ['id']
    def __unicode__(self):
        return u"#%d, %s: Tabelle %s(%s=%s): %s:%s" % (self.id, self.person.user, self.db_tabelle, self.indexspalten, self.wert, self.datenspalten, self.dwert)

class Domainkunde(UpdatelogMixin, models.Model):
    """
    Verwendung: Map Domainadresse zu Kunde

    Index:  id, domain ende
    Index:  kunde, ktarif, nserver, ticket, timestamp, person, status beginn
    Aufbau:
    id  int4    !?+ ID des Eintrags
    timestamp   datetime    !/+  Zeitstempel
    kunde   int4    !>kunde Kunde
    domain  char100 !^!%ascii  vollständiger Domainname
    beginn  uint4   !/  Zuteilung
    ende    uint4   !-!/    Freigabe
    expires uint4   !-!/    Ablauf laut Registrar
            --
    status  int2    !domainstatus   Status
    tarif   int2    !##!-!>dienst/domain   TODO: Raus damit => Tarif
    infotext    char255 !-  Kurzbeschreibung wieso (IP-Nr etc.)
    tarifname   int4    !-!>tarifname   Name des (künftigen...) Tarifs
    rechinfo    int4    !-  externe Rechnung -> erech.id
    ktarif  int4    !-!>tarifkunde
    person  int4    !-!>person  Besitzer des Webspaces o.ä.
    owner   int4    !-!>person  Firma des Domainantrags
    adminc  int4    !-!>person  Admin-C des Domainantrags
    techc   int4    !-!>person  Tech-C des Domainantrags
    zonec   int4    !-!>person  Zone-C des Domainantrags
    billc   int4    !-!>person  Bill-C des Domainantrags
    nic int2    !-!nic  Registrar für die Domain
    nserver int4    !-!>ipkunde Name+Adresse eines kundeneigenen Primary
    tsig    int2    !-!tsig TSIG-Key für allow_update
    ticket  int4    !-!>ticket  Kundenauftrag für die Domain
    nachricht   int4    !-!>person  Benachrichtigung bei Updates etc.
    flags   int4    !*domainflags   Flags, z. B. "dnszone"
    mail_ip uint1   !-  Wert des letzten Bytes der IP-Adressen, die auf mail.noris.net für E-Mails mit dieser Absenderdomain verwendet werden sollen
    """
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    domain = models.CharField(max_length=100)
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(null=True, blank=True)
    #dringend = models.IntegerField(blank=True, null=True)
    status = DescriptorField(Descr, "domainstatus", db_column="status")
    #tarif = models.ForeignKey("Dienst", blank=True, null=True, db_column="tarif", related_name="domainkunden_by_tarif")
    rechinfo = models.IntegerField(blank=True, null=True)
    ktarif = models.IntegerField(db_column='ktarif', blank=True, null=True)  # foreign key to Tarifkunde
    person = models.ForeignKey(Person, related_name='domainkunde_by_person', db_column='person', blank=True, null=True, raw_id_admin=True)
    owner = models.ForeignKey(Person, related_name='domainkunde_by_owner', db_column='owner', blank=True, null=True, raw_id_admin=True)
    adminc = models.ForeignKey(Person, related_name='domainkunde_by_adminc', db_column='adminc', blank=True, null=True, raw_id_admin=True)
    techc = models.ForeignKey(Person, related_name='domainkunde_by_techc', db_column='techc', blank=True, null=True, raw_id_admin=True)
    zonec = models.ForeignKey(Person, related_name='domainkunde_by_zonec', db_column='zonec', blank=True, null=True, raw_id_admin=True)
    nic = DescriptorField(Descr, "nic", blank=True, null=True, db_column="nic")
    nserver = models.ForeignKey("Ipkunde", db_column='nserver', blank=True, null=True)
    tsig = DescriptorField(Descr, "tsig", blank=True, null=True, db_column="tsig")
    ticket = models.IntegerField(null=True, blank=True)
    nachricht = models.ForeignKey(Person, related_name='domainkunde_by_nachricht', db_column='nachricht', blank=True, null=True, raw_id_admin=True)
    timestamp = models.DateTimeField(auto_now=True)
    billc = models.ForeignKey(Person, related_name='domainkunde_by_billc', db_column='billc', blank=True, null=True, raw_id_admin=True)
    expires = models.UnixDateTimeField(blank=True, null=True)
    flags = DescriptorSet(Descr, "domainflags", db_column="flags")
    infotext = models.CharField(max_length=255, blank=True, null=True)
    tarifname = models.IntegerField(db_column='tarifname', blank=True, null=True)  # foreign key to Tarifname
    mail_ip = models.PositiveIntegerField(null=True, blank=True)
    auftragsnr = models.PositiveIntegerField(null=True, blank=True)

    class Meta:
        verbose_name = 'Domain'
        db_table = 'domainkunde'
        ordering = ["domain"]
        unique_together = [('domain', 'ende')]
    class Admin:
        list_display = ["domain"]
    class Updatelog:
        log_related = ["kunde"]
        log_related_as = "domain"
    def __unicode__(self):
        return self.domain
    def beginn_display(self):
        return time_to_str(self.beginn, "%Y-%m-%d")


###########################################################
# 2. Stock: Abteilung Hardware, Strümpfe, Kinderkleidung.

class Hardware(UpdatelogMixin, models.Model):
    """
    Verwendung: Hardware (Server, Router etc.)
    
    Index:  id, hardware_id ende, name seriennr ende, ip ende
    Index:  eigentuemer gemailt, timestamp
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    name    char127 -   Name
    hardware_id char15  !-  eindeutige, durch noris vergebene ID; NULL für Stromleisten etc., vgl. RT#201214
    seriennr    char127 !-  SerienNummer
    ivnr    char15  !-  InventarNummer (wenn das Ding uns gehört)
    typ int2    !-!hardware_typ Typ der Hardware, z. B. "Switch", "Server" etc., vgl. RT#207798
    klasse  int2    !-!hardware_klasse  Hardwareklasse, z. B. "intern", "office" etc., vgl. RT#207798
    ip  int4    !-!>ipkunde Management-IP-Adresse
    enthalten_in    int4    !-!>hardware    Referenz auf andere Hardware, in der diese Hardware eingebaut ist, vgl. RT#207798-17
    standort    int4    !-!>person  Standort
    eigentuemer int4    !-!>person  Eigentümer
    verantwortlich  int4    !-!>person  Verantwortlicher für diese Hardware, vgl. RT#314011
    kunde   int4    !^!>kunde   zugeordneter Kunde
    ktarif  int4    !-!>tarifkunde  Tarif des Kunden, der zu dieser Hardware gehört (vgl. Ticket 10096055)
    lieferant   int2    !-!lieferant    Lieferant, bei dem wir diese Hardware gekauft haben
    status  int2    !-!hardware_status  Status der Hardware, vgl. RT#215033
    info    char255 !-  optionaler Info-Text
    beginn  uint4   !/  Gültigkeit
    ende    uint4   !-!/    Gültigkeit
    rack    int2    !^^!-!>rack Rack, in dem diese Hardware eingebaut ist
    unterste_he uint1   !-  unterste HE des Racks, in dem diese Hardware eingebaut ist
    he  uint1   !-  Höhe des Geräts in HE
    console char255 !-  (symbolischer) Port und Server für Console, vgl. RT#206285
    gemailt int4    !-  Timestamp, der beim Vermailen des Datensatz gesetzt wird, vgl. RT#219877
    flags   int4    !*hardwareflags Flags
    """
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    ktarif = models.ForeignKey(to='Tarifkunde', null=True, db_column='ktarif', blank=True)
    timestamp = models.DateTimeField(auto_now=True)
    name = models.CharField(unique=True, max_length=127)
    hardware_id = models.CharField(null=True, blank=True, unique=True, max_length=15)
    ivnr = models.CharField(null=True, blank=True, max_length=15)
    typ = DescriptorField(Descr, 'hardware_typ', null=True, blank=True, db_column="typ", choices_sort_key=lambda(descr,bla):bla, db_index=True)
    klasse = DescriptorField(Descr, 'hardware_klasse', null=True, blank=True, db_column="klasse")
    ip = models.ForeignKey('Ipkunde', null=True, db_column='ip', blank=True, raw_id_admin=True)
    enthalten_in = models.ForeignKey('self', null=True, db_column='enthalten_in', blank=True, raw_id_admin=True, related_name='enthaltene_hardware')
    seriennr = models.CharField(blank=True, null=True, max_length=127)
    standort = models.ForeignKey(Person, null=True, db_column='standort', blank=True, raw_id_admin=True, related_name='hardware_by_standort')
    lieferant = DescriptorField(Descr, 'lieferant', null=True, blank=True, db_column="lieferant", choices_sort_key=lambda(descr,bla):bla)
    status = DescriptorField(Descr, 'hardware_status', null=True, blank=True, db_column="status")
    eigentuemer = models.ForeignKey(Person, null=True, db_column='eigentuemer', blank=True, raw_id_admin=True, related_name='owned_hardware')
    verantwortlich = models.ForeignKey(Person, null=True, db_column='verantwortlich', blank=True, raw_id_admin=True, related_name='responsible_for_hardware')
    info = models.CharField(blank=True, null=True, max_length=255)
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(unique=True, null=True, blank=True)
    console = models.CharField(blank=True, null=True, max_length=255)
    rack = models.ForeignKey('Rack', null=True, db_column='rack', blank=True, raw_id_admin=True)
    unterste_he = models.PositiveSmallIntegerField(null=True, blank=True)
    he = models.PositiveSmallIntegerField(null=True, blank=True)
    gemailt = models.IntegerField(null=True, blank=True)
    flags = DescriptorSet(Descr, 'hardwareflags')
    class Meta:
        db_table = 'hardware'
        unique_together = (("hardware_id", "ende"), ("name", "seriennr", "ende"), ("ip", "ende"))
        ordering = ["hardware_id", "name", "id"]
    class Updatelog:
        log_related = ["kunde"]

    def navi_context(self):
        return {'id': self.id, 'hardware_db_id': self.id}

    def get_absolute_url(self):
        return navi_url("hardware.hw_edit_panel", self.navi_context(), True)


    def standort_display(self, level=0):
        """Beschreibung des Standorts mit Rack/RZ oder Person."""
        #raise Bla
        ort = []
        try:
            if self.standort_id:
                if self.standort.user:
                    ort.append(self.standort.user)
                else:
                    ort.append("#%d (%s)" % (self.standort_id, self.standort.name))
        except Person.DoesNotExist:
            pass  # this is a mysql artifact. ForeignKey constraints are not always followed. Hey, it's not my fault.
        if self.rack_id:
            try:
                if self.unterste_he != None and self.he != None:
                    ort.append("%s, %s, HE %d+%d" % (self.rack.rz.name, self.rack.name, self.unterste_he, self.he))
                else:
                    ort.append("%s, %s" % (self.rack.rz.name, self.rack.name))
            except Rack.DoesNotExist:
                pass  # this is a mysql artifact. ForeignKey constraints are not always followed.
        if self.enthalten_in_id and level < 5:
            try:
                hw = self.enthalten_in
                if hw.hardware_id:
                    ort.append("in: %s (%s)" % (hw.hardware_id, hw.standort_display(level+1)))
                else:
                    ort.append("in: #%d (%s)" % (hw.id, hw.standort_display(level+1)))
            except Hardware.DoesNotExist:
                pass # this is an mysql artifact. ForeignKey constraints are not always followed.
            except RuntimeError:
                print "ID!!!", self.id
                bla=self.id
                raise Bla
        return ", ".join(ort)

    def hat_standort(self):
        """Gibt zurück, ob diese Hardware irgendeine Art von Standort hat (als Boolean)"""
        return self.standort_id or self.rack_id or self.enthalten_in_id

    def wartungsvertraege_fuer_anzeige(self):
        """Gibt eine Liste Wartungsverträge zurück. Wenn es mehr als 10
        Wartungsverträge gibt, werden von den abgelaufenen Verträgen nur
        so viele angezeigt, dass es ingesamt nur 10 sind, mindestens aber
        2.
        """
        grenze = 10
        wvs = list(self.wartungsvertraghardware_set.all().order_by("-ende", "beginn", "id"))
        # Suche nach dem letzten Wartungsvertrag mit ende NULL
        # (in der Praxis haben kaum welche ende NULL)
        for i, wv in enumerate(reversed(wvs)):
            if wv.ende:
                split_index = len(wvs) - i
                wvs = wvs[split_index:] + wvs[:split_index]
                break
        for i, wv in enumerate(wvs[grenze-2:]):
            if not is_active(None, wv.ende):
                return wvs[:i+grenze]
        return wvs

    def __unicode__(self):
        result = [u"%s" % self.id]
        if self.name:
            result.append(self.name)
        ipkunde = self.ip
        if ipkunde:
            result.append(u"(%s)" % ipkunde.ipaddr_display())
        if self.hardware_id:
            result.append(u"[#%s]" % self.hardware_id)
        standort = self.standort_display()
        if standort:
            result.append(u"@ %s" % standort)
        return u" ".join(result)

class IpRegion(UpdatelogMixin, models.Model):
    """
    Verwendung:     Speichert Namen für IP-Autoallokation und Blockgröße+Bereich für
            neue Allokationen

    Index:  id, zone
    Index:  ipkunde, timestamp
    Aufbau:
    id      int4    !?+     eindeutige ID
    zone    char50  !^      Name der Region (eindeutig)
    infotext        char255 !-!^    zusätzlicher Infotext
    kunde   int4    !>kunde dem die Region primär gehört
    alloc   int1    -       Größe des nächsten zu allozierenden Blocks, wenn eine
                            Allokation nicht möglich ist
    ipkunde int4    !-!>ipkunde     Bereich, aus dem der Block geholt werden soll
    timestamp       datetime    !/+      Zeitstempel
    """
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    alloc = models.IntegerField()
    ipkunde = models.ForeignKey('Ipkunde', null=True, db_column='ipkunde', blank=True, raw_id_admin=True, related_name='ipregions_by_allocation')
    timestamp = models.DateTimeField(auto_now=True)
    zone = models.CharField(unique=True, max_length=50)
    infotext = models.CharField(blank=True, null=True, max_length=255)
    class Meta:
        db_table = 'ipregion'
    class Updatelog:
        log_related = []


# This test is incomplete.
ip6_re = re.compile(ur'^(::)?([0-9a-f]{1,4}::?){0,7}([0-9a-f]{1,4})?$', re.IGNORECASE)

class IpkundeManager(PopManager):

    def filter_name_or_ip(self, s):
        """Filtert nach name oder ip6 , die entweder nach der numerischen IP
        oder dem eingetragenen `name` sucht. Die Funktion validiert s dabei.
        """
        try:
            ip6 = Ipkunde.db_value_for_ip6(s)
            return self.get_query_set().filter(ip6__exact=ip6)
        except ValueError:
            return self.get_query_set().filter(name__iexact=s)

    def default_query(self, value, model = None, join_condition = None, active_only=True, table_suffix=''):
        if model:
            assert(join_condition)
            join_condition = [join_condition]
            join_tables = ["ipkunde ipkunde%s" % table_suffix]
            manager = model.objects
        else:
            manager = self
            join_condition = []
            join_tables = []
        # dirty HACK ...
        _, q_ip_where, q_ip_params = Ipkunde.q_ip_glob(value).get_sql(Ipkunde._meta)
        if q_ip_where:
            q_ip_where = q_ip_where[0].replace('`ipkunde`.', '`ipkunde%s`.' % table_suffix)
        if active_only:
            ips = manager.active()
        else:
            ips = manager
        return ips.extra(
            where=['(ipkunde' + table_suffix + '.name like %s or ' + q_ip_where + ')'] + join_condition,
            tables = join_tables,
            params = [glob_to_like(value)] + q_ip_params
            )

ipv4_prefix = u"0" * 24

class Ipkunde(UpdatelogMixin, models.Model):
    """
    Verwendung:     Map IP-Adresse zu Kunde

    Index:  id, ip6 bits ende
    Index:  kunde, name, mac ende, ticket, timestamp
    Aufbau:
    id      int4    !?+     ID des Eintrags
    timestamp       datetime    !/+      Zeitstempel
    kunde   int4    !>kunde Kunde
    ip6 char32      !-              IPv6-Adresse in nicht abgekürzter Hexadezimalschreibweise (IPv4-Adressen mit 24 führenden Nullen) (nur vorübergehend NULLbar, s. RT#369541)
    bits    int1    !(0-128)        zu ignorierende Bits am Ende der Adresse (32 - 1bitsInNetmask)
    tarif   int2    !>dienst        IP-Dienst, dem die IP-Daten dieser Adresse zugeordnet werden
    beginn  uint4   !/      Zeitpunkt der Zuteilung der Adresse
    ende    uint4   !-!/    Freigabezeitpunkt der Adresse
    kiste   char16  !-      Router, der die Adresse announcet
    mac     fchar12 !-      MAC-Adresse der Kiste
    name    char255 !-      primärer Domainname; bei Netzen: Kurzbeschreibung
    infotext        char255 !-      Zusatzinfo
    alloc   char    !-!alloc.n      Auto-Allokation? "+" oder "-"
    ipregion        int4    !-!>ipregion    die aus diesem Bereich Adressen vergeben kann
    ticket  int4    !-!>ticket      Kundenauftrag fr das inetnum-Objekt
    owner   int4    !-!>person      Firma des Adressblocks
    adminc  int4    !-!>person      Admin-C des Adressblocks
    techc   int4    !-!>person      Tech-C des Adressblocks
    nic     int2    !-!nic  Registrar fr den Adressblock
    status  int1    !-!ipstatus     Status des Antrags
    dest    int2    !ziel   Zielcode für IP-Traffic an diese Adresse
    nagiosconf      char255 !-      spezielle Konfigurationseinstellungen fr Nagios (vgl. RT#259990)
    flags   int4    !*ipflags       Flags, z. B. "no_virusscan"
    hostgroups  int8    !#!*hostgroups    Nagios-Hostgroups, zu denen dieser Host gehören soll (vgl. RT#299012)
    snmp_community  char255 !-  SNMP-Community für Abfragen z. B. durch CapMan (vgl. RT#349627)
    """
    objects = IpkundeManager()

    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    ip6 = models.CharField(max_length=32)
    bits = models.SmallIntegerField()
    tarif = models.ForeignKey(Dienst, db_column='tarif', raw_id_admin=True)
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(unique=True, null=True, blank=True)
    kiste = models.CharField(blank=True, null=True, max_length=16)
    alloc = CharDescriptorField(Descr, "alloc", default="n", db_column="alloc", blank=True, null=True, max_length=1)
    mac = models.CharField(blank=True, null=True, max_length=12)
    ipregion = models.ForeignKey('Ipregion', null=True, db_column='ipregion', blank=True, raw_id_admin=True, related_name='ipkunden_by_allocation')
    ticket = models.IntegerField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now=True)
    owner = models.ForeignKey(Person, null=True, db_column='owner', blank=True, raw_id_admin=True, related_name='ipkunden_by_owner')
    adminc = models.ForeignKey(Person, null=True, db_column='adminc', blank=True, raw_id_admin=True, related_name='ipkunden_by_adminc')
    techc = models.ForeignKey(Person, null=True, db_column='techc', blank=True, raw_id_admin=True, related_name='ipkunden_by_techc')
    status = DescriptorField(Descr, 'ipstatus', null=True, blank=True, db_column="status")
    nic = DescriptorField(Descr, 'nic', null=True, blank=True, db_column="nic")
    dest = DescriptorField(descr_name='ziel', descr_model=Descr, db_column='dest')
    nagiosconf = models.CharField(blank=True, null=True, max_length=255)
    flags = DescriptorSet(Descr, 'ipflags')
    infotext = models.CharField(blank=True, null=True, max_length=255)
    name = models.CharField(blank=True, null=True, max_length=255)
    snmp_community = models.CharField(blank=True, null=True, max_length=255)
    mail_ip = models.PositiveIntegerField(null=True, blank=True)
    vrf = models.CharField(max_length=255, null=True, blank=True)

    class Meta:
        db_table = 'ipkunde'
        verbose_name = 'IP address'
        unique_together = (("ip6", "bits", "ende"),)
    class Updatelog:
        log_related = ["kunde"]

    def __contains__(self, other):
        """Liegt die andere ip `other` innerhalb des Subnetzes von `self`?
        """
        if self.bits < other.bits:
            return False
        else:
            netmask = self.netmask()
            if self.ip6 and other.ip6:
                return (int(self.ip6, 16) & netmask) == (int(other.ip6, 16) & netmask)
            else:
                return False

    def __unicode__(self):
        return self.ipaddr_num_display()

    def is_ipv4(self):
        return self.ip6 and len(self.ip6)==32 and self.ip6.startswith(ipv4_prefix)

    def ipaddr_display(self):
        """Anzeige der IP-Adresse als Name oder im üblichen dotted-quad-Format (1.2.3.4),
        analog für ip6"""
        if self.name:
            return self.name
        elif self.ip6:
            return self.ipaddr_num_display()
        else:
            return u"#%s" % self.id

    def ipaddr_num_display(self):
        """Anzeige der numerischen IP-Adresse"""
        if self.ip6:
            try:
                ip6 = self.ip6.lower()
                if len(ip6)==32 and ip6.startswith(ipv4_prefix):
                    return u".".join(unicode(int(ip6[i:i+2],16)) for i in (24,26,28,30))
                elif len(ip6)==32:
                    packed = "".join(chr(int(ip6[i:i+2], 16)) for i in range(0,32,2))
                    return inet_ntop(AF_INET6, packed)
            except ValueError:
                pass
            return ip6
        else:
            return u""

    def netbits(self):
        """Anzahl 1-Bits der Subnetzmaske"""
        if self.is_ipv4():
            return 32 - self.bits
        else:
            return 128 - self.bits

    def netmask(self):
        """Die übliche Subnetzmaske, als int"""
        if self.is_ipv4():
            all_bits_p1 = 1<<32
        else:
            all_bits_p1 = 1<<128
        return all_bits_p1 - (1 << self.bits)

    @staticmethod
    def aton(s):
        """Wandelt die dotted-quad-IP in die Datenbankrepräsentation um. Nur ipv4."""
        if s:
            return htonl(unpack("i", inet_aton(s))[0])
        else:
            return None

    _ip4_address_re = re.compile(ur"^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$")
    
    @staticmethod
    def db_value_for_ip6(s):
        """Wandelt die übliche ip4- oder ip6-Eingabe (mit . oder : und ::)
        in das Format für die Datenbank um.
        Wirft ValueError, wenn der String nicht passt.

        >>> Ipkunde.db_value_for_ip6(u"1::13:ffff")
        u'0001000000000000000000000013ffff'
        >>> Ipkunde.db_value_for_ip6(None)
        >>> Ipkunde.db_value_for_ip6(u"::1")
        u'00000000000000000000000000000001'
        >>> Ipkunde.db_value_for_ip6(u"1.2.3.128")
        u'00000000000000000000000001020380'
        >>> Ipkunde.db_value_for_ip6(u"bäh")    #doctest: +ELLIPSIS
        Traceback (most recent call last):
            ...
        ValueError: ...
        """
        if not s:
            return None
        try:
            match = Ipkunde._ip4_address_re.match(s)
            if match:
                packed = inet_pton(AF_INET, s)
                return ipv4_prefix + u"".join(u"%02x" % ord(c) for c in packed)
            if s:
                packed = inet_pton(AF_INET6, s)
                return u"".join(u"%02x" % ord(c) for c in packed)
            else:
                return None
        except (socket_error, UnicodeEncodeError), err:
            raise ValueError(str(err))

    _ip4_wildcard_re = re.compile(ur'(?P<addr>(\d{0,3}\.){0,3})\*\s*$')
    _ip4_subnet_re = re.compile(ur'(?P<addr>(\d{0,3}\.){0,3}\d{0,3})(/(?P<bits>\d{0,2}))?\s*$')
    _ip6_subnet_re = re.compile(ur'([:\da-fA-F]+)/(\d{0,3})')
    _ip6_subnet_any = re.compile(ur'[:\da-fA-F*/]+')

    @staticmethod
    def q_ip_glob(s):
        """Gibt eine Query Condition (Q) zurück, die nach name oder ip6 filtert,
        mit "*" als Jokerzeichen.
        Für ip4-Adressen sind folgende Varianten zulässig:
            - eine teilweise Adresse, z.B. "1.2"
            - "*" als Jokerzeichen, z.B. "1.2.*"
            - Angabe mit Subnetz-Länge, z.B. "1.2.0.0/16" oder "1.2/16"
        
        Für ipv6 sind zulässig:
            - eine teilweise Adresse, z.B. "1234:ab"
            - "*" als  Jokerzeichen, z.B. "1234:ab*" oder "1234:ab:*" (mit anderer Bedeutung!)
            - Angabe mit Subnetz-Länge, z.B. "1234:ab/32" oder "1234:ab::0/32"
        
        Die Funktion gibt ein Q ohne Treffer zurück, wenn s syntaktisch nicht passt.

        >>> Ipkunde.q_ip_glob(u"1.255").kwargs
        {'ip6__range': (u'00000000000000000000000001ff0000', u'00000000000000000000000001ffffff')}
        >>> Ipkunde.q_ip_glob(u"1.2.3.*").kwargs
        {'ip6__range': (u'00000000000000000000000001020300', u'000000000000000000000000010203ff')}
        >>> Ipkunde.q_ip_glob(u"1.2.3.4/20").kwargs
        {'ip6__range': (u'00000000000000000000000001020000', u'00000000000000000000000001020fff')}
        >>> Ipkunde.q_ip_glob(u"200.201.1.0").kwargs
        {'ip6__range': (u'000000000000000000000000c8c90100', u'000000000000000000000000c8c90100')}
        >>> Ipkunde.q_ip_glob(u"1234:5:af").kwargs
        {'ip6__range': (u'1234000500af00000000000000000000', u'1234000500afffffffffffffffffffff')}
        >>> Ipkunde.q_ip_glob(u"1234:5:af*").kwargs
        {'ip6__range': (u'12340005af0000000000000000000000', u'12340005afffffffffffffffffffffff')}
        >>> Ipkunde.q_ip_glob(u"b9:6c::123:9800/113").kwargs
        {'ip6__range': (u'00b9006c000000000000000001238000', u'00b9006c00000000000000000123ffff')}
        >>> Ipkunde.q_ip_glob(u"b96c::/113").kwargs
        {'ip6__range': (u'b96c0000000000000000000000000000', u'b96c0000000000000000000000007fff')}
        >>> Ipkunde.q_ip_glob(u"b9::").kwargs
        {'ip6__range': (u'00b90000000000000000000000000000', u'00b90000000000000000000000000000')}
        >>> Ipkunde.q_ip_glob(u"2001:").kwargs
        {'ip6__range': (u'20010000000000000000000000000000', u'2001ffffffffffffffffffffffffffff')}
        >>> Ipkunde.q_ip_glob(u"a:*karlsruhe*").kwargs
        {'pk__exact': None}
        """
        all_ones = (1 << 128) - 1

        # ipv4 - Umsetzen in ipv6-Subnetz-Spezifikation
        quads = bits = addr = None
        match = Ipkunde._ip4_wildcard_re.match(s)
        if match:
            quads = match.group("addr")[:-1].split(".")
        else:
            match = Ipkunde._ip4_subnet_re.match(s)
            if match:
                quads = match.group("addr").split(".")
                try:
                    bits = match.group("bits")
                    if bits:
                        bits = int(bits)
                except ValueError:
                    return Q()
        if quads:
            if quads[-1] == '':
                quads.pop()
            addr = ipv4_prefix + u"".join(u"%02x" % int(q) for q in quads) + u"00" * (4 - len(quads))
            if bits:
                bits += 4 * len(ipv4_prefix)
            else:
                bits = 8 * len(quads) + 4 * len(ipv4_prefix)
            # ipv4 fertig, in addr/bits ist das Subnetz a la ipv6 beschrieben
        elif u":" in s and Ipkunde._ip6_subnet_any.match(s):
            if u"::" in s:
                left, right = s.split(u"::", 1)
                anz_gruppen = left.count(u":") + right.count(u":") + 2
                if not left:
                    left = u"0"
                if not right:
                    right = u"0"
                s = u"%s:%s%s" % (left, u"0:" * (8-anz_gruppen), right)
            match = Ipkunde._ip6_subnet_re.match(s)
            if match:
                s, bits = match.groups()
                bits = int(bits)
            parts = s.split(u":")
            if parts[-1] == u'':
                parts.pop()
            addr = u"".join(u"0" * (4-len(p)) + p for p in parts[:-1])
            if parts[-1].endswith(u"*"):
                addr += parts[-1][:-1]
            else:
                addr += u"0" * (4-len(parts[-1])) + parts[-1]
            if bits is None:
                bits = len(addr)*4
            addr += u"0" * (32-len(addr))
            # ipv6 fertig, in addr/bits ist das Subnetz beschrieben
        if addr:
            try:
                ip1 = int(addr,16) & ~ (all_ones >> bits)
                ip2 = ip1 | (all_ones >> bits)
                return Q(ip6__range=(u"%032x" % ip1, u"%032x" % ip2))
            except ValueError:
                pass
        return Q(pk__exact=None)  # garantiert keine Treffer.



class Rack(UpdatelogMixin, models.Model):
    """
    Verwendung:     Racks für Housing-Datenbank

    Index:  id, rz name
    Index:  timestamp
    Aufbau:
    id      int2    !?+     ID
    timestamp       datetime    !/+      Zeitstempel
    rz      int2    !^!>rz  RZ, in dem das Rack steht
    name    char255 !^      Bezeichnung des Racks innerhalb des RZs
    info    char255 !-      optionale zusätzliche Kurzbeschreibung
    he      uint1   -       Anzahl der HöhenEinheiten des Racks
    kunde   int4    !>kunde Kunde, dem das Rack vermietet wurde
    x       int1    -       X-Koordinate des Racks
    y       int1    -       Y-Koordinate des Racks
    """
    timestamp = models.DateTimeField(auto_now=True)
    rz = models.ForeignKey("RZ", raw_id_admin=True, db_column="rz")
    name = models.CharField(max_length=255)
    info = models.CharField(blank=True, null=True, max_length=255)
    he = models.PositiveSmallIntegerField()
    kunde = models.ForeignKey(Kunde, db_column='kunde', raw_id_admin=True)
    x = models.SmallIntegerField()
    y = models.SmallIntegerField()
    class Meta:
        db_table = 'rack'
        unique_together = (("rz", "name"),)
        ordering = ['name']
    class Updatelog:
        log_related = ["kunde"]
    def __unicode__(self):
        return u"%s, %s" % (self.rz.name, self.name)

class RZ(UpdatelogMixin, models.Model):
    """
    Verwendung:     RZs und andere für die Housing-Datenbank relevante Standorte

    Index:  id, name
    Index:  timestamp
    Aufbau:
    id      int2    !?+     ID
    timestamp       datetime    !/+      Zeitstempel
    name    char255 !^      Name des RZs
    standort        int4    !>person        Standort dieses RZs, vgl. RT#207798-30
    """
    timestamp = models.DateTimeField(auto_now=True)
    name = models.CharField(unique=True, max_length=255)
    standort = models.ForeignKey(Person, db_column='standort', raw_id_admin=True)
    class Meta:
        db_table = 'rz'
    class Updatelog:
        log_related = []


class Wartungsvertrag(UpdatelogMixin, models.Model):
    """
    Verwendung: Wartungsvertragstypen, die wir mit Zulieferern haben (ITIL-Jargon: "UPC")

    Index:  id, name
    Index:  timestamp
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    name    char255 -   Name
    sla char255 -   z. B. "24/7" etc.
    ansprechpartner int4    !>person    Ansprechpartner für diesen Wartungsvertrag
    beschreibung    text2   -   weitere Details
    """
    timestamp = models.DateTimeField(auto_now=True)
    name = models.CharField(max_length=255)
    sla = models.CharField(max_length=255)
    ansprechpartner = models.ForeignKey(to='Person', db_column='ansprechpartner')
    beschreibung = models.TextField()

    def __repr__(self):
        return '<Wartungsvertrag %s>' % self.name

    class Meta:
        db_table = 'wartungsvertrag'
    class Updatelog:
        log_related = []


class WartungsvertragHardware(UpdatelogMixin, models.Model):
    """
    Verwendung: Zuordnung von Wartungsverträgen zu Hardware

    Index:  id
    Index:  timestamp, hardware beginn
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    hardware    int4    !^!>hardware    zugeordnete Hardware
    wartungsvertrag int4    !^^!>wartungsvertrag    zugeordneter Wartungsvertrag
    beginn  uint4   !/  Beginn der Laufzeit des Wartungsvertrags
    ende    uint4   !/!-    Ende der Laufzeit des Wartungsvertrags
    """
    timestamp = models.DateTimeField(auto_now=True)
    hardware = models.ForeignKey(to='Hardware', db_column='hardware')
    wartungsvertrag = models.ForeignKey(to='Wartungsvertrag', db_column='wartungsvertrag')
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(null=True, blank=True)

    class Meta:
        db_table = 'wartungsvertrag_hardware'
    class Updatelog:
        log_related = ["wartungsvertrag"]

    def __repr__(self):
        return '<WartungsvertragHardware #%s, %s bis %s, zu %s>' % (self.id, self.beginn_display(), self.ende_display(), self.hardware)

    def is_active(self):
        return is_active(self.beginn, self.ende)

    def navi_context(self):
        return { 'wv_id': self.id, 'id': self.hardware_id }

    def beginn_display(self):
        return time_to_str(self.beginn, '%Y-%m-%d')

    def ende_display(self):
        return time_to_str(self.ende, '%Y-%m-%d')

    def get_absolute_url(self):
        return navi_url("hardware.hw_wartungsvertrag_edit", self.navi_context(), True)

###########################################################
# Tickets

class Ticket(models.Model):
    """
    Verwendung: RT-Tickets
    
    Index:  id
    Index:  ticket, kunde queue, domain, hperson, ipnr, timestamp, status bearbeiter, bearbeiter queue, queue subject
    Aufbau:
    id  int4    !?ticket    eindeutige Nummer
    timestamp   datetime    !/+ Zeitstempel
    ticket  int4    !-!>ticket  Bezug (Tickets mit id!=ticket werden nur zur Assoziation verwendet; NULL:transienter Zustand)
    kunde   int4    !^__!>kunde Kunde, der das Problem hat
    subject char255 -   Genauere Problemkurzbeschreibung
    infotext    char255 !-  Aktueller Zustand
    queue   int4    !>queue für RT, Queue
    queue_area  int4    !-!>queue_areas für RT, Area
    beginn  uint4   !/  Problem bemerkt (Unix-Zeit), Ticket generiert (rt: date_created)
    wichtig int2    !(0-maxwichtig) Dringlichkeit
    maxwichtig  int2    !(wichtig-99)   max. Dringlichkeit
    status  int2    !tickets.0  
    termin  uint4   !-!/    Wiedervorlage (RT: date_due)
    endtermin   uint4   !-!/    bis wann das Teil bearbeitet sein muß
    bearbeiter  int4    !-!>person  der Mensch, dem das Teil gehört
    eskaliere   uint4   !-!/    wann die Dringlichkeitsstufe angehoben wird
    eskaliert   uint4   !-!/    wann die Automatik das letzte Mal zugeschlagen hat
    d_told  uint4   !/  date_told von RT
    d_acted uint4   !/  date_acted von RT
    maxseq  int4    !%max   nur informativ!
    domain  int4    !-!>domainkunde hier beantragter Domainantrag
    ipnr    int4    !-!>ipkunde hier beantragter inetnum-Block
    hperson int4    !-!>person  hier beantragter NIC-Handle
    hnic    int2    !-!nic  Registrar des Handles ^^^ dies sind die Tickets der Automaten. Ticket-Verweise auf die Kundenaufträge stehen NUR in den entsprechenden Tabellen
    zeit    int4    !-  für dieses Ticket budgetierte(!) Arbeitszeit
    triggerseq  int4    !-  auslösender Eintrg (z.B. Reseller-Mail)
    """
    id = models.IntegerField(primary_key=True)
    timestamp = models.DateTimeField(auto_now=True)
    ticket = models.ForeignKey(to='Ticket', null=True, db_column='ticket', blank=True, related_name='_parent_ticket')
    kunde = models.ForeignKey(to=Kunde, db_column='kunde')
    subject = models.CharField(max_length=255)
    infotext = models.CharField(blank=True, null=True, max_length=255)
    queue = models.ForeignKey(to='Queue', db_column='queue', raw_id_admin=True)
    queue_area = models.IntegerField(null=True, blank=True)  # models.ForeignKey(to='Queue_areas', null=True, db_column='queue_area', blank=True)
    beginn = models.UnixDateTimeField()
    #notiz = models.UnixDateTimeField(null=True, blank=True)
    #ende = models.UnixDateTimeField(null=True, blank=True)
    #ende2 = models.UnixDateTimeField(null=True, blank=True)
    #dringend = DescriptorField(default=0, descr_name='dringend', descr_model=Descr, db_column='dringend')
    wichtig = models.IntegerField()
    #maxwichtig = models.IntegerField()
    status = DescriptorField(default=0, descr_name='tickets', descr_model=Descr, db_column='status')
    termin = models.UnixDateTimeField(null=True, blank=True)
    endtermin = models.UnixDateTimeField(null=True, blank=True)
    bearbeiter = models.ForeignKey(to='Person', null=True, db_column='bearbeiter', blank=True, related_name='_ticket_bearbeiter')
    #eskaliere = models.UnixDateTimeField(null=True, blank=True)
    #eskaliert = models.UnixDateTimeField(null=True, blank=True)
    d_told = models.UnixDateTimeField()
    d_acted = models.UnixDateTimeField()
    maxseq = models.IntegerField()
    #domain = models.ForeignKey(to='Domainkunde', null=True, db_column='domain', blank=True, related_name='_domain_beantragt')
    #ipnr = models.ForeignKey(to='Ipkunde', null=True, db_column='ipnr', blank=True, related_name='_ip_beantragt')
    #hperson = models.ForeignKey(to='Person', null=True, db_column='hperson', blank=True)
    #hnic = DescriptorField(descr_name='nic', null=True, descr_model=Descr, db_column='hnic', blank=True)
    zeit = models.IntegerField(null=True, blank=True)
    #triggerseq = models.IntegerField(null=True, blank=True)

    class Meta:
        db_table = 'ticket'

    def __repr__(self):
        return "<Ticket #%d>" % self.id


class Rt_defaults(models.Model):
    """
    Verwendung: individuelle Default-Einstellungen fürs RT, vgl. RT#283876
    Index:  person
    Index:  timestamp
    Aufbau:
    person  int4    !^!>person  ID der Person, für die die Einstellungen gelten
    query   text2   -   Query-String fürs RT
    timestamp   datetime    !/+ Zeitstempel
    """
    person = models.ForeignKey(to=Person, db_column='person', primary_key=True)
    query = models.TextField()
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'rt_defaults'

    def __repr__(self):
        return "<Rt_defaults for %r>" % self.person
    
class Queue(models.Model):
    """
    Verwendung:	Liste der RT-Queues globale Einstellungen pro Queue
    Index:	id, name, email
    Index:	timestamp
    Aufbau:
    id	int4	!^!?+	ID (war: Pointer auf Stringtabelle)
    name	char30	-	Name der Queue
    email	char255	-	Mailadresse der Queue
    flags	int4	!*rt_queue
    defprio	int1	!(0-deffprio)
    deffprio	int1	!(defprio-99)
    defdue	int2	-	Tage Wiedervorlage
    deffdue	int2	-	Tage Endtermin
    defbearb	int4	!-!>person	default-Verantwortlicher
    timestamp	datetime	!/+	Zeitstempel
    """
    name = models.CharField(max_length=30)
    email = models.CharField(max_length=255)
    flags = DescriptorSet(descr_name='rt_queue', descr_model=Descr, db_column='flags')
    defprio = models.IntegerField()
    #deffprio = models.IntegerField()
    defdue = models.IntegerField()
    deffdue = models.IntegerField()
    defbearb = models.ForeignKey(to=Person, null=True, db_column='defbearb', blank=True)
    timestamp = models.DateTimeField(auto_now=True)
    
    class Meta:
        db_table="queue"
    
    
class Queue_acl(models.Model):
    """
    Verwendung: Berechtigungen für RT-Queues speichern
    Index:      person queue
    Index:      timestamp
    Aufbau:
    person      int4    !>person
    queue       int4    !^!>queue
    acls        int4    !*rt_acl
    timestamp   datetime        !/+     Zeitstempel
    """
    person = models.ForeignKey(to=Person, db_column='person', primary_key=True)
    queue = models.ForeignKey(to=Queue, db_column='queue', primary_key=True)
    acls = DescriptorSet(descr_name='rt_acl', descr_model=Descr, db_column='acls')
    timestamp = models.DateTimeField(auto_now=True)
    
    class Meta:
        db_table="queue_acl"
        has_composite_primary_key=True

class Uucpkunde(models.Model):
    """
    Verwendung: Aliasnamen für Kunden
    Index:  name
    Index:  kunde, timestamp
    Aufbau:
    name    char32  -   UUCP/ISDN-Name des Kunden
    timestamp   datetime    !/+ Zeitstempel
    kunde   int4    !^!>kunde   Nummer des Kunden Die Namen werden nicht recycelt!
    """
    name = models.CharField(primary_key=True, max_length=32)
    timestamp = models.DateTimeField(auto_now=True)
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')

    class Meta:
        db_table="uucpkunde"

class OpenXChangeLog(models.Model):
    """
    Verwendung: Speichern der GET-Aufrufe der Open-XChange-Einstellseite zum Nachweis
    Index:  id, person timestamp
    Index:  timestamp
    Aufbau:
    id  uint4   !?+ ID
    person  int4    !>person    Person, die die Einstellung aufgerufen hat
    timestamp   datetime    !/+ Zeitstempel, wann der Aufruf erfolgte
    anzeige text2   -   Angezeigter html-Text zum Nachweis
    """
    person = models.ForeignKey(to='Person', db_column='person')
    timestamp = models.DateTimeField(auto_now=True)
    anzeige = models.TextField()

    class Meta:
        db_table='open_xchange_log'
        unique_together=[('person', 'timestamp')]

    def __repr__(self):
        return "<OpenXChangeLog: %s on %s>" % (self.person, self.timestamp)


class OpenXChange(UpdatelogMixin, models.Model):
    """
    Verwendung: Einstellungen der Kunden bezüglich Open XChange
    Index:  id, kunde beginn, kunde log
    Aufbau:
    id  uint4   !?+ ID
    kunde   int4    !^__!>kunde Kunde, für den die Einstellung gilt
    beginn  uint4   !/  Zeitpunkt, ab dem die Einstellung wirksam wird
    timestamp   datetime    !/+ Zeitstempel, wann die Eingabe getätigt wurde
    option  int2    !open_xchange  Gewählte Einstellung
    log uint4   !>open_xchange_log  Verweis auf die Daten des zugehörigen GET-Requests
    request text2   -   Kompletter http-Request des POST-Requests zum Nachweis
    """
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    beginn = models.UnixDateTimeField()
    timestamp = models.DateTimeField(auto_now=True)
    option = DescriptorField(descr_name='open_xchange', descr_model=Descr, db_column='option')
    log = models.ForeignKey(to='OpenXChangeLog', db_column='log')
    request = models.TextField()

    class Meta:
        db_table='open_xchange'
        unique_together=[('kunde', 'beginn'), ('kunde', 'log')]
    class Updatelog:
        log_related=['kunde']

    def __repr__(self):
        return "<OpenXChange: %s set to %s logged %s>" % (self.kunde.name, self.option.bla, repr(self.log))

class Rechnung(models.Model):
    """
    Verwendung:	speichert ab, welche Kunden welche Rechnungen bekommen haben
    Index:	rnr
    Index:	kunde datum, betrag, datum kunde, timestamp
    Aufbau:
    rnr	int4	-	Rechnungsnummer
    timestamp	datetime	!/+	Zeitstempel
    datum	uint4	!/	Zeitpunkt der Rechnungserstellung
    kunde	int4	!^!>kunde	
    infotext	char255	-	für die Buchung
    betrag	int4	!$	Cent, ohne Märchensteuer
    steuer	int4	!$	Märchensteuer dazu
    datum1	uint4	!##!-!/	Änderung 1
    datum2	uint4	!##!-!/	Änderung 1
    datum3	uint4	!##!-!/	Änderung 1
    auftragsnr	char50	!-	Auftragsnummer
    rbetrag	int4	!-!$	Cent, bezahlter Betrag
    buchung	int4	!-	Buchungsnummer
    auszug	int4	!-	Kontoauszugsnummer
    flags	int4	!*rstatus	Gemailt/gedruckt/whatever
    rtext	text3	!%ctrl!-	Text der Rechnung
    fusstext	text2	!%ctrl!-	Text unter der Rechnung
    konto	int3	!-	Sachkonto
    storniert	uint4	!-!/	Datum zu dem die Rechnung storniert wurde (NULL: nicht storniert)
    """
    rnr = models.IntegerField(primary_key=True)
    timestamp = models.DateTimeField(auto_now=True)
    datum = models.UnixDateTimeField()
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    infotext = models.CharField(max_length=255)
    betrag = models.IntegerField()
    steuer = models.IntegerField()
    #datum1 = models.UnixDateTimeField(null=True, blank=True)
    #datum2 = models.UnixDateTimeField(null=True, blank=True)
    #datum3 = models.UnixDateTimeField(null=True, blank=True)
    auftragsnr = models.CharField(max_length=50, null=True, blank=True)
    rbetrag = models.IntegerField(null=True, blank=True)
    buchung = models.IntegerField(null=True, blank=True)
    auszug = models.IntegerField(null=True, blank=True)
    flags = DescriptorSet(Descr, 'rstatus', db_column='flags')
    rtext = models.TextField(null=True, blank=True)
    fusstext = models.TextField(null=True, blank=True)
    konto = models.IntegerField(null=True, blank=True)
    storniert = models.UnixDateTimeField(null=True, blank=True)
    
    class Meta:
        db_table="rechnungen"
        
    def __repr__(self):
        return "<Rechnung #%d vom %d für %s>" % (self.rnr, self.datum, smart_str(self.kunde.name))

class Nic(models.Model):
    """
    Verwendung: Mapping Personen zu NIC-Handles
    Index:  id, person nic, nic handle
    Index:  username, timestamp, status
    Aufbau:
    id  int4    !?+ ID
    person  int4    !^!%!>person    Mensch mit Handle
    nic int2    !nic    Center, bei dem der Handle registriert ist
    handle  char128 !-  Bezeichner des Handlperoes
    username    char15  !-  damit verbundener Login
    passwort    char15  !-  damit verbundenes Passwort
    ticket  int4    !-!>ticket  auslösendes Ticket (Kundenauftrag, Domain-Automaten-Ticket)
    status  int1    !handlestatus   Status des Vorgangs
    timestamp   datetime    !/+ Zeitstempel
    """
    person = models.ForeignKey(to='Person', db_column='person')
    nic = DescriptorField(Descr, descr_name='nic', db_column='nic')
    handle = models.CharField(max_length=128, null=True, blank=True)
    username = models.CharField(max_length=15, null=True, blank=True)
    passwort = models.CharField(max_length=15, null=True, blank=True)
    ticket = models.IntegerField(null=True, blank=True)
    status = DescriptorField(Descr, 'handlestatus', db_column='status')
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table="nic"
        unique_together=(("person", "nic"), ("nic", "handle"))

    def __repr__(self):
        return "<NIC %s %s -> %s>" % (self.nic.bla, self.handle, unicode(self.person))

class Mailassoc(models.Model):
    """
    Verwendung: Assoziation zusätzlicher Mailadressen zu einem Personen- oder Kundenrecord
    Index:  id, email
    Index:  person, timestamp
    Aufbau:
    id  int4    !?+ ID
    email   char255 -   Mailadresse
    person  int4    !^!>person  zugeordnete Person
    timestamp   datetime    !/+ Zeitstempel
    """
    email = models.CharField(max_length=255)
    person = models.ForeignKey(to='Person', db_column='person')
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'mailassoc'

class TroubleTicket(models.Model):
    """
    Tabelle:    trouble_ticket
    Index:  id
    Index:  timestamp, rtticket id
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    rtticket    int4    !^!-    zugehöriges Ticket (RT oder OTRS)
    subject text2   -   Titel des Trouble Tickets
    type    int2    !tt_type    Typ des Trouble Tickets
    beginn  uint4   !/  Beginn der Störung
    ende    uint4   !/!-    (ggf. voraussichtliches) Ende der Störung
    priority    int2    !tt_priority    Störungsklasse
    flag_kunde  int4    !-!%null!*kunde Auswahl der TT-Empfänger nach Kunden-Flags (ODER-verknüpft)
    flag_person int8    !-!%null!*pwdomain  Auswahl der TT-Empfänger nach Personen-Flags (ODER-verknüpft)
    text_description    text2   !%ctrl  Abschnitt "Problembeschreibung"
    text_affected   text2   !%ctrl!-    Abschnitt "Betroffen"
    text_comment    text2   !%ctrl!-    Abschnitt "Bemerkungen"
    text_progress   text2   !%ctrl!-    Abschnitt "Update"
    text_resolve    text2   !%ctrl!-    Abschnitt "Lösung"
    additional_rcpt text2   -   zusätzliche E-Mail-Empfänger, whitespace-getrennt
    confirmer   int4    !-!>person  Person, die diese Version des Tickets bestätigt hat
    """
    timestamp = models.DateTimeField(auto_now=True)
    rtticket = models.IntegerField(null=True, blank=True)
    subject = models.TextField()
    type = DescriptorField(descr_name='tt_type', descr_model=Descr, db_column='type')
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(null=True, blank=True)
    priority = DescriptorField(descr_name='tt_priority', descr_model=Descr, db_column='priority')
    flag_kunde = DescriptorSet(Descr, 'kunde', null=True, db_column='flag_kunde', blank=True)
    flag_person = DescriptorSet(Descr, 'pwdomain', null=True, db_column='flag_person', blank=True)
    text_description = models.TextField()
    text_affected = models.TextField(null=True, blank=True)
    text_comment = models.TextField(null=True, blank=True)
    text_progress = models.TextField(null=True, blank=True)
    text_resolve = models.TextField(null=True, blank=True)
    additional_rcpt = models.TextField()
    confirmer = models.ForeignKey(to='Person', null=True, db_column='confirmer', blank=True)

    class Meta:
        db_table="trouble_ticket"

class Confitem(models.Model):
    """
    Tabelle:    confitem
    Verwendung: Liste von Setups pro Kunde zur Kategorisierung von incident-Tickets, vgl. RT#199848
    Index:  id, kunde name
    Index:  timestamp
    Aufbau:
    id  int4    !?+ ID des Eintrags
    timestamp   datetime    !/+ Zeitstempel
    kunde   int4    !^!-!>kunde Kunde
    name    char255 -   Name des CIs
    """
    timestamp = models.DateTimeField(auto_now=True)
    kunde = models.ForeignKey(to='Kunde', null=True, db_column='kunde', blank=True)
    name = models.CharField(max_length=255)

    class Meta:
        db_table="confitem"
        unique_together=(('kunde','name'),)

class Leitung(models.Model):
    """
    Tabelle:    leitung
    Verwendung: (Stand-)Leitungen verwalten und zu Kunden assoziieren
    Index:  id
    Index:  timestamp, a_hardware, b_hardware
    Aufbau:
    id  int4    !?+ ID des Eintrags
    kunde   int4    !^!>kunde   Kunde
    ktarif  int4    !-!>tarifkunde  Tarif des Kunden, der zu dieser Leitung gehört (vgl. Ticket 10096055)
    name    char255 -   noris-Bezeichnung der Leitung
    art int2    !-!leitungsart  Art der Leitung, vgl. RT#211370
    carrier int4    !-!>person  Carrier für diese Leitung
    name_carrier    char255 !-  Leitungsbezeichnung des Carriers
    infotext  char255 !-  kundeneigene Leitungsbezeichnung
    backup  int4    !-!>person  Account für das (z. B. ISDN-)Backup, vgl. RT#381245
    a_ende  int4    !-!>person  A-Endpunkt der Leitung
    a_hardware  int4    !-!>hardware    Hardware am A-Ende (RT#402575)
    b_ende  int4    !-!>person  B-Endpunkt der Leitung
    b_hardware  int4    !-!>hardware    Hardware am B-Ende (RT#402575)
    timestamp   datetime    !/+ Zeitstempel
    beginn  uint4   !/  Zeitpunkt der Zuteilung der Adresse
    ende    uint4   !-!/    Freigabezeitpunkt der Adresse
    """
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    ktarif = models.ForeignKey(to='Tarifkunde', null=True, db_column='ktarif', blank=True)
    name = models.CharField(max_length=255)
    art = DescriptorField(descr_name='leitungsart', null=True, descr_model=Descr, db_column='art', blank=True)
    carrier = models.ForeignKey(to='Person', null=True, db_column='carrier', blank=True,
                                related_name='_person_by_carrier')
    name_carrier = models.CharField(max_length=255, null=True, blank=True)
    infotext = models.CharField(max_length=255, null=True, blank=True)
    backup = models.ForeignKey(to='Person', null=True, db_column='backup', blank=True,
                                related_name='_person_by_backup')
    a_ende = models.ForeignKey(to='Person', null=True, db_column='a_ende', blank=True,
                                related_name='_person_by_a_ende')
    a_hardware = models.ForeignKey(to='Hardware', null=True, db_column='a_hardware', blank=True,
                                related_name='_hardware_by_a')
    b_ende = models.ForeignKey(to='Person', null=True, db_column='b_ende', blank=True,
                                related_name='_person_by_b_ende')
    b_hardware = models.ForeignKey(to='Hardware', null=True, db_column='b_hardware', blank=True,
                                related_name='_hardware_by_b')
    timestamp = models.DateTimeField(auto_now=True)
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(null=True, blank=True)

    class Meta:
        db_table = "leitung"

class Stunden(models.Model):
    """
    Tabelle:    stunden
    Verwendung: Arbeitszeiten für Mitarbeiter speichern
    Index:  id
    Index:  person beginn, kunde beginn, timestamp, ticket beginn, zeit beginn, dauer
    Aufbau:
    id  int4    !^perso!?+  
    person  int4    !>person    Mitarbeiter
    kunde   int4    !>kunde Kunde, für den die Arbeit geleistet wurde
    ticket  uint8   !-!%ticket  Ticket hierzu
    beginn  uint4   !/  Startzeit
    dauer   uint4   !// wie lang gebraucht?
    zeit    uint4   !-!//   wie lang gebraucht? (Cache: mit Faktor, ohne verschachtelte Einträge. NULL = muss neu berechnet werden)
    art int2    !>stunden_art   Art des Stundeneintrags (normal/Urlaub/Hotline/...)
    infotext    text2   !%ctrl!-    was wurde getan?
    timestamp   datetime    !/+ Zeitstempel
    """
    person = models.ForeignKey(to='Person', db_column='person')
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    ticket = models.PositiveIntegerField(null=True, blank=True)
    beginn = models.UnixDateTimeField()
    dauer = models.PositiveIntegerField()
    zeit = models.PositiveIntegerField(null=True, blank=True)
    art = models.ForeignKey(to='StundenArt', db_column='art')
    infotext = models.TextField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = "stunden"

class StundenArt(models.Model):
    """
    Tabelle:    stunden_art
    Verwendung: Mappt Arten von Stundeneinträgen auf lesbare Beschreibungen etc.
    Index:  id, name
    Index:  timestamp
    Aufbau:
    id  int2    !?+ ID
    name    char30  -   textuelle Beschreibung (kurz)
    infotext    char255 !-  kurze Erklärung (Tooltip)
    faktor  int2    -   dem Mitarbeiter zugeordneter Anteil, in Prozent
    timestamp   datetime    !/+ Zeitstempel
    flags   int4    !*stunden_art   Flags
    """
    name = models.CharField(max_length=30, unique=True)
    infotext = models.CharField(max_length=255, null=True, blank=True)
    faktor = models.IntegerField()
    timestamp = models.DateTimeField(auto_now=True)
    flags = DescriptorSet(Descr, 'stunden_art', db_column='flags')

    class Meta:
        db_table = "stunden_art"

def gen_hash(values):
    hash = 0
    hash_mod = 0x7edcba9f // 57
    for val in values:
        if val is not None:
            hash %= hash_mod
            hash = hash + hash * 57 + hash // 5
            hash += val
    hash &=  0x7FFFFFFF
    return hash

class AcctManager(PopManager):
    def create(self, kunde, dienst, jjmm, tt, quelle, dest, bytes,
               pakete,
               **kwargs):
        hash = gen_hash([kunde.id, dienst.id, dest.descr, jjmm, tt, quelle.descr])
        cursor = db.connection.cursor()
        cursor.execute('select max(seq) from acct where hash=%d for update'
                       % hash)
        max_seq = cursor.fetchone()[0]
        if max_seq is None:
            max_seq = 0
        return super(AcctManager, self).create(
            kunde=kunde, dienst=dienst, jjmm=jjmm, tt=tt,
            quelle=quelle, dest=dest, bytes=bytes, pakete=pakete,
            hash=hash, seq=max_seq+1, **kwargs)


class Acct(UpdatelogMixin, models.Model):
    """
    Tabelle:    acct
    Verwendung: Speicherung von detaillierten Accountingrecords
    Index:  hash seq
    Index:  jjmm dienst dest, kunde jjmm dienst, dienst quelle jjmm, timestamp, dienst timestamp, quelle: quelle dienst timestamp
    Aufbau:
    hash    int4    !%hash  interner Index
    seq int2    !%max   Folgenummer für gleiche Hashes
    kunde   int4    !^!>kunde   ID des Kunden
    jjmm    int4    !(JJJJMM)   Jahr und Monat des Eintrags
    tt  int2    !(TT/jjmm)  Tag des Eintrags
    dienst  int2    !>dienst    Typ des Eintrags
    dest    int2    !(dienst!ziel)  Ziel
    pakete  uint8   -   Zahl der IP-Pakete / Mails / Verbindungen
    bytes   uint8   -   Summe der Bytes / Einheiten
    quelle  int2    !quelle Wer hat diese Bytes gezählt?
    timestamp   datetime    !/+ Zeitstempel
    """
    objects = AcctManager()

    hash = models.IntegerField(primary_key=True)
    seq = models.IntegerField(primary_key=True)
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    jjmm = models.IntegerField()
    tt = models.IntegerField()
    dienst = models.ForeignKey(to='Dienst', db_column='dienst')
    dest = DescriptorField(descr_name='ziel', descr_model=Descr, db_column='dest')
    pakete = models.PositiveIntegerField()
    bytes = models.PositiveIntegerField()
    quelle = DescriptorField(descr_name='quelle', descr_model=Descr, db_column='quelle')
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'acct'
        has_composite_primary_key = True
    class Updatelog:
        log_index="kunde"
        log_related=[]
        log_always=["hash","seq"]

    def is_editable(self):
        heute = datetime.date.today()
        return self.jjmm >= heute.year * 100 + heute.month

class Acctassoc(UpdatelogMixin, models.Model):
    """
    Tabelle:    acctassoc
    Verwendung: Assoziation eines Accountingeintrags mit Infotext und/oder genau einem Tarif
    Index:  id, >>acct hash seq
    Index:  tarifkunde, timestamp
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    hash    int4    !^acct!%    Accountingdatensatz
    seq int2    !%  
    tarifkunde  int4    !^!-!>tarifkunde    bezogener Kundentarif
    acctinfo    int2    !-!acctinfo Text für die Rechnung
    info    text2   !%ctrl!-    Zusatztext
    """
    timestamp = models.DateTimeField(auto_now=True)
    hash = models.IntegerField(primary_key=True)
    seq = models.IntegerField(primary_key=True)
    tarifkunde = models.ForeignKey(to='Tarifkunde', null=True, db_column='tarifkunde', blank=True)
    acctinfo = DescriptorField(descr_name='acctinfo', null=True, descr_model=Descr, db_column='acctinfo', blank=True)
    info = models.TextField(null=True, blank=True)

    class Meta:
        db_table = 'acctassoc'
        unique_together = (('hash', 'seq'),)
        has_composite_primary_key = True
    class Updatelog:
        log_related=[]
        log_always=["hash","seq"]

        @staticmethod
        def custom_index(obj):
            acct = Acct.objects.get(hash=obj.hash, seq=obj.seq)
            return {'kunde': acct.kunde.id}


class RtBilling(UpdatelogMixin, models.Model):
    """
    Tabelle:    rt_billing
    Verwendung: Verlinkung von RT-Tickets mit zugehörigen Accounting-Datensätzen, vgl. RT::AddOn::Billing
    Index:  id, >acct hash seq
    Index:  ticket, timestamp
    Aufbau:
    id  int4    !?+ ID
    timestamp   datetime    !/+ Zeitstempel
    ticket  int4    !^!>ticket  Verweis auf RT-Ticket
    hash    int4    !^acct  Verweis auf Accounting-Datensatz, Teil 1/2
    seq int2    -   Verweis auf Accounting-Datensatz, Teil 2/2
    gemailt uint4   !/!-    Timestamp des Accounting-Datensatzes, der an den vertrieblichen Ansprechpartner gemailt wurde
    """
    timestamp = models.DateTimeField(auto_now=True)
    ticket = models.IntegerField()
    hash = models.IntegerField(primary_key=True)
    seq = models.IntegerField(primary_key=True)
    gemailt = models.UnixDateTimeField(null=True, blank=True)

    class Meta:
        db_table = 'rt_billing'
        unique_together = (('hash', 'seq'),)
        has_composite_primary_key = True
    class Updatelog:
        log_related=[]

        @staticmethod
        def custom_index(obj):
            acct = Acct.objects.get(hash=obj.hash, seq=obj.seq)
            return {'kunde': acct.kunde.id}


class Tarifkunde(models.Model):
    """
    Tabelle:    tarifkunde
    Verwendung: Zuordnung eines / mehrerer Tarife zu einem Kunden
    Index:  id
    Index:  kunde dienst, kunde beginn, timestamp
    Aufbau:
    id  int4    !?+ ID des Eintrags
    kunde   int4    !^!>kunde   ID des Kunden <- kunde.id
    tarifname   int4    !-!>tarifname   Tarif.
    dienst  int2    !>dienst/tarif  genutzter Dienst
    anzahl  int2    -   Zahl der belegten Anschlüsse, Ports, etc.
    beginn  uint4   !/  ab wann genutzt?
    ende    uint4   !-!/    bis wann genutzt?
    ablauf  uint4   !-!/    bis wann vereinbart?
    notiz   uint4   !-!/    wann das System zuletzt Traffic bemerkt hat
    infotext    char255 !-  Kurzbeschreibung wieso (Domainname etc.)
    rechnung    char1   !-  Einrichtungsgebühr berechnet? 0/1 TODO bool
    nextrech    int4    !-!(JJJJMM) wann die nächste Rechnung zu stellen ist
    timestamp   datetime    !/+ Zeitstempel
    """
    kunde = models.ForeignKey(to='Kunde', db_column='kunde')
    tarifname = models.ForeignKey(to='Tarifname', null=True, db_column='tarifname', blank=True)
    dienst = models.ForeignKey(to='Dienst', db_column='dienst')
    anzahl = models.IntegerField()
    beginn = models.UnixDateTimeField()
    ende = models.UnixDateTimeField(null=True, blank=True)
    ablauf = models.UnixDateTimeField(null=True, blank=True)
    notiz = models.UnixDateTimeField(null=True, blank=True)
    infotext = models.CharField(max_length=255, null=True, blank=True)
    rechnung = models.CharField(max_length=1, null=True, blank=True)
    nextrech = models.IntegerField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'tarifkunde'


class Tarifname(models.Model):
    """
    Tabelle:    tarifname
    Verwendung: Namen von Tarifen
    Index:  id, name
    Index:  timestamp
    Aufbau:
    id  int4    !^!?+   ID des Tarifs
    timestamp   datetime    !/+ Zeitstempel
    name    char255 -   Name des Tarifs
    """
    timestamp = models.DateTimeField(auto_now=True)
    name = models.CharField(max_length=255, unique=True)

    class Meta:
        db_table = 'tarifname'