import re, logging

from django.utils.cache import add_never_cache_headers
from django.conf import settings
from django.db import connection
from django.template import Template, Context




_xhtml_content_type_re = re.compile(r'^application/xhtml\+xml(;|$)')
_accept_xhtml_re = re_accept_xhtml = re.compile(r'\bapplication/xhtml\+xml\b')


class XHTMLContentTypeMiddleware:
    """Change Content-Type from  application/xhtml+xml into
    text/html for browsers that don't accept xhtml.

    It was decided (sluka/mir) that this is sufficient and we don't need
    explicit conversion from xhtml to html4 (by parsing and reassembling).

    Beware: This middleware ignores your DOCTYPE, it only looks at your
    Content-Type header."""

    def process_response(self, request, response):
        if response.has_header('Content-Encoding'):
            return response
        if _xhtml_content_type_re.match(response['Content-Type']):
            # patch_vary_headers(response, 'Accept')   -- not needed due to NoCachingMiddleware.
            accept = request.META.get('HTTP_ACCEPT', '')
            if True or not _accept_xhtml_re.search(accept):
                ct = response['Content-Type']
                ct = ct.replace('application/xhtml+xml', 'text/html')
                response['Content-Type'] = ct
        return response


class NoCachingMiddleware:
    """Add headers that inhibit any caching. All static content will be served by Apache, so all
    django generated content depends on the database, which is not under our control.
    Note that the
    django.middleware.http.ConditionalGetMiddleware does allow If-None-Match queries
    (based on ETAGS).
    """

    def process_response(self, request, response):
        add_never_cache_headers(response)
        return response

class CacheStaticPagesMiddleware:
    """Speed up serving files by the development server. Each little picture takes sooooo long."""
    def __init__(self):
        self.static_urls = re.compile(settings.STATIC_URLS)
        self.cache = {}

    def process_request(self, request):
        if request.path in self.cache:
            return self.cache[request.path]
        else:
            return None

    def process_response(self, request, response):
        if self.static_urls.match(request.path):
            self.cache[request.path] = response
        return response

class XNorisPathMiddleware:
    """Some Ajax stuff needs a separate header with the path."""
    def process_response(self, request, response):
        response['X-Noris-Path'] = request.path.encode("utf-8")
        return response

class PrintTracebackMiddleware:
    """Output any traceback on stdout"""
    def process_exception(self, request, exception):
        if settings.DEBUG:
            import traceback
            traceback.print_exc()
        return None

class SQLLogMiddleware:
    def process_response ( self, request, response ): 
        time = sum((float(q['time']) for q in connection.queries), 0.0)
        if time >= 1.0:
            logger = logging.getLogger('sql.long')
            long_queries = [q for q in connection.queries if float(q['time']) > 0.5]
            if long_queries:
                for q in long_queries:
                    logger.info("\n%s: (%2.2f) %s" % (request.path, float(q['time']), q['sql']))
            else:
                logger.info("\n%s: (%2.3f) by %d short queries" % (request.path, time, len(connection.queries)))
                for i, q in enumerate(connection.queries):
                    logger.debug("  [%2d]: (%2.2f) %s" % (i, float(q['time']), q['sql']))
                logger.debug("end %s" % request.path)
        if time <= 0.02 and len(connection.queries) <= 5:
            prio = logging.DEBUG
        else:
            prio = logging.INFO
        logger = logging.getLogger('sql.log')
        logger.log(prio, "%s: %d queries, time %2.3fs" % (request.path, len(connection.queries), time))
        for i, q in enumerate(connection.queries):
            logger.debug("  [%2d]: (%2.2f) %s" % (i, float(q['time']), q['sql']))
        return response
