# -*- encoding: utf-8 -*-

import re, datetime
import dns.resolver, dns.name, dns.exception
from decimal import Decimal, InvalidOperation

from django.core.cache import cache
from django import oldforms
from django.conf import settings
from django.core import validators
from django.db.models.query import Q
from django.db.models.base import Model as DjangoBaseModel
from django.utils.translation import ugettext, ugettext_lazy, ungettext, gettext_lazy, ugettext as _
from django.utils.html import escape, conditional_escape
from django.utils.safestring import mark_safe, mark_for_escaping
from django.utils.encoding import force_unicode
from django.http import HttpResponse, HttpResponseRedirect
from django.oldforms import DateField, DatetimeField
from django.template.defaultfilters import escapejs
from kundebunt.popkern import models
from kundebunt.popkern.async_dns import AsyncDNSResolver
from kundebunt.popkern.utils import moneyfmt, smart_repr, unique, any, glob_to_like
from kundebunt.popkern.datasources import PersonDataSource, KundenDataSource, WartungsvertragDataSource
from kundebunt.contrib.advisory import AdvisoryFieldMixin

__all__ = ['DescriptorField', 'DescriptorQueryField', 'always_test', 'PasswordConfirmationField',
    'PasswordConfirmationValidator', 'DomainLabelField', 'SubdomainField', 'EmailCollectionField',
    'EmailTargetCollectionField', 'FQDNField', 'LocalpartField', 'isValidLocalpart', 'isValidNewlineSeparatedEmailList',
    'isValidNewlineSeparatedEmailTargetList', 'hasNoDollar', 'isValidUserName', 'isValidSubdomain', 'is_resolvable',
    'hasResolvableDomain', 'hasExistingEmailTargets', 'checkSourceDomain', 'isValidMailruleType', 'MonetaryField', 'moneyfmt',
    'SearchField', 'ComboField', 'DomainComboField', 'WildcardDomainComboField', 'PersonComboField', 'KundenComboField',
    'PersonSearchField', 'KundenSearchField',
    'WartungsvertragComboField', 'isValidUnixDateTime', 'EmailField', 'hasOnlyLatin1Characters']

valid_user_name_re = re.compile(r'^[a-z][-._a-z0-9]{0,63}$')
default_resolver = None

class MultipleMatches(ValueError):
    def __init__(self, candidates):
        self.candidates = candidates

class DescriptorField(oldforms.SelectField):
    def __init__(self, field_name, model, **kwargs):
        if "choices" not in kwargs:
            field = model._meta.get_field(field_name)
            kwargs["choices"] = field.get_choices()
        super(DescriptorField, self).__init__(field_name=field_name, **kwargs)

class DescriptorQueryField(oldforms.SelectField):
    """Wie DescriptorField, sieht aber zusätzlich einen Eintrag '*' vor.
    Zur Abfrage mit Jokerzeichen. Außerdem wird '-' statt des leeren Strings
    als value verwendet.
    """
    def __init__(self, field_name, model, **kwargs):
        if "choices" not in kwargs:
            field = model._meta.get_field(field_name)
            kwargs["choices"] = [(k or '-', v) for k,v in field.get_choices()]
        kwargs["choices"].insert(0,("*","*"))
        super(DescriptorQueryField, self).__init__(field_name=field_name, **kwargs)

def always_test(fn):
    """Decorator für eine Validator-Funktion. Bestimmt, dass diese auch
    bei leerem Formfeld aufgerufen wird.
    """
    fn.always_test = True
    return fn

class PasswordConfirmationField(oldforms.PasswordField):
    def __init__(self, field_name, pass_field_name, length=30, max_length=None, is_required=False, validator_list=None, member_name=None, readonly=False):
        if validator_list == None:
            validator_list = []
        validator_list = [PasswordConfirmationValidator(field_name, pass_field_name)] + validator_list
        super(PasswordConfirmationField, self).__init__(field_name, length, max_length, is_required, validator_list, member_name, readonly)

class PasswordConfirmationValidator(object):
    def __init__(self, field_name, pass_field_name):
        self.field_name = field_name
        self.pass_field_name = pass_field_name
        self.always_test = True

    def __call__(self, field_data, all_data):
        password_data = all_data[self.pass_field_name]
        if password_data:
            if not field_data:
                raise validators.ValidationError(ugettext_lazy("Bitte das Passwort zur Ueberpruefung zusaetzlich in das Bestaetigungsfeld eingeben."))
            if field_data != password_data:
                raise validators.ValidationError(ugettext_lazy("Die beiden Passwort-Eingaben stimmen nicht ueberein."))

class DomainLabelField(oldforms.TextField):
    """A form field containing one label of a domain"""
    pass

class SubdomainField(oldforms.TextField):
    pass

#class EmailField(forms.TextField):
    #"""A form field containing one email address or mailbox name"""
    #pass

class EmailCollectionField(oldforms.LargeTextField):
    """A LargeTextField containing a list of email addresses,
    one on each line
    """
    def __init__(self, field_name, rows=10, cols=40, is_required=False, validator_list=[], max_length=None, readonly=False):
        validator_list = validator_list + [self.isValidNewlineSeparatedEmailList]
        oldforms.LargeTextField.__init__(self, field_name, rows, cols, is_required, validator_list, max_length, readonly)

    def isValidNewlineSeparatedEmailList(self, field_data, all_data):
        try:
            isValidNewlineSeparatedEmailList(field_data, all_data)
        except validators.ValidationError, e:
            raise validators.CriticalValidationError, e.messages

class EmailTargetCollectionField(oldforms.LargeTextField, AdvisoryFieldMixin):
    """A LargeTextField containing a list of email addresses or localparts,
    one on each line
    """
    def __init__(self, field_name, rows=10, cols=40, is_required=False, validator_list=[], max_length=None, readonly=False):
        AdvisoryFieldMixin.__init__(self)
        validator_list = validator_list + [self.isValidNewlineSeparatedEmailTargetList]
        oldforms.LargeTextField.__init__(self, field_name, rows, cols, is_required, validator_list, max_length, readonly)

    def isValidNewlineSeparatedEmailTargetList(self, field_data, all_data):
        try:
            isValidNewlineSeparatedEmailTargetList(field_data, all_data)
        except validators.ValidationError, e:
            raise validators.CriticalValidationError, e.messages

class FQDNField(oldforms.TextField, AdvisoryFieldMixin):
    def __init__(self, field_name, length=30, max_length=None, is_required=False, validator_list=[], member_name=None, readonly=False):
        AdvisoryFieldMixin.__init__(self)
        validator_list = validator_list + [validators.MatchesRegularExpression(r'^(?:[A-Za-z0-9-]+\.)+[A-Za-z]{2,4}$', ugettext_lazy('Bitte geben Sie hier einen Domain-Namen an. Ein Domain-Name kann nur Buchstaben, Ziffer, "-" und "." enthalten.'))]
        oldforms.TextField.__init__(self, field_name, length, max_length, is_required, validator_list, member_name, readonly)
    """A form field containing a fully quoted domain name (without trailing dot)"""

class LocalpartField(oldforms.TextField):
    """A form field containing the localpart of an email address"""
    def __init__(self, field_name, length=30, max_length=None, is_required=False, validator_list=[], member_name=None, readonly=False):
        validator_list = validator_list + [self.isValidLocalpart]
        oldforms.TextField.__init__(self, field_name, length, max_length, is_required, validator_list, member_name, readonly)

    def isValidLocalpart(self, field_data, all_data):
        try:
            isValidLocalpart(field_data, all_data)
        except validators.ValidationError, e:
            raise validators.CriticalValidationError, e.messages

def isValidLocalpart(field_data, all_data):
    """>>> isValidLocalpart("abc.abc_67860-434+333",None)
    >>> isValidLocalpart("?",None)
    >>> isValidLocalpart('"ich\ bin\ eine\ E-Mail-Adresse"',None)
    >>> isValidLocalpart("ich bin keine E-Mail-Adresse",None)  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte eine ... E-Mail-Adresse eingeben.']
    >>> isValidLocalpart("@sind-nicht-erlaubt", None) #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte eine ... E-Mail-Adresse eingeben.']
    >>> isValidLocalpart('kein.am.ende.', None) #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte eine ... E-Mail-Adresse eingeben.']
    """
    validators.isValidEmail(field_data+"@example.com", all_data)

def isValidNewlineSeparatedEmailList(field_data, all_data):
    """>>> isValidNewlineSeparatedEmailList("bla@example.de\\nblo@example.de", None)
    >>> isValidNewlineSeparatedEmailList("bla@example.de blo@example.de", None) #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte ... E-Mail-Adressen eingeben, ...']
    """
    for supposed_email in field_data.split('\n'):
        try:
            validators.isValidEmail(supposed_email.strip(), '')
        except validators.ValidationError:
            raise validators.ValidationError, ugettext("Enter valid e-mail addresses, each on a separate line.")

def isValidNewlineSeparatedEmailTargetList(field_data, all_data):
    """Check whether the field contains lines with a single email target on each line.
    >>> nl=chr(10)
    >>> isValidNewlineSeparatedEmailTargetList(nl.join(["bla@example.de","blo@example.de","mbox2"]), None)
    >>> isValidNewlineSeparatedEmailTargetList("bla@example.de blo@example.de", None) #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte ... Zeile 1 ...']
    """
    hasNoDollar(field_data)
    for i, supposed_email in enumerate(field_data.split('\n')):
        try:
            validators.isValidEmail(supposed_email.strip(), '')
        except validators.ValidationError:
            try:
                isValidUserName(supposed_email.strip(), '')
            except validators.ValidationError:
                raise validators.ValidationError, ugettext("Enter valid e-mail addresses or mailboxes, each on a separate line. Something is wrong with line %d") % (i+1)

def hasNoDollar(field_data, all_data=None):
    """Überprüfung für Felder, die kein Dollarzeichen enthalten dürfen.
    >>> hasNoDollar("blabae a ad &/%/&/%/:.,. ")
    >>> hasNoDollar("asdasdd$dasdas")
    Traceback (most recent call last):
        ...
    ValidationError: [u'In diesem Feld sind keine Dollarzeichen ("$") erlaubt.']
    """
    if field_data.find("$") != -1:
        raise validators.ValidationError(ugettext('In diesem Feld sind keine Dollarzeichen ("$") erlaubt.'))

def isValidUserName(field_data, all_data=None):
    """Check whether the field contains a valid Noris user name.
    >>> isValidUserName("slu1")
    >>> isValidUserName("")
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte einen Benutzernamen eingeben; ein Benutzername beginnt mit einem Kleinbuchstaben und besteht aus den Zeichen a-z 0-9 - . _. Er kann maximal 64 Zeichen lang sein.']
    >>> isValidUserName("???")
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte einen Benutzernamen eingeben; ein Benutzername beginnt mit einem Kleinbuchstaben und besteht aus den Zeichen a-z 0-9 - . _. Er kann maximal 64 Zeichen lang sein.']
    >>> isValidUserName("bla$bla")
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte einen Benutzernamen eingeben; ein Benutzername beginnt mit einem Kleinbuchstaben und besteht aus den Zeichen a-z 0-9 - . _. Er kann maximal 64 Zeichen lang sein.']
    >>> isValidUserName("bla%bla")
    Traceback (most recent call last):
        ...
    ValidationError: [u'Bitte einen Benutzernamen eingeben; ein Benutzername beginnt mit einem Kleinbuchstaben und besteht aus den Zeichen a-z 0-9 - . _. Er kann maximal 64 Zeichen lang sein.']
    """
    if not valid_user_name_re.match(field_data):
        raise validators.ValidationError, ugettext("Bitte einen Benutzernamen eingeben; ein Benutzername beginnt mit einem Kleinbuchstaben und besteht aus den Zeichen a-z 0-9 - . _. Er kann maximal 64 Zeichen lang sein.")

def isValidSubdomain(field_data, all_data):
    pass

def is_resolvable(domain_string, rdatatypes, resolver):
    """Testet, ob der String in line auflösbar ist, und gibt entsprechend ein Boolean zurück."""
    for rdatatype in rdatatypes:
        answer = resolver.resolve(domain_string, rdatatype)
        if answer != None:
            return True
    return False


class hasResolvableDomain(object):
    """Testet, ob ein Feld eine DNS-auflösbare Domain enthält.
    Das funktioniert für Domains, Email-Adressen, auch für einen ganzen Text,
    bei dem in jeder Zeile ein Eintrag steht.

    >>> from kundebunt.popkern.async_dns import AsyncDNSResolver
    >>> resolver=AsyncDNSResolver()
    >>> nl=chr(10)
    >>> validator = hasResolvableDomain(resolver)
    >>> validator("noris.de")
    >>> validator("blazlsprazl.noris.de")
    Traceback (most recent call last):
        ...
    ValidationError: [u'blazlsprazl.noris.de: Die angegebene Domain ist im DNS nicht vorhanden.']
    >>> validator("braslfasl@noris.de")
    >>> validator(nl.join(["noris.de","braslfasl@noris.de"]))
    >>> validator("")
    >>> validator("aosidjasoasdioj")
    Traceback (most recent call last):
        ...
    ValidationError: [u'aosidjasoasdioj: Die angegebene Domain ist im DNS nicht vorhanden.']
    >>> validator("..")
    Traceback (most recent call last):
        ...
    ValidationError: [u'..: Die angegebene Domain ist im DNS nicht vorhanden.']
    >>> validator = hasResolvableDomain(resolver, ["mx"],"fehlermeldung")
    >>> validator("noris.de")
    >>> validator(nl.join(["blablara@noris.de","service.noris.de"]))
    Traceback (most recent call last):
        ...
    ValidationError: [u'service.noris.de: fehlermeldung']

    >>> validator = hasResolvableDomain(resolver, subdomain_field_name='subdomain')
    >>> validator('service.noris.net', {'subdomain': 'raxlbratzl'})
    Traceback (most recent call last):
        ...
    ValidationError: [u'raxlbratzl.service.noris.net: Die angegebene Domain ist im DNS nicht vorhanden.']
    >>> validator('raxlbratzl.noris.net', {'subdomain': '.'})
    Traceback (most recent call last):
        ...
    ValidationError: [u'raxlbratzl.noris.net: Die angegebene Domain ist im DNS nicht vorhanden.']

    >>> validator('noris.net', {'subdomain': 'service'})
    >>> validator('noris.net', {'subdomain': '.service'})
    """

    def __init__(self, resolver, rrtypes = None, error_message=gettext_lazy("Die angegebene Domain ist im DNS nicht vorhanden."), subdomain_field_name=None, allow_wildcard=False):
        self.resolver = resolver
        if rrtypes == None:
            rrtypes = ["A", "MX"]
        self.rrtypes = [dns.rdatatype.from_text(s) for s in rrtypes]
        self.error_message = error_message
        self.subdomain_field_name = subdomain_field_name
        self.allow_wildcard = allow_wildcard

    def _domains_to_check(self, field_data, all_data):
        """generiert alle zu überprüfenden Domains."""
        if not self.allow_wildcard or field_data!="*":
            for line in field_data.split("\n"):
                line = line.strip()
                if line:
                    at_pos = line.find('@')
                    if at_pos != -1:
                        # es ist eine email-adresse
                        domain = line[at_pos+1:]
                    else:
                        domain = line
                    if self.subdomain_field_name:
                        subdomain = all_data.get(self.subdomain_field_name,'')
                        if subdomain and subdomain[0] != '.':
                            domain = "%s.%s" % (subdomain, domain)
                    if len(domain)>=2 and domain[0] == '.' and domain[1] != '.':
                        domain = domain[1:]
                    elif len(domain)>=3 and domain[:2] == '*.' and domain[2] != '.':
                        domain = domain[2:]
                    yield domain


    def __call__(self, field_data, all_data=None):
        for domain in self._domains_to_check(field_data, all_data):
            resolvable = is_resolvable(domain + ".", self.rrtypes, self.resolver)
            if not resolvable:
                raise validators.ValidationError, "%s: %s" % (domain,self.error_message)

    def prepare(self, field_data, all_data=None):
        """submits the dns query to the resolver so that the reply can be ready in __call__"""
        for domain in self._domains_to_check(field_data, all_data):
            for rrtype in self.rrtypes:
                self.resolver.submit(domain + ".", rrtype)

class hasExistingEmailTargets(object):
    """Testet, ob ein Feld in jeder Zeile entweder eine Email-Adresse mit
    einer DNS-auflösbaren Domain oder aber einen Mailbox-Namen, der in der
    Datenbank als Mailbox eingetragen ist.

    >>> nl=chr(10)
    >>> from kundebunt.popkern.async_dns import AsyncDNSResolver
    >>> resolver = AsyncDNSResolver()
    >>> validator = hasExistingEmailTargets(resolver)
    >>> validator(nl.join(["blaxnkratzn@noris.de","fany","faslgasl@noris.de"]))
    >>> validator("fany")
    >>> validator("")
    >>> validator("noris.de","mir@noris.de")
    Traceback (most recent call last):
        ...
    ValidationError: [u'noris.de: Dieses Postfach ist bei noris network nicht registriert.']
    >>> validator(nl.join(["fany","rarara@raxlbratzl.service.noris.de"]))
    Traceback (most recent call last):
        ...
    ValidationError: [u'rarara@raxlbratzl.service.noris.de: Die angegebene Domain ist im DNS nicht vorhanden.']
    >>> validator("mir@blablagaga.m1.spieleck.de")
    """
    def __init__(self, resolver):
        self.resolver = resolver

    def _items_to_check(self, field_data):
        """generiert alle zu prüfenden Email-Adressen und Mailboxen

        Jedes Item ist ein Tupel (Bool:es_ist_eine_email_adresse, String:domain, String:ganze_zeile)
        """
        for line in field_data.split('\n'):
            line = line.strip()
            if line:
                at_pos = line.find('@')
                if at_pos != -1:
                    # es ist eine Email-Adresse.
                    yield (True, line[at_pos+1:], line)
                else:
                    # eine Mailbox
                    yield (False, '', line)

    def __call__(self, field_data, all_data=None):
        messages = []
        for is_email, domain, line in self._items_to_check(field_data):
            if is_email:
                if not is_resolvable(domain + u".", [dns.rdatatype.A, dns.rdatatype.MX], self.resolver):
                    messages.append("%s: %s" % (escape(line), ugettext("Die angegebene Domain ist im DNS nicht vorhanden.")))
            else:
                # eine Mailbox
                cached_result = cache.get('formfields.existing_mailbox.%s' % line)
                if not cached_result:
                    if models.Person.objects.filter(user__exact=line).has_flag("pwuse","mail").exists():
                        cache.set('formfields.existing_mailbox.%s' % line,  True, 5)
                    else:
                        messages.append(u"%s: %s" % (escape(line), ugettext("Diese Mailbox ist bei noris network nicht registriert.")))
        if messages:
            raise validators.ValidationError(mark_safe(u"<br />".join(messages)))

    def prepare(self, field_data, all_data=None):
        for is_email, domain, line in self._items_to_check(field_data):
            if is_email:
                for rdatatype in [dns.rdatatype.A, dns.rdatatype.MX]:
                    self.resolver.submit(domain + ".", rdatatype)
        return

class checkSourceDomain(object):
    """Testet. ob eine Domain aktiv ist und ob Sie dem Kunden zugeordnet ist.

    >>> from kundebunt.popkern.models import Domainkunde
    >>> from django.core import validators
    >>> validator = checkSourceDomain([Domainkunde(domain="bla.de"), Domainkunde(domain="blubb.de")], False)
    >>> validator("blubb.de")

    >>> validator("rabswrzlxl.de")
    Traceback (most recent call last):
        ...
    ValidationError: [u"Die Domain 'rabswrzlxl.de' ist bei noris network nicht registriert (kein Datenbankeintrag)."]
    >>> validator("noris.net") #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [u"Die Domain 'noris.net' ist nicht ... Sie zur Bearbeitung freigegeben. ..."]
    >>> validator = checkSourceDomain([Domainkunde(domain="bla.de"), Domainkunde(domain="blubb.de")], True)
    >>> validator("noris.net")

    >>> validator("rabswrzlxl.de")
    Traceback (most recent call last):
        ...
    ValidationError: [u"Die Domain 'rabswrzlxl.de' ist bei noris network nicht registriert (kein Datenbankeintrag)."]
    """
    def __init__(self, allowed_domains, user_is_staff, allow_wildcard=False):
        self.allowed_domains = allowed_domains
        self.user_is_staff = user_is_staff
        self.allow_wildcard = allow_wildcard

    def __call__(self, field_data, all_data=None):
        if self.allow_wildcard and field_data=="*":
            return
        for domain in self.allowed_domains:
            if domain.domain == field_data:
                return
        if models.Domainkunde.objects.active().filter(domain__iexact=field_data).exists():
            if not self.user_is_staff:
                raise validators.ValidationError, ugettext("Die Domain '%s' ist nicht fuer Sie zur Bearbeitung freigegeben. Aenderungen nur durch die noris network AG.") % field_data
        else:
            raise validators.ValidationError, ugettext("Die Domain '%s' ist bei noris network nicht registriert (kein Datenbankeintrag).") % field_data

class isValidMailruleType(object):
    """Testet, ob ein Benutzer einen bestimmten rule type angeben darf.

    >>> validator = isValidMailruleType(user_is_staff=True)
    >>> validator("virt")
    >>> validator("sms1")
    >>> validator("sms10")
    >>> validator("xadadas")  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    >>> validator = isValidMailruleType(user_is_staff=False)
    >>> validator("virt")
    >>> validator("sms1")  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    >>> validator("sms10")  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    >>> validator("xadadas")  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    """
    def __init__(self, user_is_staff):
        self.user_is_staff = user_is_staff

    def __call__(self, field_data, all_data=None):
        try:
            descr = models.Mailrule.get_typ_descr(field_data)
        except models.Descr.DoesNotExist:
            raise validators.ValidationError, ugettext("Unzulaessiger Regeltyp.")
        if (not self.user_is_staff
              and not descr.descr in models.Mailrule.get_typ_group_list('kunde')):
            raise validators.ValidationError, ugettext("Unzulaessiger Regeltyp.")

class isValidUnixDateTime(object):
    """Testet, ob ein DateTime-Field innerhalb der Grenzen liegt, die
    durch ein UnixDateTimeField abgebildet werden können.

    >>> import datetime
    >>> validator = isValidUnixDateTime()
    >>> validator('1970-01-01 1:00')
    >>> validator('2038-01-19 0:00')
    >>> validator('1970-01-01 0:00')  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    >>> validator('2038-01-20 0:00')  #doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    ValidationError: [...]
    >>> isValidUnixDateTime.parse_date('2000-07-11')
    datetime.datetime(2000, 7, 11, 0, 0)
    """
    min_date = datetime.datetime.fromtimestamp(0)
    max_date = datetime.datetime.fromtimestamp((1<<31)-1)

    def __init__(self, parser = DatetimeField.html2python):
        self.parser = parser
        
    def __call__(self, field_data, all_data=None):
        field_data = self.parser(field_data)
        if field_data:
            if field_data < self.min_date or field_data > self.max_date:
                raise validators.ValidationError(
                    _("Der Datumsbereich ist auf den Bereich von %(min)s bis %(max)s eingeschraenkt.")
                    % {'min': self.min_date.isoformat(' '), 'max': self.max_date.isoformat(' ')})

    @staticmethod
    def parse_date(value):
        return datetime.datetime.combine(DateField.html2python(value), datetime.time(0,0,0))

class hasOnlyLatin1Characters(object):
    """Testet, ob ein Textfeld ausschließlich iso-8859-1-Zeichen enthält.

    >>> validator = hasOnlyLatin1Characters(message=u"Geht nicht.")
    >>> validator(u'axzAXZ059!?=+#,&*/%.()[]_$§{}@;')
    """
    def __init__(self, message):
        self.message = message
        
    def __call__(self, field_data, all_data=None):
        try:
            field_data.encode('iso-8859-1', 'strict')
        except UnicodeEncodeError:
            raise validators.ValidationError(self.message)
    
    
class MonetaryField(oldforms.TextField):
    """Ein Feld für Geldbeträge. Die Beträge werden als String verwaltet.

    >>> f = MonetaryField("test")
    >>> f.get_validation_errors({"test": ''})
    {}
    >>> f.get_validation_errors({"test": '0'})
    {}
    >>> f.get_validation_errors({"test": '0,20'})
    {}
    >>> f.get_validation_errors({"test": '2.80'})
    {}
    >>> f.get_validation_errors({"test": '2.380'}).keys()
    ['test']
    >>> MonetaryField.html2python('0')
    '0,00'
    >>> MonetaryField.html2python('0.2')
    '0,20'
    >>> MonetaryField.html2python('232')
    '232,00'
    >>> MonetaryField.html2python('1,2')
    '1,20'
    """
    
    def __init__(self, *args, **kwargs):
        oldforms.TextField.__init__(self, *args, **kwargs)
        self.validator_list.append(validators.MatchesRegularExpression(
            r"^\s*\d+([,.]\d\d)?\s*$",
            _("Bitte einen gueltigen Geldbetrag eingeben.")))

    @staticmethod
    def html2python(data):
        if data:
            try:
                return moneyfmt(Decimal(data.replace(",", ".")), sep='', dp=',')
            except InvalidOperation:
                raise ValueError
        else:
            return None

class SearchField(oldforms.TextField):
    """Ein Feld für Jokersuchen. Besonderheit gegenüber dem
    normalen TextField ist lediglich der Hilfetext."""
    def render(self, data):
        if data is None:
            data = ''
        maxlength = ''
        readonly = ''
        if self.max_length:
            maxlength = 'maxlength="%s" ' % self.max_length
        if self.readonly:
            readonly = 'readonly="readonly"'
        if isinstance(data, unicode):
            data = data.encode(settings.DEFAULT_CHARSET)
        if self.readonly:
            pattern = '''<input type="%(type)s" id="%(id)s" class="%(cls)s%(required)s" name="%(name)s" size="%(size)s" value="%(value)s" %(maxlength)s %(readonly)s/>'''
        else:
            pattern = '''<input type="%(type)s" id="%(id)s" class="%(cls)s%(required)s" name="%(name)s" size="%(size)s" value="%(value)s" %(maxlength)s %(readonly)s/><div class="field_help"><a href="#"><span class="htrigger">S</span><span class="htext">%(hilfe)s</span></a></div>'''
        return mark_safe( pattern % (dict(
                        type='text',
                        id=self.get_id(), cls=self.css_class(),
                        required=self.is_required and ' required' or '',
                        name=self.field_name, size=self.length, value=escape(data),
                        maxlength=maxlength, readonly=readonly,
                        hilfe=_(
"""Dieses Feld definiert ein Suchkriterium, das mit den Jokerzeichen '<tt>*</tt>' und '<tt>?</tt>' arbeitet. '<tt>?</tt>' steht fuer genau ein beliebiges Zeichen, '<tt>*</tt>' fuer beliebig viele davon. Wenn man das Feld leer laesst, wird es bei der Suche ignoriert. Hier sind ein paar Beispiele fuer sinnvolle Suchkriterien:

    <span class="dd"><tt>*abc*</tt></span>
    <span class="dt">Sucht nach Eintraegen, in denen 'abc' als Teilstring irgendwo vorkommt, z.B. 'XXXabcXXX'.</span>
    
    <span class="dd"><tt>abc*</tt></span>
    <span class="dt">Sucht nach Eintraegen, die mit 'abc' beginnen, z.B. 'abcXXX'.</span>
    
    <span class="dd"><tt>*abc</tt></span>
    <span class="dt">Sucht nach Eintraegen, die auf 'abc' enden, z.B. 'XXXabc'.</span>

Ein Sonderfall ist das '<tt>-</tt>' (alleine im Eingabefeld). Damit kann man nach leeren bzw. fehlenden Eintraegen suchen. Das ist allerdings nicht bei allen Eingabefeldern sinnvoll.
""")
                )))


class ComboField(oldforms.FormField, AdvisoryFieldMixin):
    """Eine Kombination aus SelectField und TextField für Verwendung mit JavaScript

    Auf der Benutzeroberfläche ist dieses Feld entweder
    - wie ein SelectField (wenn choices_only==True)
    - oder wie ein TextField, eventuell mit einer Vorschlagsliste

    Die Datenschnittstelle ist aber immer wie beim TextField, d.h. es wird jeweils
    der angezeigte String als `data` übergeben, und nicht eine id, wie beim SelectField.
    ids spielen keine Rolle, und `choices` ist eine Liste ohne ids, also [string, ...]

    Wenn choices_only und der aktuelle Wert nicht in der Liste der zulässigen Werte
    vorkommt, wird der Wert angezeigt, aber bei der Validierung angemeckert.

    Geplant: Mit JavaScript fungiert, wenn choices_only==False ist,
    die Werteliste als Vorschlagsliste, und tab-completion ist per tab_complete möglich.

    >>> from django.utils.encoding import smart_str
    >>> from datasources import ArrayDataSource
    >>> class Data1(ArrayDataSource):
    ...   data=[["eins"], ["zwei"]]
    ...   @staticmethod
    ...   def is_request_permitted(request):
    ...     return True
    ...   @staticmethod
    ...   def from_request(request):
    ...     return Data1()

    >>> f = ComboField('test', data_source=Data1(), use_javascript=False)
    >>> # Normalfall, ein Wert aus `choices`
    >>> print f.render("zwei") #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField" name="test" size="1" >
      <option value=""></option>
      <option value="eins">eins</option>
      <option value="zwei" selected="selected">zwei</option>
    </select>

    >>> # Ein nicht vorhandener Wert wird einfach dazugefügt.
    >>> print f.render("drei") #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField" name="test" size="1" >
      <option value=""></option>
      <option value="eins">eins</option>
      <option value="zwei">zwei</option>
      <option value="drei" selected="selected">drei</option>
    </select>

    >>> # Noch nichts angegeben
    >>> print f.render('') #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField" name="test" size="1" >
      <option value="" selected="selected"></option>
      <option value="eins">eins</option>
      <option value="zwei">zwei</option>
    </select>

    >>> # html2python gibt immer den erhaltenen Wert zurück
    >>> print f.html2python('zwei')
    zwei
    >>> print f.html2python('drei')
    drei
    >>> len(f.html2python(''))
    0
    >>> # Validierung
    >>> f.get_validation_errors({"test": "zwei"})
    {}
    >>> f.get_validation_errors({"test": ""})
    {}
    >>> f.get_validation_errors({"test": "drei"})["test"][0].encode('utf-8')
    'Bitte eine g\\xc3\\xbcltige Auswahl treffen.'
    >>> # Tests mit is_required.
    >>> f = ComboField('test', data_source=Data1(), is_required=True, use_javascript=False)
    >>> # Normalfall
    >>> print f.render("zwei") #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField required" name="test" size="1" >
      <option value="eins">eins</option>
      <option value="zwei" selected="selected">zwei</option>
    </select>

    >>> # Nicht vorhandener Wert, wird auch hier einfach dazugefügt
    >>> print f.render("drei") #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField required" name="test" size="1" >
      <option value="eins">eins</option>
      <option value="zwei">zwei</option>
      <option value="drei" selected="selected">drei</option>
    </select>

    >>> # Wenn als leer vorbesetzt, ergibt sich eine Eingabeaufforderung.
    >>> print f.render('').encode('ascii','ignore') #doctest: +NORMALIZE_WHITESPACE
    <select id="id_test" class="vComboField required" name="test" size="1" >
      <option value="" selected="selected">- Bitte auswhlen -</option>
      <option value="eins">eins</option>
      <option value="zwei">zwei</option>
    </select>
    >>> # Validierung
    >>> f.get_validation_errors({"test": "zwei"})
    {}
    >>> print f.get_validation_errors({"test": ""})
    {'test': [u'Bitte dieses Feld ausf\\xfcllen.']}
    >>> print f.get_validation_errors({"test": "drei"})["test"][0].encode("ascii","ignore")
    Bitte eine gltige Auswahl treffen.

    >>> # Tests mit choices_only==False
    >>> f = ComboField('test', data_source=Data1(), choices_only=False, use_javascript=False)
    >>> # Normalfall
    >>> print f.render("zwei")
    <input type="text" id="id_test" class="vComboField" name="test" size="30" value="zwei"  />
    >>> # Nicht vorhandener Wert wird einfach eingetragen
    >>> print f.render("drei")
    <input type="text" id="id_test" class="vComboField" name="test" size="30" value="drei"  />
    >>> print f.html2python('drei')
    drei

    >>> # Validierung
    >>> f.get_validation_errors({"test": "zwei"})
    {}
    >>> f.get_validation_errors({"test": ""})
    {}
    >>> f.get_validation_errors({"test": "drei"})
    {}

    >>> # Mit is_required
    >>> f = ComboField('test', data_source=Data1(), choices_only=False, is_required=True, use_javascript=False)
    >>> # Normalfall
    >>> f.render("zwei")
    u'<input type="text" id="id_test" class="vComboField required" name="test" size="30" value="zwei"  />'
    >>> # Nicht vorhandener Wert
    >>> print f.render("vier")
    <input type="text" id="id_test" class="vComboField required" name="test" size="30" value="vier"  />
    >>> print f.html2python("vier")
    vier

    >>> # Validierung
    >>> f.get_validation_errors({"test": "zwei"})
    {}
    >>> f.get_validation_errors({"test": ""})
    {'test': [u'Bitte dieses Feld ausf\\xfcllen.']}
    >>> f.get_validation_errors({"test": "drei"})
    {}
    """
    choices_title = ugettext_lazy('Vorschlaege')
    min_query_length = 1

    def __init__(self, field_name, data_source, size=1,
                 is_required=False, validator_list=None, member_name=None, readonly=False,
                 choices_only=True, length=30, max_length=None, use_javascript=None,
                 choice_error_message=ugettext_lazy("Bitte eine gueltige Auswahl treffen."),
                 null_display="", max_choices=200, min_query_length=None):
        AdvisoryFieldMixin.__init__(self)
        self.length, self.max_length = length, max_length
        if validator_list is None: validator_list = []
        self.field_name = field_name
        self.size, self.is_required = size, is_required
        # choices in a SelectField (value, human-readable key) tuples
        self.data_source = data_source
        self.use_javascript = use_javascript
        self.choices_only = choices_only
        if choices_only:
            self.validator_list = [self.isValidChoice] + validator_list
        else:
            self.validator_list = validator_list
        self.readonly = readonly
        if member_name != None:
            self.member_name = member_name
        self.nondisplay_choices = []
        self.choice_error_message = choice_error_message
        self.null_display = null_display
        self.max_choices = max_choices
        if min_query_length is not None:
            self.min_query_length = min_query_length

    def render(self, data):
        if self.use_javascript==None:
            if not self.readonly and self.choices_only and len(self.data_source) <= 30:
                return self.render_non_javascript(data)
            else:
                return self.render_javascript(data)
        else:
            if self.use_javascript:
                return self.render_javascript(data)
            else:
                return self.render_non_javascript(data)

    def render_javascript(self, data):
        if data is None:
            data = u''
        maxlength = u''
        readonly = u''
        if self.max_length:
            maxlength = u'maxlength="%s" ' % self.max_length
        if self.readonly:
            readonly = u'readonly="readonly"'
        #if isinstance(data, unicode):
            #data = data.encode(settings.DEFAULT_CHARSET)
        if self.readonly:
            pattern = u'''<input type="%(type)s" id="%(id)s" class="%(cls)s%(required)s" name="%(name)s" size="%(size)s" value="%(value)s" %(maxlength)s %(readonly)s/>'''
        else:
            pattern = u'''<input type="%(type)s" id="%(id)s" class="%(cls)s%(required)s" name="%(name)s" size="%(size)s" value="%(value)s" %(maxlength)s %(readonly)s/><div class="field_help"><a href="#"><span class="htrigger">*</span><span class="htext">%(combo_hilfe)s</span></a></div>
<div id="%(id)s_ac_div" class="accontainer"></div>
<script type="text/javascript">
var %(prefix)s_ds = new YAHOO.widget.DS_XHR(%(url)s,%(ds_spec)s);
%(prefix)s_ds.responseType = YAHOO.widget.DS_XHR.TYPE_JSON;
%(prefix)s_ds.queryMatchSubset = true;
var %(prefix)s_ac = new YAHOO.widget.AutoComplete('%(id)s', '%(id)s_ac_div', %(prefix)s_ds);
%(prefix)s_ac.minQueryLength = %(min_query_length)d;
%(prefix)s_ac.animVert = false;
%(prefix)s_ac.queryDelay = 1.0;
%(prefix)s_ac.maxResultsDisplayed = 200;
</script>
'''
        return mark_safe( pattern % (dict(
                        type=u'text',
                        id=self.get_id(), cls=self.css_class(),
                        prefix = u"combo_" + self.get_id().replace(u".",u"_"),
                        required=self.is_required and u' required' or u'',
                        name=force_unicode(self.field_name), size=self.length, value=force_unicode(escape(data)),
                        min_query_length = self.min_query_length,
                        maxlength=maxlength, readonly=readonly,
                        url=smart_repr(self.data_source.url()),
                        ds_spec="['data',%s]" % ",".join((smart_repr(s) for s in self.data_source.att_names)),
                        choices_only=self.choices_only and u"true" or u"false",
                        combo_hilfe=_(u"Dieses Feld bietet Auswahlmoeglichkeiten an. Druecken Sie die Taste mit dem Pfeil nach rechts am Eingabeende, um passende Moeglichkeiten anzuzeigen. Tab uebernimmt den hervorgehobenen Eintrag. Esc schliesst das Auswahlmenue. Die Liste ist unvollstaendig, wenn es zuviele Egaenzungsmoeglichkeiten gibt.")
                        )
               ))

    javascript_header =  u'''<script src="/yui-noris/yahoo.js /">
                            <script src="/yui-noris/dom.js /">
                            <script src="/yui-noris/event.js /">
                            <script src="/yui-noris/connection.js /">
                            <script src="/yui-noris/autocomplete.js /">'''


    def render_non_javascript(self, data):
        disabled = ''
        choices = self.data_source.get_choices()
        if self.readonly:
            disabled=u'disabled="disabled"'
        if self.choices_only and (self.max_choices == None or len(choices) <= self.max_choices):
            output = [u'<select id="%s" class="%s%s" name="%s" size="%s" %s>' % \
                (self.get_id(), self.css_class(),
                self.is_required and u' required' or u'', self.field_name, self.size,
                disabled)]
            str_data = force_unicode(data) # normalize to string
            found = False
            if not self.is_required or str_data==u'' or data==None:
                # Wir müssen einen Leereintrag generieren
                selected_html=u''
                display_name=self.null_display
                if str_data==u'' or data==None:
                    selected_html = u' selected="selected"'
                    found = True
                if self.is_required:
                    if not display_name:
                        display_name=ugettext('- Bitte auswaehlen -')
                output.append(u'    <option value="%s"%s>%s</option>' % (u'', selected_html, force_unicode(mark_for_escaping(display_name))))
            for value, display_name in choices:
                selected_html = ''
                if force_unicode(value) == str_data:
                    selected_html = u' selected="selected"'
                    found = True
                output.append(u'    <option value="%s"%s>%s</option>' % (force_unicode(mark_for_escaping(value)), selected_html, force_unicode(mark_for_escaping(display_name))))
            if not found:
                output.append(u'    <option value="%s"%s>%s</option>' % (force_unicode(mark_for_escaping(data)), u' selected="selected"', force_unicode(mark_for_escaping(data))))
            output.append(u'</select>')
            return mark_safe(u'\n'.join(output))
        else:
            maxlength = u''
            readonly = u''
            if self.max_length:
                maxlength = u'maxlength="%s" ' % self.max_length
            if self.readonly:
                readonly = u'readonly="readonly"'
            #if isinstance(data, unicode):
                #data = data.encode(settings.DEFAULT_CHARSET)
            return mark_safe(u'<input type="%s" id="%s" class="%s%s" name="%s" size="%s" value="%s" %s %s/>' % \
            (u'text', self.get_id(), self.css_class(), self.is_required and u' required' or u'',
            self.field_name, self.length, escape(data), maxlength, readonly))

    #@staticmethod
    #def html2python(data):
        #return data
        #if data=='':
            #return None
        #else:
            #return forms.FormField.html2python(data)

    def add_nondisplay_choices(self, choice_list):
        """adds acceptable input values which will not be displayed in a select field.
        This is for special cases, such as accepting the current value."""
        self.nondisplay_choices.append(choice_list)

    def isValidChoice(self, data, form):
        if self.readonly:
            return
        str_data = force_unicode(data)
        if str_data not in self.data_source:
            str_nondisplay_choices = [force_unicode(item) for item in self.nondisplay_choices]
            if str_data not in str_nondisplay_choices:
                raise validators.ValidationError(self.choice_error_message % {u'data': str_data, u'choices': force_unicode(self.data_source)})

    # always get vComboField..., even for subclasses
    def css_class(self, rw_class_name='ComboField'):
        return super(ComboField, self).css_class(rw_class_name)

class DomainComboField(ComboField):
    
    min_query_length = 0
    
    def __init__(self, field_name, data_source=None, size=1,
                 is_required=False, validator_list=None, member_name=None, readonly=False,
                 choices_only=True, length=30, max_length=None,
                 use_javascript=None, null_display=''):
        if validator_list==None:
            validator_list = []
        validator_list = [validators.MatchesRegularExpression(r'^(?:\*\.|\.|)(?:[A-Za-z0-9-]+\.)+[A-Za-z]+$', ugettext_lazy('Bitte geben Sie hier einen Domain-Namen an. Ein Domain-Name kann nur Buchstaben, Ziffern, "-" und "." enthalten.'))] + validator_list
        super(DomainComboField, self).__init__(field_name, data_source, size,
                 is_required, validator_list, member_name, readonly,
                 choices_only, length, max_length, use_javascript=use_javascript,
                 null_display=null_display,
                 choice_error_message=ugettext_lazy("Bitte waehlen Sie eine Ihnen zugeordnete aktive Domain."))


class WildcardDomainComboField(DomainComboField):
    """Erlaubt auch "*"
    """
    
    min_query_length = 0
    
    def __init__(self, field_name, data_source=None, size=1,
                 is_required=False, validator_list=None, member_name=None, readonly=False,
                 choices_only=True, length=30, max_length=None):
        if validator_list==None:
            validator_list = []
        validator_list = [validators.MatchesRegularExpression(r'^(?:\*\.|\.|)(?:[A-Za-z0-9-]+\.)+[A-Za-z]+$|^\*$', ugettext_lazy('Bitte geben Sie hier einen Domain-Namen an. Ein Domain-Name kann nur Buchstaben, Ziffern, "-" und "." enthalten. Ausserdem ist "*" ohne weitere Eingaben erlaubt.'))] + validator_list
        super(DomainComboField, self).__init__(field_name, data_source, size,
                 is_required, validator_list, member_name, readonly,
                 choices_only, length, max_length, choice_error_message=ugettext_lazy("Bitte waehlen Sie eine Ihnen zugeordnete aktive Domain."))

class MultiComboField(ComboField):

    manager = NotImplemented

    def get_matches_table(self, candidates):
        raise NotImplementedError

    @classmethod
    def html2python(cls, data):
        if data in (None, u""):
            return None
        candidates = list(unique(cls.get_matches(data)))
        if len(candidates)==0:
            raise ValueError
        if len(candidates)==1:
            return candidates[0]
        else:
            raise MultipleMatches(candidates)

    @classmethod
    def get_matches(cls, data):
        return cls.manager.default_query(data, queryset_required=False)


    def isValidChoice(self, data, form):
        if data in (None, u"") or self.readonly:
            return
        if isinstance(data, DjangoBaseModel):
            obj = data
        else:
            if any(data==unicode(item) for item in self.nondisplay_choices):
                return
            candidates = list(self.get_matches(data))
            if len(candidates)==0:
                raise validators.ValidationError(u"Es gibt keinen zur Eingabe passenden Eintrag.")
            elif len(candidates)==1:
                obj = candidates[0]
            else:
                candidates = [c for c in candidates if c in self.data_source]
                if len(candidates)==0:
                    raise validators.ValidationError(u"Es gibt keinen zur Eingabe passenden Eintrag.")
                if len(candidates)==1:
                    msg = u"Es gibt weitere passende Einträge, aber nur ein Eintrag ist erlaubt. Bitte klicken Sie ihn zum Übernehmen an."
                else:
                    msg = u"<p>Auf die Eingabe passen mehrere Einträge. Bitte klicken Sie den richtigen zur Auswahl an.</p>"
                if len(candidates) <= 50:
                    raise validators.ValidationError(mark_safe(
                            u'%s<table class="multi_match" cellspacing="2" width="100%%">'
                            u'%s'
                            u'</table>'
                            % (msg, self.get_match_rows(candidates)
                        )))
                else:
                    raise validators.ValidationError(
                            u"Auf die Eingabe passen %d Einträge. Bitte grenzen Sie die Suche weiter ein."
                            % len(candidates)
                        )
        if obj in self.data_source:
            return
        else:
            raise validators.ValidationError, self.choice_error_message % {
                    'data': data, 'choices': str(self.data_source)}

    def onclick(self, data):
        return (u'var el=getElementById(&quot;%s&quot;);'
                u'el.value=&quot;%s&quot;;'
                u'this.parentNode.parentNode.parentNode.className=&quot;val_error val_resolved&quot;;'
                u'el.focus();' % (self.get_id(), conditional_escape(escapejs(data))))

def _f(s):
    if not s:
        return u""
    else:
        return conditional_escape(s)

class PersonComboField(MultiComboField):
    """Ein ComboField zur Eingabe von Personen-Referenzen. Neben dem Benutzername kann auch
    die (numerische) id eingegeben werden. Die datasource liefert nur die Benutzernamen.
    In data steht das Personen-Objekt selbst.
    """

    manager = models.Person.objects
    
    def __init__(self, *args, **kwargs):
        if not "data_source" in kwargs:
            kwargs["data_source"] = PersonDataSource()
        super(PersonComboField, self).__init__(*args, **kwargs)

    def get_match_rows(self, candidates):
        return (u'<col align="right" /><col /><col /><col /><col class="hide_in_compact" />'
                '<tr><th>ID</th><th>User</th><th>Kunde</th><th>Name</th><th>E-Mail</th></tr>'
                u'\n%s\n'
                % u"\n".join([
                    u'<tr class="row%s" onclick="%s">'
                    u'<td align="right">#%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s&nbsp;</td></tr>'
                    % (i % 2 + 1, self.onclick(PersonComboField.extract(p)),
                       p.id, _f(p.user), _f(p.kunde.name), _f(p.name), _f(p.email))
                    for (i, p) in enumerate(candidates)
                ]))

    def extract_data(self, data_dict):
        """Gibt den Wert für den Kontext zurück, der in das
        html-Feld eingetragen werden soll. Wenn das Personen-Felder
        in data_dict enthalten und nicht leer|None ist, basiert
        die Anzeige auf der Person, andernfalls auf den data_dict[feldname]
        """
        return PersonComboField.extract(data_dict.get(self.field_name))

    @staticmethod
    def extract(data):
        if isinstance(data, models.Person):
            """Ausgabe in einem PersonComboField"""
            if data.user:
                return u"%s (%s)" % (data.user, data.name)
            elif data.name:
                return u"%d (%s)" % (data.id, data.name)
            else:
                return unicode(data.id)
        else:
            return data

class PersonSearchField(PersonComboField):
    @staticmethod
    def html2python(data):
        return data
    

class KundenComboField(MultiComboField):
    """Ein ComboField zur Eingabe von Personen-Referenzen. Neben dem Benutzername kann auch
    die (numerische) id eingegeben werden. Die datasource liefert nur die Benutzernamen.
    In data steht aber immer der Kunden-Name (anders als bei PersonComboField).

    Wenn `person` nicht zum Personal gehört, wird automatisch ein SimpleKundenComboField
    statt des KundenComboFields erzeugt. Siehe RT#427334.
    """

    manager = models.Kunde.objects

    def __new__(cls, person, *args, **kwargs):
        if person.is_staff():
            return super(KundenComboField, cls).__new__(cls, person, *args, **kwargs)
        else:
            return SimpleKundenComboField(person, *args, **kwargs)

    def __init__(self, person, *args, **kwargs):
        if not "data_source" in kwargs:
            kwargs["data_source"] = KundenDataSource(person)
        self.person = person
        super(KundenComboField, self).__init__(*args, **kwargs)

    def get_match_rows(self, candidates):
        return (u'<col align="right" /><col />'
                '<tr><th style="text-align:right">ID</th><th>Kunde</th></tr>'
                u'\n%s\n'
                % u"\n".join([
                    u'<tr class="row%s" onclick="%s">'
                    u'<td align="right">#%s</td><td>%s</td></tr>'
                    % (i % 2 + 1, self.onclick(k.name),
                       k.id, _f(k.name))
                    for (i, k) in enumerate(candidates)
                ]))

    @classmethod
    def html2python(cls, data):
        if data in (None, u""):
            return data
        else:
            return super(KundenComboField, cls).html2python(data).name


class SimpleKundenComboField(KundenComboField):
    """Wie KundenComboField, führt aber keine Suche durch.

    Dieser Fall ist für Kundenzugriff gedacht, da der externe Webserver auf einige
    Tabellen, die bei der Kundensuche benutzt werden, keinen Zugriff hat.
    Siehe RT#427334.

    Der Nutzer dieser Klasse muss selber prüfen, ob der Benutzer überhaupt Zugriff
    auf diesen Kunden hat. (Aufgrund der durch django.oldforms vorgegebenen Struktur
    ist html2python eine Klassenmethode, und daher auch get_matches ...)
    """

    def __new__(cls, *args, **kwargs):
        return super(KundenComboField, cls).__new__(cls, *args, **kwargs)

    @classmethod
    def get_matches(cls, data):
        return cls.manager.filter(name__exact=data)

class KundenSearchField(KundenComboField):
    @staticmethod
    def html2python(data):
        return data


class WartungsvertragComboField(ComboField):
    """Ein ComboField zur Auswahl von Wartungsverträgen. Neben dem Namen kann auch die
    (numerische) id eingegeben werden. Die datasource liefert nur die Namen. In data
    steht das Wartungsvertrags-Objekt selbst.
    """
    _numeric_re = re.compile(r'\s*#?\s*([1-9][0-9]*)(\s.*)?$')
    min_query_length = 0

    def __init__(self, *args, **kwargs):
        if not "data_source" in kwargs:
            kwargs["data_source"] = WartungsvertragDataSource()
        super(WartungsvertragComboField, self).__init__(*args, **kwargs)

    @staticmethod
    def _parse_post_data(data):
        """schaut sich die Benutzereingaben in `data` an
        und gibt (pk_int, user_string) zurück, wobei eins
        von beiden immer None ist. Für ein leeres data wird
        (None, '') zurückgegeben.
        """
        match = WartungsvertragComboField._numeric_re.match(data)
        if match:
            return (int(match.group(1)), None)
        else:
            return (None, data)

    def isValidChoice(self, data, form):
        if self.readonly:
            return
        if isinstance(data, models.Wartungsvertrag):
            (pk, user) = (data.id, None)
        else:
            (pk, user) = self.__class__._parse_post_data(data)
        if pk is not None:
            if self.data_source.query_set().filter(pk=pk).exists():
                return
        elif user:
            if user in self.data_source:
                return
            str_nondisplay_choices = [str(item) for item in self.nondisplay_choices]
            if user in str_nondisplay_choices:
                return
        else:
            return
        raise validators.ValidationError, self.choice_error_message % {'data': data, 'choices': str(self.data_source)}

    @staticmethod
    def html2python(data):
        if data is None:
            return None
        else:
            (pk, user) = WartungsvertragComboField._parse_post_data(data)
            try:
                if pk is not None:
                    return models.Wartungsvertrag.objects.get(pk=pk)
                elif user:
                    return models.Wartungsvertrag.objects.get(name=user)
                else:
                    return None
            except models.Wartungsvertrag.DoesNotExist:
                raise ValueError

    def extract_data(self, data_dict):
        """Gibt den Wert für den Kontext zurück, der in das
        html-Feld eingetragen werden soll. Wenn das Personen-Felder
        in data_dict enthalten und nicht leer|None ist, basiert
        die Anzeige auf der Person, andernfalls auf den data_dict[feldname]
        """
        wv = data_dict.get(self.field_name)
        if isinstance(wv, models.Wartungsvertrag):
            if wv.name:
                return wv.name
            else:
                return "#%d" % wv.id
        elif wv:
            return wv
        else:
            data = data_dict.get(self.get_member_name(), None)
            if data is None:
                data = ''
            return data

class EmailField(oldforms.EmailField, AdvisoryFieldMixin):
    def __init__(self, *args, **kwargs):
        if 'resolver' in kwargs:
            resolver = kwargs.pop('resolver')
        else:
            resolver = AsyncDNSResolver()
        super(EmailField, self).__init__(*args, **kwargs)
        self.add_advisors([hasResolvableDomain(resolver)])
    pass
