# -*- encoding: utf-8 -*-

"""
Klassen für Eingabefelder und Formulare, sowie Unterstützung in der Formularverarbeitung
"""

import types, logging
from django.db import models, connection
from django.utils.functional import curry
from django.db.models.query import QuerySet, Q
from django.dispatch import dispatcher
from django.db.models import signals, fields
from django import oldforms
from django.utils.safestring import mark_safe
from django.utils.html import escape
from django.conf import settings



def follow_only(model_cls, field_list):
    """builds the "follow" dict that specifies which fields to take from
    the POST data.

    Manipulators are passed a dict { fieldname : Boolean } as `follow`.
    `follow_only()` builds this dict from the model class and a list of
    field names (containing the fields to take from the POST data.

    This should really go into a library module or to the django contributions.
    """

    res = dict.fromkeys(model_cls._meta.get_follow().keys(), False)
    res.update(dict.fromkeys(field_list,True))
    return res



class ReadonlyTextField(oldforms.TextField):
    """Like a TextField, but readonly.
    """
    def render(self, data):
        if data is None:
            data = ''
        maxlength = ''
        if self.max_length:
            maxlength = 'maxlength="%s" ' % self.max_length
        if isinstance(data, unicode):
            data = data.encode(settings.DEFAULT_CHARSET)
        return mark_safe('<input type="%s" id="%s" class="v%s%s" name="%s" size="%s" value="%s" readonly="readonly" %s/>' % \
            (self.input_type, self.get_id(), self.__class__.__name__, self.is_required and ' required' or '',
            self.field_name, self.length, escape(data), maxlength))


###############################################################
# Hier kommen die Highlights: Deskriptoren und Konsorten


def contribute_descr_methods(field, cls, name):
    """Fügt eine Reihe von deskriptorspezifischen Zugriffsfunktionen
    in die Django-Model-Klasse ein.
    """
    def FIELD_in_group(fieldname, obj, groupname):
        """Gibt zurück, ob der Deskriptor für ein Feld in einer Gruppe ist.
        """
        bitmask = obj.__getattribute__("get_%s_group_bitmask" % fieldname)(groupname)
        return field._descr_groups[obj.__getattribute__(name).descr] & bitmask != 0
        #return obj.__getattribute__(fieldname).gruppe & bitmask != 0
    def get_FIELD_group_list(obj, fieldname, groupname):
        """Gibt die Liste der Codes (Feld ``descr`` aller Deskriptoren in einer Gruppe zurück
        """
        bitmask = obj._group_codes[groupname]
        return [ code for code,group in obj._descr_groups.iteritems() if group & bitmask != 0 ]
    #def all_FIELD_descr(obj):
        #return field._descr_model.objects.filter(typ=field._descr_typ)
    def get_FIELD_descr_type(obj):
        """Gibt den typ (als Integer) eines Deskriptor-Felds zurück
        """
        return obj._descr_typ
    def get_FIELD_descr_model(obj):
        """Gibt das Modell des zugehörigen Deskriptors zurück (wohl meist kundebunt.popkern.models.Descr)
        """
        return obj._descr_model
    def get_FIELD_descr(obj, name):
        """Holt den Descriptor eines bestimmten Namens aus der Datenbank"""
        return obj._descr_model.objects.filter(typ=obj._descr_typ, bla=name).get()
    def get_FIELD_descr_in_group(obj, groupname):
        bitmask = obj._group_codes.get(groupname,0)
        return (obj._descr_objs[code] for code, group in obj._descr_groups.iteritems() if group & bitmask != 0)
    
    setattr(cls, 'get_%s_code' % name, staticmethod(curry(dict.get, field._descr_codes)))
    setattr(cls, 'get_%s_descr' % name, staticmethod(curry(get_FIELD_descr, field)))
    setattr(cls, 'get_%s_descr_type' % name, staticmethod(curry(get_FIELD_descr_type, field)))
    setattr(cls, 'get_%s_descr_model' % name, staticmethod(curry(get_FIELD_descr_model, field)))
    setattr(cls, 'get_%s_descr_obj' % name, staticmethod(curry(dict.__getitem__, field._descr_objs)))
    if field._group_codes:
        setattr(cls, 'get_%s_group_bitmask' % name, staticmethod(curry(dict.__getitem__, field._group_codes)))
        setattr(cls, 'get_%s_group_list' % name, staticmethod(curry(get_FIELD_group_list, field, name)))
        setattr(cls, '%s_in_group' % name, curry(FIELD_in_group, name))
        setattr(cls, 'get_%s_descr_in_group' % name, staticmethod(curry(get_FIELD_descr_in_group, field)))
        #setattr(cls, 'all_%s_descr' % name, staticmethod(curry(all_FIELD_descr, field)))

class ReverseDescrObjectDescriptor(object):
    """Lädt den Descr-Eintrag aus dem verweisenden Object.
    """
    def __init__(self, field_with_rel):
        self.field = field_with_rel

    def __get__(self, instance, instance_type=None):
        if instance is None:
            raise AttributeError, "%s must be accessed via instance" % self.field.name
        return self.field._descr_objs.get(getattr(instance, self.field.attname), None)

    def __set__(self, instance, value):
        raise AttributeError, "%s is a read only field." % self.field.name

    def __repr__(self):
        return "<%s for field '%s'>" % (self.__class__.__name__, self.field.name)


class _NeverTheSame(object):
    """Ein String, der != sich selber ist. singleton.

    Die einzige Instanz dieser Klasse wird mit _NeverTheSame.singleton()
    erzeugt.

    Wir brauchen das in DescriptorField, um die model validation von django
    zu behumpsen. Dieser Test geht davon aus, dass es zwischen Modellen, die
    mit ForeignKey verknüpft sind, Zugriffsfunktionen (genauer: Deskriptoren) gibt.
    Das ist aber bei einem DecriptorField in der Richtung vom Descr nicht möglich
    (da sinnlos). Entsprechend wird eine Instanz dieser Klasse als related_name
    eingetragen.
    """
    def __eq__(self, other):
        return False
    def __ne__(self, other):
        return True
    def __str__(self):
        return "noaccess"
    @classmethod
    def singleton(cls):
        if not hasattr(cls, "_singleton"):
            cls._singleton = cls()
        return cls._singleton


class DescrCacheAccess(object):
    cache = {}

    def __init__(self, attname):
        self.attname = attname

    def __get__(self, field, field_type):
        try:
            return self.cache[field.descr_name][self.attname]
        except KeyError:
            descr_model = field._descr_model
            descr_name = field.descr_name
            group_name = field.group_name
            cache_entry = dict(
                typ    = None,
                codes  = DescrCacheDict(descr_model, descr_name, group_name),
                labels = DescrCacheDict(descr_model, descr_name, group_name),
                groups = DescrCacheDict(descr_model, descr_name, group_name),
                objs = DescrCacheDict(descr_model, descr_name, group_name),
                group_typ = None,
                group_codes = group_name and DescrCacheDict(descr_model, descr_name, group_name),
                group_labels = group_name and DescrCacheDict(descr_model, descr_name, group_name))
            self.cache[field.descr_name] = cache_entry
            self.__class__.populate_cache(descr_model, descr_name, group_name)
            return self.cache[field.descr_name][self.attname]

    #def __set__(self, field, value):
        #try:
            #self.cache[field.descr_name][self.attname].update(value)
        #except KeyError:
            #self.cache[field.descr_name][self.attname] = DescrCacheDict(field, value)

    @classmethod
    def populate_cache(cls, descr_model, descr_name, group_name):
        logging.info("populating descr cache for %s" % descr_name)
        cache_entry = cls.cache[descr_name]
        info = DescrCacheAccess._get_descriptor_info(descr_model, descr_name)
        typ, codes, labels, groups, objs = info
        cache_entry['typ'] = typ
        cache_entry['codes'].set(codes)
        cache_entry['labels'].set(labels)
        cache_entry['groups'].set(groups)
        cache_entry['objs'].set(objs)
        if group_name is not None:
            g_typ, g_codes, g_labels, dummy, dummy2 = cls._get_descriptor_info(
                    descr_model, group_name)
            for (key,val) in g_codes.iteritems():
                g_codes[key] = 1 << val
            cache_entry['group_typ'] = g_typ
            cache_entry['group_codes'].set(g_codes)
            cache_entry['group_labels'].set(g_labels)

    @staticmethod
    def _get_descriptor_info(descr_model, name):
        """Fuer eine Noris-spezifische Erweiterung von ForeignKey zur Behandlung von "Descriptoren."
        Gibt ein Tupel (descr_typ, descr_codes, descr_labels) zurück
        - descr_typ = der (numerische) typ des Deskriptors
        - descr_codes = { descriptor_label (aka "bla") : code }
        - descr_labels = { code : descriptor_label }
        - descr_groups = { code : group }
                                   (group is an int treated as bitmask)
        - descr_obj = { code: descr }
        """
        from django.db import connection
        cursor = connection.cursor()
        try:
            cursor.execute("""select id from descr_typ
                           where name=%s""",
                           [name])
            row = cursor.fetchone()
            if row == None:
                raise KeyError, "No info for descriptor type  %s in descr_typ" % name
        except:
            return (0, {}, {}, {}, {})
        typ = row[0]
        codes = {}
        labels = {}
        groups = {}
        descrobjs = {}
        for d in descr_model.objects.filter(typ=typ):
            codes[d.bla] = d.descr
            labels[d.descr] = d.bla
            groups[d.descr] = d.gruppe
            descrobjs[d.descr] = d
        return (typ, codes, labels, groups, descrobjs)


class DescrFieldMixin(object):

    def __init__(self, descr_model, descr_name, group_name=None):
        self._descr_model = descr_model
        self.descr_name = descr_name
        self.group_name = group_name

    _descr_typ = DescrCacheAccess('typ')
    _descr_codes = DescrCacheAccess('codes')
    _descr_labels = DescrCacheAccess('labels')
    _descr_groups = DescrCacheAccess('groups')
    _descr_objs = DescrCacheAccess('objs')
    _group_typ = DescrCacheAccess('group_typ')
    _group_codes = DescrCacheAccess('group_codes')
    _group_labels = DescrCacheAccess('group_labels')


class DescrCacheDict(dict):
    def __init__(self, descr_model, descr_name, group_name , *args, **kwargs):
        self.descr_model = descr_model
        self.descr_name = descr_name
        self.group_name = group_name
        super(DescrCacheDict, self).__init__(*args, **kwargs)

    def __getitem__(self, key):
        try:
            return super(DescrCacheDict, self).__getitem__(key)
        except KeyError:
            DescrCacheAccess.populate_cache(self.descr_model, self.descr_name, self.group_name)
            return super(DescrCacheDict, self).__getitem__(key)

    def set(self, mapping):
        self.clear()
        self.update(mapping)


class DescriptorField(models.ForeignKey, DescrFieldMixin):
    def __init__(self, descr_model, descr_name, choices_sort_key=lambda (descr,bla):descr, **kwargs):
        DescrFieldMixin.__init__(self, descr_model, descr_name, "%s_ident" % descr_name)
        # add a "limit_choices_to" into kwargs:
        kwargs.setdefault("limit_choices_to",{}).update(
            [('typ__exact', self._descr_typ)])
        choices = self._descr_labels.items()
        choices.sort(key=choices_sort_key)
        kwargs["choices"] = choices
        kwargs.setdefault("related_name", _NeverTheSame.singleton())
        models.ForeignKey.__init__(self, descr_model, to_field = "descr", **kwargs)


    def get_descriptor(self, descr):
        """returns the (cached) descriptor object for a given database value aka descr"""
        return self._descr_objs[descr]

    def get_internal_type(self):
        return 'IntegerField'

    def contribute_to_class(self, cls, name):
        self.related_name = "%s_by_%s" % (cls._meta.module_name, name)
        contribute_descr_methods(self, cls, name)
        super(models.ForeignKey, self).contribute_to_class(cls, name)
        setattr(cls, self.name, ReverseDescrObjectDescriptor(self))

    def contribute_to_related_class(self, cls, related):
        """für DescriptorFields sind revers-Deskriptoren a la descr.blabla_set sinnfrei."""
        pass



class ReverseCharDescrObjectDescriptor(ReverseDescrObjectDescriptor):
    """Lädt den Descr-Eintrag aus dem verweisenden Object.
    """
    def __get__(self, instance, instance_type=None):
        if instance is None:
            raise AttributeError, "%s must be accessed via instance" % self.field.name
        value = getattr(instance, self.field.attname)
        if value == None:
            return None
        else:
            return self.field._descr_objs[ord(value)]

class ChrDescriptor(object):
    """Macht die Konvertierung int->char für ein CharDescriptorField.
    Letztlich liest man immer ein char, man kann aber auch integers hineinschreiben.
    """
    def __init__(self, field):
        self.field = field
        self.attname = "_ChrDescriptor_%s" % field.attname

    def __get__(self, instance, instance_type=None):
        if instance is None:
            raise AttributeError, "%s must be accessed via instance" % self.field.name
        return getattr(instance, self.attname)

    def __set__(self, instance, value):
        if instance is None:
            raise AttributeError, "%s must be accessed via instance" % self.field.name
        if not value is None and not isinstance(value, types.StringTypes):
            value = chr(value)
        object.__setattr__(instance, self.attname, value)


class CharDescriptorField(fields.CharField, DescrFieldMixin):
    """Ein faszinierendes Feld. Wird als char gespeichert, ist aber von der Idee her ein
    ForeignKey zu einem Integer-Feld. Daher von CharField abgeleitet, vieles stammt
    aber von ForeignKey. ForeignKey oder Related kann aber nicht als Basisklasse dienen, weil ansonsten
    Django davon ausgeht, dass da auch ein Integer-Feld und ein foreign key dahinter steckt.
    <sarkasmus>  Super-Idee, dieser char-Deskriptor  </sarkasmus>
    """
    empty_strings_allowed = False
    def __init__(self, descr_model, descr_name, descr_group=None, **kwargs):
        DescrFieldMixin.__init__(self, descr_model, descr_name, descr_group or ("%s_ident" % descr_name))
        # add a "limit_choices_to" into kwargs:
        choices = [(chr(key),val) for (key,val) in self._descr_labels.items()]
        choices.sort(key=lambda (descr,bla):descr)
        kwargs["choices"] = choices
        fields.CharField.__init__(self, **kwargs)

        self.db_index = True

    def get_attname(self):
        return '%s_id' % self.name

    def prepare_field_objs_and_params(self, manipulator, name_prefix):
        params = {'validator_list': self.validator_list[:], 'member_name': name_prefix + self.attname}
        #if self.rel.raw_id_admin:
            #field_objs = self.get_manipulator_field_objs()
            #params['validator_list'].append(curry(manipulator_valid_rel_key, self, manipulator))
        #else:
        if self.radio_admin:
            field_objs = [oldforms.RadioSelectField]
            params['ul_class'] = fields.get_ul_class(self.radio_admin)
        else:
            if self.null:
                field_objs = [oldforms.NullSelectField]
            else:
                field_objs = [oldforms.SelectField]
        params['choices'] = self.get_choices_default()
        return field_objs, params

    def get_db_prep_save(self, value):
        return value or None

    def flatten_data(self, follow, obj=None):
        if not obj:
            # In required many-to-one fields with only one available choice,
            # select that one available choice. Note: For SelectFields
            # (radio_admin=False), we have to check that the length of choices
            # is *2*, not 1, because SelectFields always have an initial
            # "blank" value. Otherwise (radio_admin=True), we check that the
            # length is 1.
            if not self.blank:
                choice_list = self.get_choices_default()
                if self.radio_admin and len(choice_list) == 1:
                    return {self.attname: choice_list[0][0]}
                if not self.radio_admin and len(choice_list) == 2:
                    return {self.attname: choice_list[1][0]}
        return fields.CharField.flatten_data(self, follow, obj)

    def contribute_to_class(self, cls, name):
        contribute_descr_methods(self, cls, name)
        super(CharDescriptorField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, ReverseCharDescrObjectDescriptor(self))
        setattr(cls, self.attname, ChrDescriptor(self))

    def get_internal_type(self):
        return 'CharField'


class DescriptorSet(models.IntegerField, DescrFieldMixin):
    def __init__(self, descr_model, descr_name, **kwargs):
        DescrFieldMixin.__init__(self, descr_model, descr_name, "%s_ident" % descr_name)
        models.IntegerField.__init__(self, **kwargs)
        #for (key,val) in self._descr_codes.iteritems():
            #self._descr_codes[key] = 1 << val

    def contribute_to_class(self, cls, name):
        def get_FIELD_descr_model(obj):
            """Gibt das Modell des zugehörigen Deskriptors zurück (wohl meist kundebunt.popkern.models.Descr)
            """
            return obj._descr_model
        def has_FIELD_flag(fieldname, obj, flagname):
            bitmask = get_FIELD_bitmask(flagname)
            return obj.__getattribute__(fieldname) & bitmask != 0
        def FIELD_set(fieldname, descrobjs, obj):
            bits = obj.__getattribute__(fieldname)
            res = []
            for descr in descrobjs:
                if (1<<descr.descr) & bits:
                    res.append(descr)
            return res
        def get_FIELD_descr_in_group(obj, groupname):
            bitmask = obj._group_codes[groupname]
            print bitmask
            return (obj._descr_objs[code] for code, group in obj._descr_groups.iteritems() if group & bitmask != 0)
        def get_FIELD_descr_type(obj):
            """Gibt den typ (als Integer) eines Deskriptor-Felds zurück
            """
            return obj._descr_typ
        def FIELD_display(fieldname, descrobjs, obj):
            flags = [d.bla for d in FIELD_set(fieldname, descrobjs, obj)]
            flags.sort()
            return u" ".join(flags)
        def get_FIELD_bitmask(flag):
            return 1 << self._descr_codes[flag]

        setattr(cls, 'get_%s_bitmask' % name, staticmethod(get_FIELD_bitmask))
        setattr(cls, 'has_%s_flag' % name, curry(has_FIELD_flag, name))
        setattr(cls, '%s_set' % name, curry(FIELD_set, name, self._descr_objs.values()))
        setattr(cls, 'get_%s_descr_in_group' % name, staticmethod(curry(get_FIELD_descr_in_group, self)))
        setattr(cls, 'get_%s_descr_model' % name, staticmethod(curry(get_FIELD_descr_model, self)))
        setattr(cls, 'get_%s_descr_type' % name, staticmethod(curry(get_FIELD_descr_type, self)))
        setattr(cls, '%s_display' % name, curry(FIELD_display, name, self._descr_objs.values()))
        models.IntegerField.contribute_to_class(self, cls, name)

    def get_internal_type(self):
        return "BigIntegerField"



class PopQuerySet(models.query.QuerySet):
    """This customized QuerySet is aware of our "descriptors"
    """
    def has_any_bit(self, descr_field_name, bitmask):
        """filters using a descriptor and a bitmask.
        A row passes this filter if the descriptor has at least any of the
        bits in the (numeric) <bitmask> set.
        """
        qn = connection.ops.quote_name
        return self.extra(where=
            ["%s.%s & %d != 0"
             % (qn(self.model._meta.db_table),
                qn(descr_field_name),
                bitmask)])

    def has_flag(self, descr_field_name, flagname):
        """filters using a descriptor and a name (string) of a bit.
        A row passes if the descriptor has (at least) set the given bit.
        """
        bitmask = getattr(self.model,"get_%s_bitmask" % descr_field_name)(flagname)
        return self.has_any_bit(descr_field_name, bitmask)

    def in_group(self, descr_field_name, groupname):
        """filters using a descriptor and a name (string) of a group.
        A row passes if the descriptor belongs to the group.
        """
        #bitmask = getattr(self.model,"get_%s_group_bitmask" % descr_field_name)(groupname)
        groups = getattr(self.model,"get_%s_group_list" % descr_field_name)(groupname)
        #attrname = self.model._meta.get_field(descr_field_name).attrname
        params = {"%s__descr__in" % descr_field_name: groups}
        return self.filter(**params)


    def descr_eq(self, descr_field_name, symbolic_name):
        """filters using a descriptor and the symbolic name of a value.
        A row passes if the descriptor has exactly the given values
        associated with `symbolic_name`
        """
        qn = connection.ops.quote_name
        value = getattr(self.model,"get_%s_code" % descr_field_name)(symbolic_name)
        return self.extra(where=["(%s.%s = %d)" % (qn(self.model._meta.db_table),
                                                   descr_field_name, value)])

    def descr_ne(self, descr_field_name, symbolic_name):
        value = getattr(self.model,"get_%s_code" % descr_field_name)(symbolic_name)
        return self.extra(where=["(%s != %d)" % (descr_field_name, value)])

    def active(self):
        """nur aktive Objekte holen.

        Wenn ein Objekt ein Attribut 'ende' hat, wird dieses herangezogen:
        - ende is NULL: ist aktiv
        - ende > epoch: ist aktiv

        Hat ein Objekt kein 'ende', wird die Gültigkeit des zugehörigen Kundeneintrag (laut Attribut 'kunde')
        genommen.
        """
        qn = connection.ops.quote_name
        opts = self.model._meta
        try:
            dummy = opts.get_field("ende")
        except fields.FieldDoesNotExist:
            field = opts.get_field("kunde")
            rel_table = qn(field.rel.to._meta.db_table)
            return self.extra(tables = [rel_table],
                                where = ["(%(kunde_table)s.id=%(table)s.%(field)s and "
                                         "(%(kunde_table)s.ende is null or %(kunde_table)s.ende >= UNIX_TIMESTAMP(NOW())))"
                                         % { "table": qn(opts.db_table),
                                             "field": qn(field.column),
                                             "kunde_table": rel_table }])
        else:
            return self.extra(where = ["(%(table)s.ende is null or %(table)s.ende >= UNIX_TIMESTAMP(NOW()))"
                        % { "table": (qn(self.model._meta.db_table)) }])

    def exists(self):
        return self[:1].count()>=1

class EmptyQuerySet(models.query.EmptyQuerySet):
    
    def active(self):
        return self

    def exists(self):
        return False


class PopManager(models.manager.Manager):
    """This manager uses the Noris version of QuerySet
    """
    def get_query_set(self):
        return PopQuerySet(self.model)
    def has_any_bit(self, descr_field_name, bitmask):
        return self.get_query_set().has_any_bit(descr_field_name, bitmask)
    def has_flag(self, descr_field_name, flagname):
        return self.get_query_set().has_flag(descr_field_name, flagname)
    def in_group(self, descr_field_name, groupname):
        return self.get_query_set().in_group(descr_field_name, groupname)
    def descr_eq(self, descr_field_name, symbolic_name):
        return self.get_query_set().descr_eq(descr_field_name, symbolic_name)
    def descr_ne(self, descr_field_name, symbolic_name):
        return self.get_query_set().descr_ne(descr_field_name, symbolic_name)
    def all_descr(self, descr_field_name):
        """returns all descriptors possible for the given field_list
        """
        descr_model = getattr(self.model,"get_%s_descr_model" % descr_field_name)()
        descr_typ = getattr(self.model,"get_%s_descr_type" % descr_field_name)()
        return descr_model.objects.filter(typ=descr_typ)
    def active(self):
        return self.get_query_set().active()

# the following "magic stuff" replaces the normal Django default manager in all models
# with NorisManager. Just copied from the original manager module ...

def _ensure_pop_default_manager(sender):
    cls = sender
    if not hasattr(cls, '_default_manager'):
        # Create the default manager, if needed.
        if hasattr(cls, 'objects'):
            raise ValueError, "Model %s must specify a custom Manager, because it has a field named 'objects'" % cls.__name__
        cls.add_to_class('objects', PopManager())

# the default manager for a class gets set by the dispatcher mechanismn.
# These two lines rewire the connections so that the PopManager
# replaces the default django manager

dispatcher.disconnect(models.manager.ensure_default_manager, signal=signals.class_prepared)
dispatcher.connect(_ensure_pop_default_manager, signal=signals.class_prepared)


#class StructuredEmailField(fields.TextField):
    #"""Behandelt eine Email-Adresse in drei Feldern localpart, subdomain, domain.
    #In der Datenbank wird das ganze als ein Char-Feld behandelt.
    #"""
    #from formfields import LocalpartField, SubdomainField, FQDNField
    #from utils import EmailAddress
    #def set_allowed_domains(self, allowed_domains, only=True):
        #self.allowed_domains = allowed_domains
        #self.only = only

    #def get_db_prep_save(self, value):
        #return fields.Field.get_db_prep_save(self, str(value))

    #def get_db_prep_lookup(self, lookup_type, value):
        #if lookup_type == 'range':
            #value = [str(v) for v in value]
        #else:
            #value = str(value)
        #return super(StructuredEmailField, self).get_db_prep_lookup(lookup_type, value)

    #def get_manipulator_field_objs(self):#
        #return [LocalpartField, SubdomainField, FQDNField]

    #def get_manipulator_field_names(self, name_prefix):
        #return [name_prefix + self.name + '_localpart', name_prefix + self.name + '_subdomain', name_prefix + self.name + '_domain']

    #def get_manipulator_new_data(self, new_data, rel=False):
        #"""
        #Given the full new_data dictionary (from the manipulator), returns this
        #field's data.
        #"""

        #localpart_field, subdomain_field, domain_field = self.get_manipulator_field_names('')
        #if rel:
            #localpart, subdomain, domain = [new_data.get(field,[None])[0] for field in self.get_manipulator_field_names('')]
        #else:
            #localpart, subdomain, domain = [new_data.get(field) for field in self.get_manipulator_field_names('')]
        #return EmailAddress((localpart, subdomain, domain))

    #def flatten_data(self,follow, obj = None):
        #val = self._get_val_from_obj(obj)
        #if not val.is_checked:
            #val.check_domain(self.allowed_domains)
        #localpart, subdomain, domain = val.localpart, val.subdomain, val.domainkunde.domain
        #localpart_field, subdomain_field, domain_field = self.get_manipulator_field_names('')
        #return {localpart_field: localpart, subdomain_field: subdomain, domain_field: domain }

    #def prepare_field_objs_and_params(self, manipulator, name_prefix):
        #params = {'validator_list': self.validator_list[:]}
        #if self.maxlength and not self.choices: # Don't give SelectFields a maxlength parameter.
            #params['maxlength'] = self.max_length

        #if self.choices:
            #if self.radio_admin:
                #field_objs = [forms.RadioSelectField]
                #params['ul_class'] = get_ul_class(self.radio_admin)
            #else:
                #field_objs = [forms.SelectField]

            #params['choices'] = self.get_choices_default()
        #else:
            #field_objs = self.get_manipulator_field_objs()
        #return (field_objs, params)