# -*- encoding: utf-8 -*-
import re
from itertools import chain, islice
from django.http import HttpResponse
from django.conf import settings
from django.utils import simplejson
from django.utils.datastructures import MultiValueDict
from django.utils.html import escape
from django.db.models.query import Q
from django.db.models.loading import get_model
from kundebunt.popkern import models
from kundebunt.popkern.utils import glob_to_like, simple_glob_filter, merge_sorted, unique, identity

__all__ = ["KundenDataSource", "DomainDataSource", "WildcardDomainDataSource",
           "PersonDataSource", "RackDataSource", "RZDataSource", "HardwareDataSource",
           "IpkundeDataSource", "LaenderDataSource", "DescriptorDataSource", "WartungsvertragDataSource"]

def _escapeNotNone(s):
    if s == None:
        return s
    else:
        return escape(s)

class QuerySetDataSource(object):
    """Derived classes must define:
    - attribute `att_names`: [ attribute_name, ... ]
    - method _query_set()
    - method url()
    - optional method additional_data() -> [ (attval1, attval2), ...]
    - optional class method from_request(request, **kwargs) -> instance
    """

    def get_choices(self):
        if len(self.att_names) == 1:
            return list(chain(self.query_set().valuelist(self.att_names[0], self.att_names[0]),
                              ((x,x) for (x,) in self.additional_data())))
        else:
            return list(chain(self.query_set().valuelist(*self.att_names),
                              self.additional_data()))

    def __contains__(self, s):
        return (s in (v[0] for v in self.additional_data())
                or self.query_set().filter(**{"%s__exact" % self.att_names[0]: s}).exists())

    def __len__(self):
        return self.query_set().count() + len(self.additional_data())

    def __str__(self):
        return str([x[0] for x in (self.get_choices() + self.additional_data())])

    def __iter__(self):
        return chain(self.query_set(), self.additional_data)

    def additional_data(self):
        return []

    def query_set(self, sort_column=None):
        try:
            if self._sort_column==sort_column:
                return self._query_set_cache
        except AttributeError:
            pass
        self._sort_column = sort_column
        self._query_set_cache = self._query_set(sort_column)
        return self._query_set_cache

    def get_values(self, qset):
        """Returns the list of value dictionaries during serving"""
        return qset.values(* self.att_names)

    @classmethod
    def serve_json(cls, request, **kwargs):
        data_source = cls.from_request(request, **kwargs)
        prefix = request.GET.get('query', '')
        qset = data_source.query_set().filter(
            **{"%s__istartswith" % data_source.att_names[0]: prefix}
            ).distinct()
        additional_hits = (dict(zip(data_source.att_names, row))
                           for row in data_source.additional_data()
                           if row[0].startswith(prefix))
        content = simplejson.dumps({'data':list(chain(data_source.get_values(qset), additional_hits))}, ensure_ascii=False)
        return HttpResponse(
            content=content,
            mimetype="application/json; charset=utf-8") # rfc 4627

    # Falls Zweifel über die Sicherheit des generierten Codes aufkommen:
    # http://robubu.com/?p=24  oder  http://kid666.com/blog/2006/12/23/security-ajax-json-satisfaction/
    @classmethod
    def serve_json_page(cls, request, **kwargs):
        sort_column = request.REQUEST.get("sortColumn", None)
        direction = request.REQUEST.get("sortDir", None)
        page = int(request.REQUEST.get("page", 1))
        page_size = int(request.REQUEST.get("pageSize", 0))
        first = (page-1) * page_size
        data_source = cls.from_request(request, **kwargs)
        if sort_column:
            if direction == "DESC":
                sort_column = "-" + data_source.att_names[int(sort_column)]
            else:
                sort_column = data_source.att_names[int(sort_column)]
        qset = data_source.query_set(sort_column)
        additional_hits = (dict(zip(data_source.att_names, row))
                           for row in data_source.additional_data())
        values = chain(data_source.get_values(qset[first:first+page_size]), additional_hits)
        esc_values = (dict((k,_escapeNotNone(v)) for k,v in x.iteritems()) for x in values)
        result = {'data':list(islice(esc_values, page_size))}
        if page==1:
            result['total_count'] = qset.count()
        return HttpResponse(
            content=simplejson.dumps(result, ensure_ascii=False),
            mimetype="application/json; charset=utf-8") # rfc 4627

    @classmethod
    def serve_csv(cls, request, **kwargs):
        data_source = cls.from_request(request, **kwargs)
        prefix = request.GET.get('query', '')
        qset_hits = data_source.query_set().filter(
            **{"%s__istartswith" % data_source.att_names[0]: prefix}
            ).valuelist(*data_source.att_names)
        if len(data_source.att_names)==1:
            qset_hits = ((x,) for x in qset_hits)
        additional_hits = (row for row in data_source.additional_data() if row[0].startswith(prefix))
        return HttpResponse(
            content=["%s\n" % '\t'.join(hit)
                     for hit in chain(qset_hits, additional_hits)],
            mimetype='text/plain')

    @classmethod
    def from_request(cls, request, **kwargs):
        return cls(**kwargs)

    #@classmethod
    #def serve_json(cls, request, query):
        #pass



class MultiQuerySetDataSource(object):
    """Derived classes must define:
    - attribute `att_names`: [ attribute_name, ... ]
    - method _query_set()
    - method url()
    - optional method additional_data() -> [ (attval1, attval2), ...]
    - optional class method from_request(request, **kwargs) -> instance
    """

    def get_choices(self):
        if len(self.att_names) == 1:
            return list(chain(chain(* (qs.valuelist(att_name, att_name) for (att_name, qs) in self._canonical_query_info())),
                              ((x,x) for (x,) in self.additional_data())))
        else:
            return list(chain(chain(* (qs.valuelist(* att_names) for (att_names, qs) in self._canonical_query_info())),
                              self.additional_data()))

    def __contains__(self, s):
        return (s in (v[0] for v in self.additional_data())
                or any(qs.filter(**{"%s__exact" % att_name: s}).exists()) for (att_name, qs) in self.query_info())

    def __len__(self):
        return sum(qs.count() for qs in self.query_sets()) + len(self.additional_data())

    def __str__(self):
        return str([x[0] for x in (self.get_choices() + self.additional_data())])

    #def __iter__(self):
        #return chain(chain(* self.query_sets()), self.additional_data)

    def additional_data(self):
        return []

    def query_info(self):
        try:
            return self._cache
        except AttributeError:
            self._cache = self._query_info()
            return self._cache

    def _canonical_query_info(self):
        return self.query_info()

    def query_sets(self, sort_column=None):
        return [qs for (attname, qs) in self.query_info()]

    def hits(self, prefix):
        bla = self.att_names
        by_qset = unique(merge_sorted([({self.att_names[0]: v}
                                        for v in qs.filter(**{"%s__istartswith" % att_name: prefix}).valuelist(att_name))
                                       for (att_name, qs) in self.query_info()]))
        additional_hits = ({self.att_names[0]: row[0]}
                           for row in self.additional_data()
                           if row[0].startswith(prefix))
        return chain(by_qset, additional_hits)

    @classmethod
    def serve_json(cls, request, **kwargs):
        data_source = cls.from_request(request, **kwargs)
        prefix = request.GET.get('query', '')
        content = simplejson.dumps({'data': list(data_source.hits(prefix))}, ensure_ascii=False)
        return HttpResponse(
            content=content,
            mimetype="application/json; charset=utf-8") # rfc 4627

    ## Falls Zweifel über die Sicherheit des generierten Codes aufkommen:
    ## http://robubu.com/?p=24  oder  http://kid666.com/blog/2006/12/23/security-ajax-json-satisfaction/
    #@classmethod
    #def serve_json_page(cls, request, **kwargs):
        #sort_column = request.REQUEST.get("sortColumn", None)
        #direction = request.REQUEST.get("sortDir", None)
        #page = int(request.REQUEST.get("page", 1))
        #page_size = int(request.REQUEST.get("pageSize", 0))
        #first = (page-1) * page_size
        #data_source = cls.from_request(request, **kwargs)
        #if sort_column:
            #if direction == "DESC":
                #sort_column = "-" + data_source.att_names[int(sort_column)]
            #else:
                #sort_column = data_source.att_names[int(sort_column)]
        #qsets = data_source.query_set(sort_column)
        #additional_hits = (dict(zip(data_source.att_names, row))
                           #for row in data_source.additional_data())
        #values = chain(data_source.get_values(qset[first:first+page_size]), additional_hits)
        #esc_values = (dict((k,_escapeNotNone(v)) for k,v in x.iteritems()) for x in values)
        #result = {'data':list(islice(esc_values, page_size))}
        #if page==1:
            #result['total_count'] = sum(qs.count() for qs in qsets)
        #return HttpResponse(
            #content=simplejson.dumps(result, ensure_ascii=False),
            #mimetype="application/json; charset=utf-8") # rfc 4627

    #@classmethod
    #def serve_csv(cls, request, **kwargs):
        #data_source = cls.from_request(request, **kwargs)
        #prefix = request.GET.get('query', '')
        #qset_hits = chain *(qs.filter(**{"%s__istartswith" % data_source.att_names[0]: prefix}).valuelist(*data_source.att_names)
                            #for qs in data_source.query_set())
        #if len(data_source.att_names)==1:
            #qset_hits = ((x,) for x in qset_hits)
        #additional_hits = (row for row in data_source.additional_data() if row[0].startswith(prefix))
        #return HttpResponse(
            #content=["%s\n" % '\t'.join(hit)
                     #for hit in chain(qset_hits, additional_hits)],
            #mimetype='text/plain')

    @classmethod
    def from_request(cls, request, **kwargs):
        return cls(**kwargs)

    #@classmethod
    #def serve_json(cls, request, query):
        #pass



class ArrayDataSource(object):
    """Derived classes must define:
    - attribute data: [ (row1_col1, row1_col2), ... ]
    - staticmethod from_request(request) -> Instance
    - staticmethod is_request_permitted(request)
    """
    def get_choices(self):
        data = self.data
        if len(data[0]) == 1:
            return [(x,x) for (x,) in data]
        else:
            return self.data

    def __contains__(self, s):
        return str(s) in (str(x[0]) for x in self.data)

    def __len__(self):
        return len(self.data())

    def __str__(self):
        return str([x[0] for x in self.data])

    def __iter__(self):
        return iter(self.data)

    @classmethod
    def serve_csv(cls, request, **kwargs):
        data_source = cls.from_request(request, **kwargs)
        prefix = request.GET.get('prefix', '')
        return HttpResponse(
            content=("%s\n" % '\t'.join(x) for x in data_source.data if x[0].startswith(prefix)),
            mimetype='text/plain')

    @classmethod
    def from_request(cls, request, **kwargs):
        return cls(**kwargs)



class DomainDataSource(QuerySetDataSource):
    att_names = ['domain']

    def __init__(self, person, kunde_id=None):
        self.person = person
        self.kunde_id = kunde_id

    def url(self):
        base = settings.POPKERN_ROOTURL + '/data/domains/'
        if self.kunde_id != None:
            return "%s%d/" % (base, self.kunde_id)
        else:
            return base

    def _query_set(self, sort_column):
        if self.kunde_id != None and self.person.is_staff():
            return models.Domainkunde.objects.active().filter(kunde=self.kunde_id)
        else:
            return self.person.managed_domainkunden()

    @classmethod
    def from_request(cls, request, kunde_id=None):
        return cls(request.person, kunde_id and int(kunde_id))

class WildcardDomainDataSource(DomainDataSource):
    def url(self):
        base = settings.POPKERN_ROOTURL + '/data/wildcard-domains/'
        if self.kunde_id != None:
            return "%s%d/" % (base, self.kunde_id)
        else:
            return base

    def additional_data(self):
        return [('*',)]


class KundenDataSource(MultiQuerySetDataSource):
    att_names = ['name']

    def __init__(self, person):
        self.person = person

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/kunden/'

    def _query_info(self):
        if not self.person.is_staff():
            return [("name", self.person.managed_kunden())]
        else:
            qs = models.Kunde.objects.active()
            return [
                    ("name",   models.Kunde.objects.active().order_by("name")),
                    ("name",   models.Uucpkunde.objects.active().order_by("name")),
                    #("domain", models.Domainkunde.objects.active().order_by("domain")),
                    #("user",   models.Person.objects.active().filter(user__isnull=False).order_by("user")),
                    #("email",  models.Person.objects.active().filter(email__isnull=False).order_by("email")),
                    #("handle", models.Nic.objects.order_by("handle")),
                   ]

    @classmethod
    def from_request(cls, request, **kwargs):
        return cls(request.person)


class PersonDataSource(MultiQuerySetDataSource):
    att_names = ['user']

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/personen/'

    def _query_info(self):
        personen = models.Person.objects
        return [
                ("user",        personen.filter(user__isnull=False).order_by("user")),
                ("suchbegriff", personen.filter(suchbegriff__isnull=False).order_by("suchbegriff")),
                #("handle",      models.Nic.objects.order_by("handle")),
                #("name",        personen.filter(name__isnull=False).order_by("name")),
                ("email",       personen.filter(email__isnull=False).order_by("email")),
                ("fon",         personen.filter(fon__isnull=False).order_by("fon")),
                ("fax",         personen.filter(fax__isnull=False).order_by("fax")),
                ("pager",       personen.filter(pager__isnull=False).order_by("pager")),
               ]

    def _canonical_query_info(self):
        return self.query_info()[0:1]
    

class RackDataSource(QuerySetDataSource):
    att_names = ['name']

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/racks/'

    def _query_set(self, sort_column):
        return models.Rack.objects.all()


class RZDataSource(QuerySetDataSource):
    att_names = ['name']

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/rz/'

    def _query_set(self, sort_column):
        return models.RZ.objects.all()

class HardwareDataSource(QuerySetDataSource):
    att_names = ['id', 'hardware_id', 'ivnr', 'name', 'ip', 'kunde', 'standort']

    def __init__(self, editor_person, filter_data):
        self.editor_person = editor_person
        self.data = filter_data
        super(HardwareDataSource, self).__init__(self)

    @classmethod
    def from_request(cls, request):
        from kundebunt.hardware.forms import HWSearchForm
        manipulator = HWSearchForm(request.person)
        data = request.POST.copy()
        data.update(request.GET)
        manipulator.do_html2python(data)
        return cls(request.person, data)

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/hardware/'

    def _query_set(self, sort_column):
        data = self.data
        if self.editor_person.is_staff():
            qset = models.Hardware.objects.all()
        else:
            qset = models.Hardware.objects.filter(kunde__exact=self.editor_person.kunde_id)
        qset = simple_glob_filter(qset, data,
                ('name', 'hardware_id', 'ivnr', 'seriennr', 'typ', 'klasse', 'lieferant',
                 'status', 'info', 'kunde', 'standort', 'verantwortlich', 'eigentuemer',
                 'ip'))
        #data.get("ip") not in ignored_values:
            #if data["ip"] not in null_values:
                #qset = qset.filter(  models.Ipkunde.q_ip_glob(data["ip"]).join("ip")
                                #| Q(ip__name__like=glob_to_like(data["ip"])))
            #else:
                #qset = qset.filter(Q(ip__isnull=True))
        if data.get("aktiv"):
            if data["aktiv"] == "nur_aktiv":
                qset = qset.active()
        if data.get("rack", "*") not in ("*", "", None):
            if data["rack"] != '-':
                qset = qset.filter(rack__name__like=glob_to_like(data["rack"]))
            else:
                qset = qset.filter(rack__isnull=True)
        if data.get("rz", "*") != "*":
            if data["rz"] not in (None, "", "-"):
                qset = qset.filter(rack__rz__name__iexact=data["rz"])
            else:
                qset = qset.filter(rack__isnull=True)

        # sorting
        if sort_column:
            if sort_column.endswith('kunde'):
                qset = qset.order_by("%s__name" % sort_column, "id")
            elif sort_column == '-ip':
                qset = qset.order_by('-+__ip__ip6',"-id")
            elif sort_column == 'ip':
                qset = qset.order_by('+__ip__ip6', "id")
            elif sort_column == '-standort':
                qset = qset.order_by('-+__enthalten_in__hardware_id', '-enthalten_in', '-+__rack__rz__name', '-+__rack__name', '-+__standort__user', '-standort', '-unterste_he', '-id')
            elif sort_column == 'standort':
                qset = qset.order_by('+__enthalten_in__hardware_id', 'enthalten_in', '+__rack__rz__name', '+__rack__name', '+__standort__user', 'standort', 'unterste_he', 'id')
            else:
                qset = qset.order_by(sort_column, "id")
        return qset.distinct()

    def get_values(self, qset):
        return (dict((
            ('id', rec.id),
            ('hardware_id', rec.hardware_id or ''),
            ('name', rec.name or ''),
            ('standort',  rec.standort_display()),
            ('ip', rec.ip_id and rec.ip.ipaddr_display() or ''),
            ('kunde', rec.kunde.name),
            ('ivnr', rec.ivnr or ''),
            )) for rec in qset)

class IpkundeDataSource(QuerySetDataSource):
    att_names = ['name']

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/ipkunde/'

    def _query_set(self, sort_column):
        return models.Ipkunde.objects.active().filter(name__isnull=False)

class LaenderDataSource(QuerySetDataSource):
    att_names = ["name"]

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/land/'

    def _query_set(self, sort_column):
        return models.Land.objects.all()

class DescriptorDataSource(QuerySetDataSource):
    att_names = ["bla"]

    def __init__(self, model, fieldname):
        self.model = model
        self.fieldname = fieldname

    def url(self):
        return "%s/data/descr/%s/%s/" % (settings.POPKERN_ROOTURL, self.model._meta.object_name, self.fieldname)

    def _query_set(self, sort_column):
        return self.model.objects.all_descr(self.fieldname).order_by("bla")

    @classmethod
    def from_request(cls, request, modelname, fieldname):
        return cls(get_model('popkern', modelname, False), fieldname)

class WartungsvertragDataSource(QuerySetDataSource):
    att_names = ['name']

    def url(self):
        return settings.POPKERN_ROOTURL + '/data/wartungsvertrag/'

    def _query_set(self, sort_column):
        return models.Wartungsvertrag.objects.all()
