import dns.resolver, threading, time
from django.conf import settings

class AsyncDNSResolver(object):
    def __init__(self):
        self.threads = {}
        self.answers = {}  # answers is modified within threads; apply locking!
        self.lock = threading.Lock()

    def submit(self, domain_string, rdatatype):
        query = (domain_string, rdatatype)
        if not query in self.threads and not query in self.answers:
            thread = threading.Thread(target=_run_thread, args=(domain_string, rdatatype, self.answers, self.lock))
            thread.setDaemon(True)
            thread.start()
            self.final_submit_time = time.time()
            self.threads[query] = thread


    def resolve(self, domain_string, rdatatype):
        self.submit(domain_string, rdatatype)
        query = (domain_string, rdatatype)
        if self.threads[query].isAlive():
            self.threads[query].join(self._compute_timeout())
        self.lock.acquire()
        try:
            answer = self.answers.get(query)
        finally:
            self.lock.release()
        return answer

    def _compute_timeout(self):
        return max(settings.DNS_LIFETIME - (time.time() - self.final_submit_time) , 0)


def _run_thread(domain_string, rdatatype, answers, lock):
    resolver = dns.resolver.Resolver()
    resolver.timeout = settings.DNS_TIMEOUT
    resolver.lifetime = settings.DNS_LIFETIME
    try:
        answer = resolver.query(domain_string, rdatatype)
    except (dns.exception.Timeout, dns.resolver.NXDOMAIN, dns.resolver.NoAnswer, dns.name.EmptyLabel):
        answer = None
    lock.acquire()
    try:
        answers[(domain_string, rdatatype)] = answer
    finally:
        lock.release()

