# -*- encoding: UTF-8 -*-

from datetime import datetime
from itertools import chain, groupby
import logging
import ldap
from django.conf import settings
from django.core.urlresolvers import reverse
from django.db import connection
from django.db.models import Q
from django.db.models.query import QuerySet
from django.http import HttpResponseNotFound, HttpResponse, HttpResponseForbidden
from django.utils import simplejson
from django.shortcuts import render_to_response
from django.template import RequestContext

from kundebunt.acct.forms import monatsanfang
from kundebunt.contrib.csrf.middleware import csrf_exempt
from kundebunt.popkern import models, updatelog
from kundebunt.popkern.utils import is_active
from kundebunt.kunde_auth.views import login as auth_login, logout as auth_logout
from kundebunt.kunde_auth.decorators import person_can_log_in
from kundebunt.popkern.utils import glob_to_like

customer_user_attributes = ["id", "user", "name", "email", "kunde", "valid"]

class BadAccess(Exception):
    def __init__(self, response):
        self.response = response

def _check_otrs_credentials(request):
    """überprüft passwort und IP der anderen Seite. Wenn der Zugriff nicht
    erlaubt ist, wird die Ausnahme BadAccess ausgelöst. Andernfalls werden
    die restlichen Parameter zurückgegeben.
    """
    if not request.META["REMOTE_ADDR"] in settings.OTRS_CLIENT_IPS:
        raise BadAccess(HttpResponseForbidden("Access denied."))
    params = request.POST.copy()
    password = params.pop('password', [''])[0]
    if password != settings.OTRS_PASSWORD:
        raise BadAccess(HttpResponseForbidden("Wrong or missing authentication."))
    return params

def _email_suche(v, valid):
    manager = models.Person.objects
    if valid:
        manager = manager.active()
    personen = manager.filter(email=v)
    if not personen:
        personen = manager.filter(mailassoc__email=v)
        if not personen:
            at_sep_pos = v.find(u"@")
            if at_sep_pos != -1:
                labels = v[at_sep_pos + 1:].split(u".")
                if labels in ([u'noris', u'de'], [u'noris', u'net']):
                    # Interne Mailaccounts per Benutzernamen (localpart) finden
                    personen = manager.filter(kunde=1, user=v[:at_sep_pos])
                if not personen:
                    while labels:
                        domains = models.Domainkunde.objects.active().filter(domain=u".".join(labels))
                        if domains:
                            break
                        del labels[0]
                    if domains:
                        assert(len(domains)==1)
                        kunde = domains[0].kunde
                        personen = [models.Person(kunde=kunde, email=v, id=v)]
    if len(personen)>1:
        personen = []
    return personen


_ldap_connection = None

def _otrs_daten(person):
    global _ldap_connection
    is_staff = person.user and person.is_staff()

    # einfache Attribute
    data = dict(
            id=person.id, user=person.user, email=person.email,
            pager = person.pager, fax=person.fax,
            kunde=person.kunde.name, kprio=person.kunde.get_kprio_display(),
            ticket_kunden=';'.join(person.managed_kunden(['rtweb']).valuelist("name")),
            valid=int(is_active(person.kunde.beginn, person.kunde.ende)))
            
    # namen
    (anrede, vorname, nachname) = person.zerlege_name(person.name, keep_hyphen=True)
    if anrede and anrede[0]:
        name = person.name[len(anrede[0]):]
    else:
        name = person.name
    anrede = data['salutation'] = ' '.join(anrede)
    data['first_name'] = ' '.join(vorname)
    vorname = ' '.join(vorname[:1])
    nachname = data['name'] = ' '.join(nachname)
    data['display_name'] = name
    if is_staff and vorname:
        if person.user == 'flo':
            data['salutation_de'] = u'Mahlzeit Flo'
        else:
            data['salutation_de'] = u'Moin %s' % vorname
    elif anrede.startswith(u'Herr'):
        data['salutation_de'] = u'Sehr geehrter %s %s' % (anrede, nachname)
    elif anrede.startswith(u'Frau'):
        data['salutation_de'] = u'Sehr geehrte %s %s' % (anrede, nachname)
    else:
        data['salutation_de'] = u'Sehr geehrte Damen und Herren'

    # fon
    if is_staff and settings.LDAP_URL is not None:
        for i in (1, 2):
            try:
                if not _ldap_connection:
                    _ldap_connection = ldap.ldapobject.SmartLDAPObject(settings.LDAP_URL, retry_delay=2, retry_max=1 )
                found = _ldap_connection.search_st(
                        settings.LDAP_STAFF_BASE,
                        ldap.SCOPE_ONELEVEL,
                        '(&(uid=%s)(objectClass=person))' % person.user,
                        attrlist=["telephoneNumber"],
                        timeout=5)
                break
            except ldap.LDAPError:
                    _ldap_connection = None
                    # retry
        else:
            logging.warn('ldap lookup for telephoneNumber failed, uid=%s.' % person.user)
            found = []
        data['fon'] = ", ".join(", ".join(val.get('telephoneNumber',[])) for (dn,val) in found)
    if not data.get('fon'):
        data["fon"] = person.fon

    # Rollen / Dienste
    rollen = {}
    for km in sorted(person.kundemail_set.all(),
                     key=lambda km:km.dienst.bla):
        if not km.dienst_in_group('hide'):
            rollen.setdefault(km.kunde.name, []).append(km.dienst.bla)
    data['rollen'] = rollen

    return data

def acct(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    ticket_data = {}

    # Djangos ORM kann nicht benutzt werden, weil die Verknüpfung von
    # acct mit den anderen Tabellen über den zusammengesetzten FK (hash, seq)
    # geht

    sql = """
    SELECT rtb.ticket as ticket,
           k.name as kunden_name,
           acct.hash as hash,
           acct.seq as seq,
           acct.jjmm as jjmm,
           acct.tt as tt,
           dienst.name as dienst,
           acct.bytes as bytes,
           asso.info as info,
           asso.acctinfo as acctinfo
    FROM rt_billing rtb
    JOIN acct on acct.hash=rtb.hash and acct.seq=rtb.seq
    LEFT JOIN dienst on dienst.id = acct.dienst
    JOIN kunde k on k.id = acct.kunde
    LEFT JOIN acctassoc asso on asso.hash=acct.hash and asso.seq=acct.seq
    WHERE rtb.ticket in (%s)
    ORDER BY ticket, jjmm, tt
    """ % ",".join(str(int(t)) for t in params.getlist("ticket"))

    Acctassoc = models.Acctassoc
    cursor = connection.cursor()
    cursor.execute(sql)
    cols = [x[0] for x in cursor.description]
    results = groupby(
        (dict(zip(cols, row)) for row in cursor.fetchall()),
        key = lambda row: row["ticket"])
    now = datetime.now()
    jjmm_now = now.year * 100 + now.month
    for ticket, acct_list in results:
        ticket_data[int(ticket)] = dict(
            add_url=reverse('add-acct-without-customer', args=(ticket,)),
            acct=[
                dict(
                    date=dict(
                        day=acct["tt"],
                        month=acct["jjmm"] % 100,
                        year=acct["jjmm"] // 100),
                    worktime=acct["bytes"],
                    dienst=acct["dienst"],
                    additional_text=acct["info"],
                    std_text=acct["acctinfo"] and Acctassoc.get_acctinfo_descr(acct["acctinfo"]).infotext,
                    kunde=acct["kunden_name"],
                    editable=acct["jjmm"] >= jjmm_now,
                    edit_url=reverse('kundebunt.acct.views.edit_acct', args=(acct["hash"],acct["seq"])))
            for acct in acct_list])
    for ticket in params.getlist('ticket'):
        if int(ticket) not in ticket_data:
            ticket_data[int(ticket)] = dict(
                add_url=reverse('add-acct-without-customer', args=(ticket,)),
                acct=[])
    content=simplejson.dumps(
        {'data': {'ticket': ticket_data}},
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

def acct_change_kunde(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    try:
        ticket_numbers = [int(s) for s in params.getlist('ticket')]
        kunde = models.Kunde.objects.get(name=params['kunde'])
        person = models.Person.objects.get(user=params['person'])
    except ValueError:
        return HttpResponseForbidden('ticket numbers must be integers')
    except KeyError:
        return HttpResponseForbidden('missing parameters')
    except models.Kunde.DoesNotExist:
        return HttpResponseForbidden('wrong kunde')
    except models.Person.DoesNotExist:
        return HttpResponseForbidden('wrong person')
    updatelog.set_person(person)
    _monatsanfang = monatsanfang()
    cursor = connection.cursor()
    # Gibt es alte unveränderliche Einträge mit einem abweichenden Kunden?
    cursor.execute('SELECT 1 FROM rt_billing r '
                   'JOIN acct on acct.hash=r.hash and acct.seq=r.seq '
                   'WHERE r.ticket in (%s) and acct.jjmm < %d and kunde != %d '
                   'LIMIT 1'
                   % (",".join(str(n) for n in ticket_numbers),
                      _monatsanfang.year*100 + _monatsanfang.month,
                      kunde.id))
    if cursor.rowcount != 0:
        return HttpResponseForbidden('Forbidden: Cannot change old entries')
    for (hash, seq) in (models.RtBilling.objects
            .filter(ticket__in=ticket_numbers)
            .valuelist('hash','seq')):
        acct = models.Acct.objects.get(hash=hash, seq=seq,
            jjmm__gte=_monatsanfang.year*100 + _monatsanfang.month)
        acct.kunde = kunde
        print "%d-%d" % (hash, seq)
        acct.save()
    content=simplejson.dumps(
        {'data': dict(success=True)},
        ensure_ascii=True)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627


def customer_user(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    try:
        valid = int(params.pop('valid', ['0'])[0])
    except ValueError:
        return HttpResponseNotFound("valid must be an integer")
    if len(params) > 1:
        return HttpResponseNotFound("Too many query parameters, only password, valid and one other is allowed.")
    manager = models.Person.objects
    if valid:
        manager = manager.active()
    if len(params) == 0:
        personen = manager.all()
    else:
        k,v = params.items()[0]
        v=v.strip()
        if k=="search":
            if v==u"*":
                # wir liefern hier nur die Benutzer, die in den Admin-Listen zur Rechteverwaltung
                # auftauchen sollen.
                ids = models.Kundemail.objects.descr_eq('dienst','rtweb').valuelist('person')
                personen = manager.in_bulk(list(ids)).values()
            else:
                personen = (
                        [k.hauptperson for k in models.Kunde.objects.default_query(v, queryset_required=False)
                                       if k.hauptperson is not None]
                        + [p for p in models.Person.objects.default_query(v, queryset_required=False)
                             if not valid or is_active(p.kunde.beginn, p.kunde.ende)]
                        )
                if not personen and u"@" in v:
                    personen = _email_suche(v, valid)
        elif k=='id':
            if v.isdigit():
                personen = manager.filter(id=v)
            elif "@" in v:
                personen = _email_suche(v, valid)
            else:
                personen = manager.filter(user=v)
        elif k=='kunde':
            personen = manager.filter(kunde__name=v)
        elif k=="email":
            personen = _email_suche(v, valid)
        else:
            return HttpResponseNotFound("Got unknown search criterium `%s`" % k)
    if isinstance(personen, QuerySet):
        personen = personen.order_by("kunde", "name", "id").select_related(follow_fields=(('kunde',)))
    content=simplejson.dumps({'data': [_otrs_daten(p) for p in personen]},
                             ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627


def customer_id(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    if len(params) > 1:
        return HttpResponseNotFound("Too many query parameters, only one is allowed.")
    manager = models.Kunde.objects
    if len(params) == 0:
        kunden = manager.all()
    else:
        k, v = params.items()[0]
        v=v.strip()
        if k == "search":
            if v == u"*":
                kunden = manager.all()
            else:
                kunden = manager.default_query(v)
    content = simplejson.dumps(
        { 'data': [{'name': k.name,
                    'betriebsteam': k.betriebsteam(),
                    'kprio': k.get_kprio_display()} for k in kunden]},
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

def open_troubletickets(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    if len(params) > 1:
        return HttpResponseNotFound("Too many query parameters, only one is allowed.")
    ticket_id = params.get('ticket_id')
    if ticket_id is None:
        return HttpResponseNotFound("Need a query paramter for the ticket id.")
    content = simplejson.dumps(
        {'data':
            [   {'troubleticket_id': tt.id}
                for tt in models.TroubleTicket.objects.all().filter(rtticket=ticket_id)
            ]},
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

def confitems(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    ids = params.getlist('id')
    query = models.Confitem.objects.all()
    if ids:
        query = query.filter(id__in=ids)
    kunden = params.getlist('kunde')
    if kunden:
        kunden_ids = [[k.id for k in models.Kunde.objects.default_query(
                            s, queryset_required=False)]
                      for s in kunden]
        kunden_ids = reduce(list.__add__, kunden_ids, [])
        query = query.filter(kunde__id__in=kunden_ids)
    namen = params.getlist('name')
    if namen:
        query = query.filter(
            reduce(lambda q1,q2: q1 | q2,
                   (Q(name__like=glob_to_like(n)) for n in namen)))
    content = simplejson.dumps(
        {'data':
            [ { 'id': ci.id, 'name': ci.name, 'kunde': ci.kunde.name}
              for ci in query
            ]
        },
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

def _carrier_data(person):
    """Gibt die Carrier-Daten für leitung() zu einer person zurück.
    Wenn person NULL ist, wird immer noch ein dict zurückgegeben, allerdings
    steht dann überall NULL als Wert
    """
    if person is None:
        return dict(id=None, name=None)
    else:
        return dict(id=person.id, name=person.name)

def leitung(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    ids = params.getlist('id')
    kunden = params.getlist('kunde')
    query = models.Leitung.objects.all()
    if ids:
        query = query.filter(id__in=ids)
    if kunden:
        kunden_ids = [[k.id for k in models.Kunde.objects.default_query(
                            s, queryset_required=False)]
                      for s in kunden]
        kunden_ids = reduce(list.__add__, kunden_ids, [])
        query = query.filter(kunde__id__in=kunden_ids)
    carriers = params.getlist('carrier')
    if carriers:
        carrier_ids = [[p.id for p in models.Person.objects.default_query(
                            s, queryset_required=False)]
                      for s in carriers]
        carrier_ids = reduce(list.__add__, carrier_ids, [])
        query = query.filter(carrier__id__in=carrier_ids)
    namen = params.getlist('name')
    if namen:
        query = query.filter(
            reduce(lambda q1,q2: q1 | q2,
                   (Q(name__like=glob_to_like(n)) for n in namen)))
    content = simplejson.dumps(
        {'data':
            [ dict(id=l.id,
                   name=l.name,
                   name_kunde=l.infotext,
                   infotext=l.infotext,
                   name_carrier=l.name_carrier,
                   carrier=_carrier_data(l.carrier),
                   art=l.art and l.art.bla,
                   kunde_info=l.kunde.name)
              for l in query
            ]
        },
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

def _sum_to_int(decimal):
    if decimal is None:
        return 0
    return int(decimal)

def ticket_stunden(request):
    try:
        params = _check_otrs_credentials(request)
    except BadAccess, err:
        return err.response
    try:
        tickets = [int(s) for s in params.getlist('ticket')]
    except ValueError:
        return HttpResponseForbidden('ticket ids must be integers')
    if tickets:
        cursor = connection.cursor()
        cursor.execute(
            'SELECT SUM(stunden.zeit), SUM(stunden.zeit/stunden_art.faktor)*100 '
            'FROM   stunden, stunden_art '
            'WHERE  stunden.ticket IN (%s) '
            'AND stunden_art.id = stunden.art'
            % ','.join((str(t) for t in tickets)))
        row = [_sum_to_int(val) for val in cursor.fetchone()]
    else:
        row = (0,0)
    content = simplejson.dumps(
        {'data': {'simple_sum': row[0],
                  'weighted_sum': row[1]}},
        ensure_ascii=False)
    return HttpResponse(content, mimetype="application/json; charset=utf-8") # rfc 4627

@csrf_exempt
def check_otrs_access(request):
    """FastCGIAccessChecker - erlaubt den Zugriff, wenn die Person zur Session zu irgend einem
    Kunden eine rtweb-Assoziierung hat. Andernfalls wird Status 403 zurückgegeben.
    In letzterem Fall wird dann sinnvollerweise über ein zu konfigurierendes Apache-ErrorDocument
    die Loginseite aufgerufen.
    """
    person = request.person
    if person is not None and person.managed_kunden(['rtweb']).exists():
        response = HttpResponse(u"OK", content_type="text/plain")
        response['REMOTE_USER'] = person.user
        return response
    else:
        # Against specification, the apache forwards the content-length header
        # of response of the fastcgi access checker, so it should match the
        # final response (which is the login view, by ErrorDocument)
        response = login(request)
        response.status_code = 403
        return response

def login(request):
    person = request.person
    if person is not None and person_can_log_in(person) and not person.managed_kunden(['rtweb']).exists():
        # Der Kunde darf sich zwar einloggen, er hat jedoch keine Berechtigung zum Ansehen von Tickets
        response = render_to_response('otrs/no_rtweb_access.html',
            context_instance=RequestContext(request, {}));
        response.status_code = 403
        return response
    else:
        return auth_login(request, redirect_to=settings.OTRS_CUSTOMER_URL)

def logout(request):
    return auth_logout(request)
