# -*- encoding: utf-8 -*-

import re, datetime

from django import newforms
from django.newforms.util import ErrorList
from django.newforms.forms import DeclarativeFieldsMetaclass
from django.newforms.widgets import Widget, Select, TextInput
from django.utils.dates import MONTHS
from django.utils.encoding import force_unicode, smart_unicode
from django.utils.html import escape, conditional_escape
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy

__all__ = ['DivErrorList', 'BaseForm', 'Form', 'ValidationError', 'FrontCheckboxInput',
           'FrontBooleanField', 'DynamicMultipleChoiceField', 'SelectDateWidget',
           'validate_only_latin1_characters']

class DivErrorList(ErrorList):
    """Eine ErrorList, die Fehler als <div class="val_error" anzeigt.
    """
    def __unicode__(self):
        return self.as_div()

    def as_div(self):
        if not self: return u''
        return mark_safe(u'\n'.join(u'<div class="val_error">%s</div>' % force_unicode(e) for e in self))

class DivFormat(object):
    normal_row = u'<div class="formrow">%(errors)s<div class="inner">%(label)s%(field)s%(help_text)s</div></div>'
    error_row  = u'<div class="val_error">%s</div>'
    row_ender  = u'</div></div>'
    help_text_html = (u'<div class="field_help"><a href="#"><span class="htrigger">?'
                      u'</span><span class="htext">%s</span></a></div>')
    errors_on_separate_row = False
    
    @classmethod
    def render_form(cls, form):
        return form._html_output(
            normal_row = cls.normal_row,
            error_row = cls.error_row,
            row_ender = cls.row_ender,
            help_text_html = cls.help_text_html,
            errors_on_separate_row = cls.errors_on_separate_row)
    
class BoundField(newforms.forms.BoundField):
    def as_div(self):
        """render the field complete with error output etc. in  div
        """
        output = []
        format = self.form.format
        bf_errors = self.form.error_class([conditional_escape(error) for error in self.errors]) # Escape and cache in local variable.
        if format.errors_on_separate_row and bf_errors:
            output.append(format.error_row % force_unicode(bf_errors))
        if self.label:
            label = conditional_escape(force_unicode(self.label))
            # Only add the suffix if the label does not end in
            # punctuation.
            if self.form.label_suffix:
                if label[-1] not in ':?.!':
                    label += self.form.label_suffix
            label = self.label_tag(label) or ''
        else:
            label = ''
        if self.field.help_text:
            help_text = format.help_text_html % force_unicode(self.field.help_text)
        else:
            help_text = u''
        output.append(format.normal_row % {'errors': force_unicode(bf_errors), 'label': force_unicode(label), 'field': unicode(self), 'help_text': help_text})
        return mark_safe(u'\n'.join(output))

    
class BaseForm(newforms.BaseForm):
    """Leicht modifizierte Fassung von newforms.BaseForm:

    - Fehlermeldungen werden als divs angezeigt (DivErrorList)
    - unterstützt __contains__()
    - stellt has_errors() zur Kompatibilität mit oldforms zur Verfügung
    - rendern als eine Reihe von <div>s möglich
    """
    format = DivFormat
    
    def __init__(self, *args, **kwargs):
        if not 'error_class' in kwargs:
            kwargs['error_class'] = DivErrorList
        super(BaseForm, self).__init__(*args, **kwargs)

    def __contains__(self, name):
        return name in self.fields

    def __getitem__(self, name):
        "Returns our version of BoundField with the given name."
        try:
            field = self.fields[name]
        except KeyError:
            raise KeyError('Key %r not found in Form' % name)
        return BoundField(self, field, name)

    def as_div(self):
        return self.format.render_form(self)

    #def as_inner_div(self):
        #"""
            #<div class="inner">
            #<input  class="submit" type="submit" value="{% trans "Suchen" %}" />
            #<label for="id_cust_search">{% trans "Schnellsuche:" %} </label>
            #<input type="text" id="id_cust_search" name="search" class="vTextField" />
            #{% fieldhelp %}{% trans "Der eingegebene Text wird als Teilstring gesucht. Jokerzeichen wie '*' oder '?' sind hier nicht moeglich." %}{% endfieldhelp %}
            #<input type="hidden" id="id_quick" name="quick" />
            #</div>
        #"""
        #return self._html_output(
            #normal_row = u'%(errors)s%(label)s%(field)s%(help_text)s',
            #error_row = u'<div class="val_error">%s</div>',
            #row_ender = u'',
            #help_text_html = u'<div class="field_help"><a href="#"><span class="htrigger">?'
                             #u'</span><span class="htext">%s</span></a></div>',
            #errors_on_separate_row = False
            #)

    def has_errors(self):
        return self.is_bound and not self.is_valid()


class Form(BaseForm):
    "A collection of Fields, plus their associated data."
    # This is a separate class from BaseForm in order to abstract the way
    # self.fields is specified. This class (Form) is the one that does the
    # fancy metaclass stuff purely for the semantic sugar -- it allows one
    # to define a form using declarative syntax.
    # BaseForm itself has no way of designating self.fields.
    __metaclass__ = DeclarativeFieldsMetaclass



class ValidationError(newforms.ValidationError):
    """DivErrorList statt ErrorList benutzen
    """
    def __init__(self, message):
        """
        ValidationError can be passed any object that can be printed (usually
        a string) or a list of objects.
        """
        if isinstance(message, list):
            self.messages = DivErrorList([smart_unicode(msg) for msg in message])
        else:
            message = smart_unicode(message)
            self.messages = DivErrorList([message])

#class UnrenderedWidget(newforms.HiddenInput):
    #"""Ein Widget, das innerhalb der Form gar nicht auftaucht.
    #Es muss im Template explizit gerendered werden.
    #"""
    #def render(self, name, value, attrs=None):
        #return u''


class FrontCheckboxInput(newforms.CheckboxInput):
    def __init__(self, label, field, *args, **kwargs):
        super(FrontCheckboxInput, self).__init__(*args, **kwargs)
        self.label = label
        self.field = field

    def render(self, *args, **kwargs):
        if 'attrs' in kwargs:
            id = kwargs['attrs'].get('id')
        else:
            id = ''
        if id:
            for_id = u'for="%s"' % escape(force_unicode(id))
        else:
            for_id = u''
        return u'%s<label %s>%s</label>' % (
            super(FrontCheckboxInput, self).render(*args, **kwargs),
            for_id,
            escape(force_unicode(self.label)),
            )
            

class FrontBooleanField(newforms.BooleanField):
    widget = FrontCheckboxInput
    def __init__(self, label, *args, **kwargs):
        widget = kwargs.get('widget') or self.widget
        if isinstance(widget, type):
            widget = widget(label=label, field=self, attrs={'class': 'before_label'})
        kwargs['widget'] = widget
        kwargs['label'] = u''
        super(FrontBooleanField, self).__init__(*args, **kwargs)

        
def validate_only_latin1_characters(field_data, message=None):
    """Testet, ob ein Textfeld ausschließlich iso-8859-1-Zeichen enthält.

    >>> validate_only_latin1_characters(u'axzAXZ059!?=+#,&*/%.()[]_$§{}@;')
    """
    try:
        field_data.encode('iso-8859-1', 'strict')
    except UnicodeEncodeError:
        if not message:
            message = mark_safe(ugettext_lazy(
                    u"Einige Zeichen des Passworts liegen außerhalb des üblichen Zeichensatzes (iso-latin-1). "
                    u"Es kann nicht garantiert werden, dass der Zugriff mit allen Schnittstellen und Clients möglich ist. "
                    u'Deswegen sind in Passwörtern nur Buchstaben, Ziffern und die Zeichen "<tt>!?=+#,&*/%.()[]{}_$§@;</tt>" verwendbar.'
                    ))
            raise newforms.ValidationError(message)


class DynamicMultipleChoiceField(newforms.MultipleChoiceField):
    """Ein MultipleChoiceField, das unzulässige Werte einfach ignoriert.

    Das normale MultipleChoiceField lässt es nicht zu, dass der Benutzer
    Werte selektiert, die nicht in der Werteliste stehen. Das funktioniert
    aber nicht gut, wenn sich die Liste der zulässigen Werte in Abhängigkeit
    von anderen Feldern ändert. Hier kann dann das DynamicMultipleChoiceField
    eingesetzt werden.
    """
    def clean(self, value):
        """
        Validates that the input is a list or tuple.
        """
        if self.required and not value:
            raise ValidationError(self.error_messages['required'])
        elif not self.required and not value:
            return []
        if not isinstance(value, (list, tuple)):
            raise ValidationError(self.error_messages['invalid_list'])
        # Validate that each value in the value list is in self.choices.
        valid_values = set([smart_unicode(k) for k, v in self.choices])
        new_value = [smart_unicode(val) for val in value if val in valid_values]
        return new_value





RE_DATE = re.compile(r'(\d{4})-(\d\d?)-(\d\d?)$')

class SelectDateWidget(Widget):
    """
    A Widget that splits date input into three <select> boxes.

    This also serves as an example of a Widget that has more than one HTML
    element and hence implements value_from_datadict.
    """
    month_field = '%s_month'
    day_field = '%s_day'
    year_field = '%s_year'

    def __init__(self, attrs=None, years=None):
        # years is an optional list/tuple of years to use in the "year" select box.
        self.attrs = attrs or {}
        if years:
            self.years = years
        else:
            this_year = datetime.date.today().year
            self.years = range(this_year-9, this_year+1)

    def render(self, name, value, attrs=None):
        try:
            year_val, month_val, day_val = value.year, value.month, value.day
        except AttributeError:
            year_val = month_val = day_val = None
            if isinstance(value, basestring):
                match = RE_DATE.match(value)
                if match:
                    year_val, month_val, day_val = [int(v) for v in match.groups()]

        output = []

        if 'id' in self.attrs:
            id_ = self.attrs['id']
        else:
            id_ = 'id_%s' % name

        day_choices = [(i, i) for i in range(1, 32)]
        local_attrs = self.build_attrs(id=self.day_field % id_)
        select_html = TextInput(attrs={'maxlength': '2', 'size': '2', 'style': 'width: 2em'}).render(self.day_field % name, day_val, local_attrs)
        output.append(select_html)
        output.append(u'.')

        month_choices = MONTHS.items()
        month_choices.sort()
        local_attrs['id'] = self.month_field % id_
        select_html = Select(choices=month_choices).render(self.month_field % name, month_val, local_attrs)
        output.append(select_html)

        year_choices = [(i, i) for i in self.years]
        local_attrs['id'] = self.year_field % id_
        select_html = Select(choices=year_choices).render(self.year_field % name, year_val, local_attrs)
        output.append(select_html)

        return mark_safe(u'\n'.join(output))

    def id_for_label(self, id_):
        return '%s_month' % id_
    id_for_label = classmethod(id_for_label)

    def value_from_datadict(self, data, files, name):
        y, m, d = data.get(self.year_field % name), data.get(self.month_field % name), data.get(self.day_field % name)
        if y and m and d:
            return '%s-%s-%s' % (y, m, d)
        return data.get(name, None)

