# -*- encoding: utf-8 -*-

"""Dieses Modul enthält die Views zur Mailbox-Behandlung.

Der aktuelle Entwicklungstrend sieht folgende Vorgehensweise vor:

    - die view function (die letztlich aufgerufen wird, wenn eine URL angefragt wird),
      holt sich die benötigten Daten aus der Datenbank und prüft, ob die Anfrage zulässig ist.
      Bei Create/Update-Aktionen wird dann eine der generischen Views aus django.views.generic aufgerufen,
      die die weitere Behandlung übernimmt. Anzeige-Views rufen stattdessen im Normalfall direkt die
      Templates auf, die generischen sind da nicht so flexibel.

      Sonderfälle beim Speichern/Anlege        else:
            mailbox = models.Person.objects.filter(user=
n von Daten werden in einem spezialisierten `Manipulator` erledigt,
      der dann als Parameter der generischen View übergeben wird.
"""
import StringIO, csv, datetime, logging
from itertools import chain
import cPickle as pickle
import md5

from django import oldforms
from django.http import HttpResponse, Http404, HttpResponseRedirect
from django.template import loader, RequestContext
from django.shortcuts import render_to_response
from django.utils.datastructures import DotExpandedDict, SortedDict
from django.views.generic.create_update import update_object, create_object, delete_object
from django.conf import settings
from django.utils.encoding import force_unicode
from django.utils.translation import ugettext_lazy

from kundebunt import newforms
from kundebunt.popkern import models
from kundebunt.popkern.fields import PopQuerySet
from kundebunt.popkern.navigation import navi_url
from kundebunt.popkern.utils import domain_sort_key, partition_dict
from kundebunt.kunde_auth.decorators import service_flag_required, staff_only, check_readonly
from kundebunt.mailadmin.forms import *
from kundebunt.popkern.utils import any

def export_as_csv(unicode_rows):
    doc = StringIO.StringIO()
    try:
        writer = csv.writer(doc, dialect='excel')
        rows = ([x.encode('utf-8') for x in row] for row in unicode_rows)
        rows = list(rows)
        writer.writerows(rows)
        bla = doc.getvalue()
        #raise Bla
        #rows = list(rows)
        #for row in rows:
            #bla = doc.getvalue()
            #writer.writerow(row)
        response = HttpResponse(doc.getvalue(), content_type='text/csv; charset=utf-8')
        return response
    finally:
        doc.close()

def export_as_plain(rows):
    response = HttpResponse('', content_type='text/plain; charset=utf-8')
    rows = list(rows)
    max_length = [0] * len(rows[0])
    lengths = [[len(col) for col in row] for row in rows]
    max_lengths = [max(x) for x in zip(*lengths)]
    formatter = "".join([u"%%-%ds " % l for l in max_lengths]) + "\n"
    del lengths

    for row in chain(rows[0:1], [tuple('-' * l for l in max_lengths)], rows[1:]):
        response.write((formatter % row).encode('utf-8'))
    return response

    
def _export_mailrules(rules):
    """Iterator, der für jede Mailrule ein Tupel mit den auszugebenden Daten produziert.
    """
    yield((_(u'Quelle'), u'', u'', _(u'Art'), _(u'Ziel'), u'', u''))
    for rule in rules:
        quelle = rule.quell_adresse()
        ziel_adressen = rule.ziel_adressen()
        if rule.ziel_ist_meldung() or not ziel_adressen:
            yield (quelle.localpart, quelle.separator, quelle.display_domain(),
                            force_unicode(rule.typ_display()),
                            u'', u'', u'')
        else:
            for ziel in ziel_adressen:
                if ziel.domain or not ziel.localpart:
                    ziel_domain = ziel.domain
                else:
                    ziel_domain = _(u'(Postfach)')
                yield (quelle.localpart, quelle.separator, quelle.display_domain(),
                                force_unicode(rule.typ_display()),
                                ziel.localpart, ziel.separator, ziel_domain,
                                )

def _export_mailboxes(mailboxes, rules):
    """Iterator, der für jede Regel zu Postfächern ein Tupel mit den auszugebenden Daten produziert.
    """
    yield (_(u'E-Mail-Adresse'), u'', u'', _(u'Postfach/Ziel'), _(u'Realname'))
    for mailbox in mailboxes:
        for rule in rules.get(mailbox.user, [None]):
            if rule is None:
                yield (_(u'Keine E-Mail-Adresse vergeben'), u'', u'', mailbox.user, mailbox.name or u'')
            else:
                quelle = rule.quell_adresse()
                yield (quelle.localpart, quelle.separator, quelle.display_domain(),
                       mailbox.user, mailbox.name or u'')

class KundeNotActive(Exception):
    """Error, der gefragte Kunde ist nicht aktiv.

    Diese Meldung wird von check_kunde getriggert, wenn staff einen inaktiven Kunden abfragt.
    Über response() kann direkt ein entsprechendes Template aufgerufen werden.
    """
    def __init__(self, kunden_name):
        self.kunden_name = kunden_name
        Exception.__init__(self)
    def response(self, request):
        return render_to_response("mailadmin/kunde_inactive.html",
                                  context_instance = RequestContext(request, {"kunden_name": self.kunden_name}))
    def __repr__(self):
        return "%r is not active." % self.kunden_name

class KundeWithoutAssociations(KundeNotActive):
    """Exception, der Kunde ist zu keinem Kundeneintrag assoziiert, somit ist unklar, für wen er agieren kann.

    Diese Meldung wird von check_kunde getriggert.
    Über response() kann direkt ein entsprechendes Template aufgerufen werden.
    """
    def __init__(self):
        Exception.__init__(self)
    def response(self, request):
        return render_to_response("mailadmin/kunde_without_associations.html",
                                  context_instance = RequestContext(request))
    def __repr__(self):
        return "%r is without associations." % self.kunden_name


def check_kunde(person, kunden_name):
    if person.is_staff():
        if kunden_name == None:
            kunden = models.Kunde.objects.managed_by(person.id).distinct()
        else:
            kunden = models.Kunde.objects.filter(name=kunden_name)
            if not kunden.active().exists():
                raise KundeNotActive(kunden_name)
    else:
        kunden = models.Kunde.objects.managed_by(person.id)
        if kunden_name != None:
            kunden = kunden.filter(name=kunden_name)
            if kunden.count() == 0:
                raise Http404
        elif kunden.count() == 0:
            raise KundeWithoutAssociations
    return kunden

@service_flag_required
def portal(request, kunden_name=None):
    """
    zeigt die Postfaecher und Mailrules zu allen Kunden an,
    die von der eingeloggten person (request.person) verwaltet werden.
    """
    if not kunden_name and request.person.is_staff():
        return mailbox_qsearch(request)
    
    try:
        kunden = check_kunde(request.person, kunden_name)
    except KundeNotActive, err:
        return err.response(request)
    mailboxes = PopQuerySet.list_union(models.Person, [kunde.mailbox_set() for kunde in kunden]).order_by("user")
    mailbox_count = mailboxes.count()
    too_many_mailboxes = mailbox_count > settings.INDEX_LIMIT
    if too_many_mailboxes:
        mailboxes = []
    domain_title = None
    rules = PopQuerySet.list_union(models.Kunde, [kunde.mailrule_set.all() for kunde in kunden])
    mailrule_count = rules.count()
    if mailrule_count <= settings.INDEX_LIMIT:
        too_many_rules = False
        too_many_domains = False
        domains = []
        rules =sorted(rules, key=lambda rule: domain_sort_key(rule.quelle))
    else:
        too_many_rules = True
        rules = []

        domains = PopQuerySet.list_union(models.Kunde, [kunde.domainkunde_set.active() for kunde in kunden])
        domain_count = domains.count()
        if domain_count <= settings.INDEX_LIMIT:
            too_many_domains = False
            domain_title=_("Domains")
            domains = sorted(domains, key=lambda dom: domain_sort_key(dom.domain))
        else:
            too_many_domains = True
            domains = []
    if request.person.is_staff():
        no_domains = False
    else:
        for kunde in models.Kunde.objects.managed_by(request.person.id):
            if kunde.domainkunde_set.active().exists():
                no_domains = False
                break
        else:
            no_domains = True
    
    template = loader.get_template("mailadmin/portal.html")
    mailbox_rules = {}
    for rule in models.Mailrule.objects.for_mailbox_query(mailboxes):
        for z in rule.ziel.split(','):
            mailbox_rules.setdefault(z, []).append(rule)
    context = dict(kunden=kunden,
                   too_many_mailboxes=too_many_mailboxes,
                   mailboxes=mailboxes,
                   mailbox_rules=mailbox_rules,
                   mailbox_count=mailbox_count,
                   too_many_domains=too_many_domains,
                   domain_title=domain_title,
                   domains=domains,
                   too_many_rules=too_many_rules,
                   mailrule_count=mailrule_count,
                   rules=rules,
                   single_kunde=len(kunden)==1,
                   get_new_mailbox_url= "create-mailbox/",
                   no_domains=no_domains,
                   cust_search_form=KundeQuickSearchForm(request.person),
                   otrs_enabled=settings.OPEN_XCHANGE_OPTIONS is not None,
                    )
    if len(kunden)==1:
        context["kunden_name"] = kunden[0].name
    return HttpResponse(template.render(RequestContext(request,context)))


@service_flag_required
@check_readonly(allow_get=False)
def create_mailbox(request, kunden_name, num_extra=1):
    """
    Anlegen einer Mailbox
    """
    try:
        kunde = check_kunde(request.person, kunden_name).get()
    except KundeNotActive, err:
        return err.response(request)
    manipulator=MailboxAddManipulator(kunde, request.person, num_extra=num_extra)
    response = create_object(
        request,
        models.Person,
        template_name="mailadmin/mailbox.html",
        post_save_redirect=navi_url('mailadmin.edit_mailbox', {"mailbox_name": "%(user)s"}, True),
        manipulator=manipulator,
        extra_context = {'quelle_localpart_help': models.quelle_localpart_help()}
    )
    if hasattr(manipulator, 'new_mailbox_hint'):
        request.session['new_mailbox_hint'] = manipulator.new_mailbox_hint
    return response

@service_flag_required
@check_readonly
def edit_mailbox(request, mailbox_name=None):
    """
    Bearbeiten einer Mailbox
    """
    if request.POST.get('force_create')=='on':
        dot_data = DotExpandedDict(request.POST)['mailrule']
        return create_mailbox(request, request.POST.get("kunden_name",''), num_extra = len(dot_data.keys()))
    try:
        mailbox = models.Person.objects.filter(user=mailbox_name).has_flag("pwuse","mail").select_related().get()
        check_kunde(request.person, mailbox.kunde.name)
    except models.Person.DoesNotExist:
        raise Http404
    except KundeNotActive, err:
        return err.response(request)
    manipulator = MailboxChangeManipulator(mailbox.id, request.person)
    hinweis = request.session.get('new_mailbox_hint')
    if hinweis:
        del request.session['new_mailbox_hint']
    response = update_object(
        request,
        models.Person,
        object_id=mailbox.id,
        template_name="mailadmin/mailbox.html",
        post_save_redirect=navi_url("mailadmin.edit_mailbox", {"mailbox_name": "%(user)s"}, True),
        manipulator=manipulator,
        extra_context = {"kunden_name": mailbox.kunde.name,
                         'quelle_localpart_help': models.quelle_localpart_help(),
                         'new_mailbox_hint': hinweis,
                        },
        )
    if hasattr(manipulator, 'new_mailbox_hint'):
        request.session['new_mailbox_hint'] = manipulator.new_mailbox_hint
    return response

@service_flag_required
@check_readonly(allow_get=False)
def delete_mailbox(request, mailbox_name):
    """
    Löschen einer Mailbox
    """
    try:
        mailbox = models.Person.objects.filter(user=mailbox_name).has_flag("pwuse","mail").get()
        check_kunde(request.person, mailbox.kunde.name)
    except models.Person.DoesNotExist:
        raise Http404
    except KundeNotActive, err:
        return err.response(request)
    all_mailrules = mailbox.mailrules()
    allowed_domains = request.person.managed_domainkunden()
    editable_mailrules = []
    if request.person.is_staff:
        editable_mailrules = all_mailrules
    else:
        for rule in all_mailrules:
            manipulator = MailruleChangeManipulator2(rule.id, request.person, allowed_domains)
            if manipulator.is_editable():
                editable_mailrules.append(rule)
    mailbox_is_deleteable = len(all_mailrules) == len(editable_mailrules)
    context = RequestContext(request,{"kunden_name": mailbox.kunde.name,
                                      "mailbox": mailbox,
                                      "mailrules": editable_mailrules,
                                      "mailbox_is_deleteable": mailbox_is_deleteable})
    if request.META['REQUEST_METHOD'] == 'POST':
        for rule in editable_mailrules:
            rule.delete()
        if mailbox_is_deleteable:
            mailbox.pwuse = mailbox.pwuse & ~ models.Person.get_pwuse_bitmask("mail")
            mailbox.save()
        return HttpResponseRedirect(navi_url("mailadmin.portal", context))
    else:
        return render_to_response(template_name="mailadmin/delete_mailbox.html",
                                  context_instance=context)

@service_flag_required
@check_readonly(allow_get=False)
def delete_mailrule(request, mailrule_id):
    """
    Löschen einer Mailrule
    """
    try:
        mailrule = models.Mailrule.objects.get(id=mailrule_id)
        check_kunde(request.person, mailrule.kunde.name)
    except models.Mailrule.DoesNotExist:
        raise Http404
    except KundeNotActive, err:
        return err.response(request)
    manipulator = MailruleChangeManipulator2(mailrule_id, request.person)
    context = RequestContext(request,{"kunden_name": mailrule.kunde.name,
                                      "mailrule": mailrule,
                                      "mailrule_is_deleteable": manipulator.is_editable()})
    if request.META['REQUEST_METHOD'] == 'POST':
        if manipulator.is_editable():
            mailrule.delete()
        return HttpResponseRedirect(navi_url("mailadmin.portal", context))
    else:
        return render_to_response(template_name="mailadmin/delete_mailrule.html",
                                  context_instance=context)

@service_flag_required
def mailrule_search(request, as_csv=False, as_plain=False):
    kunden = models.Kunde.objects.managed_by(request.person.id).distinct()
    manipulator = MailruleSearchForm(request.person)
    data = request.GET.copy()
    manipulator.prepare(data)
    if request.GET:
        rules = manipulator.search_results(data)
        noquery = manipulator.noquery(request.GET)

    else:
        rules = []
        noquery = True
    context = RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, data, {}),
            "rules": rules,
            "noquery": noquery,
            "search_params": request.META['QUERY_STRING'],
            "kunden_name": manipulator.kunden_name,
            })
    if as_csv:
        return export_as_csv(_export_mailrules(rules))
    elif as_plain:
        return export_as_plain(_export_mailrules(rules))
    else:
        return render_to_response("mailadmin/mailrule_search.html", context_instance = context)

@service_flag_required
def mailrule_qsearch(request, as_csv=False, as_plain=True):
    kunden = models.Kunde.objects.managed_by(request.person.id).distinct()
    manipulator = MailruleQuickSearchForm(request.person)
    if request.GET:
        rules = manipulator.search_results(request.GET)
        noquery = manipulator.noquery(request.GET)
    else:
        rules = []
        noquery = True
    context = RequestContext(request,
              {"form": oldforms.FormWrapper(manipulator, request.GET, {}),
               "rules": rules,
               "noquery": noquery,
               "search_params": request.META['QUERY_STRING'],
               "kunden_name": manipulator.kunden_name,
              })
    if as_csv:
        return export_as_csv(_export_mailrules(rules))
    elif as_plain:
        return export_as_plain(_export_mailrules(rules))
    else:
        return render_to_response("mailadmin/mailrule_qsearch.html", context_instance=context)

@service_flag_required
def mailbox_qsearch(request, as_csv=False, as_plain=False):
    manipulator = MailboxQuickSearchForm(request.person)
    if request.GET:
        mailboxes = manipulator.search_results(request.GET)
        noquery = manipulator.noquery(request.GET)
    else:
        mailboxes = []
        noquery = True
    mailbox_rules = {}
    for rule in models.Mailrule.objects.for_mailbox_query(mailboxes):
        for z in rule.ziel.split(','):
            mailbox_rules.setdefault(z, []).append(rule)
    context = RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, request.GET, {}),
             "mailboxes": mailboxes,
             "mailbox_rules": mailbox_rules,
             "noquery": noquery,
             "search_params": request.META['QUERY_STRING'],
             "kunden_name": manipulator.kunden_name,
           })
    if as_csv:
        return export_as_csv(_export_mailboxes(mailboxes, mailbox_rules))
    elif as_plain:
        return export_as_plain(_export_mailboxes(mailboxes, mailbox_rules))
    else:
        return render_to_response("mailadmin/mailbox_qsearch.html", context_instance=context)

@service_flag_required
def kunde_qsearch(request):
    if not request.person.is_staff():
        request.user.message_set.create(message=_('Diese Funktion steht vorerst nur fuer noris-network-Mitarbeiter zur Verfuegung.'))
        return HttpResponseRedirect(navi_url('mailadmin', {}, True))
    manipulator = KundeQuickSearchForm(request.person)
    if request.GET:
        data = request.GET
        kunden_name = manipulator.search_result(data)
        if kunden_name:
            return HttpResponseRedirect(navi_url("mailadmin.portal", models.Kunde.objects.get(name=kunden_name).navi_context(), True))
    else:
        data = manipulator.flatten_data()
    return render_to_response(
        "mailadmin/kunde_qsearch.html",
        context_instance=RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, data, {}),
            }))

@service_flag_required
@check_readonly(allow_get=False)
def create_mailrule(request, kunden_name, rule_type=None):
    """
    Anlegen einer Mailregel
    """
    try:
        kunde = check_kunde(request.person, kunden_name).get()
    except KundeNotActive, err:
        return err.response(request)
    if rule_type == None:
        rule_type = 'virt'
    #rule_type = models.Mailrule.objects.all_descr("typ").get(bla=rule_type)
    return create_object(
        request,
        models.Mailrule,
        template_name="mailadmin/mailrule2.html",
        manipulator=MailruleAddManipulator2(kunde, rule_type, request.person),
        extra_context = {"kunden_name": kunden_name})

@service_flag_required
@check_readonly
def edit_mailrule(request, mailrule_id, rule_type):
    """
    Bearbeiten einer Mailregel
    """
    if request.POST.get('force_create')=='on':
        return create_mailrule(request, request.POST.get("kunden_name",''))
    try:
        mailrule = models.Mailrule.objects.get(id=mailrule_id)
        check_kunde(request.person, mailrule.kunde.name)
    except models.Mailrule.DoesNotExist:
        raise Http404
    except KundeNotActive, err:
        return err.response(request)
    manipulator = MailruleChangeManipulator2(mailrule_id, request.person, new_rule_type=rule_type)
    return update_object(
        request,
        models.Mailrule,
        template_name="mailadmin/mailrule2.html",
        post_save_redirect=navi_url("mailadmin.edit_mailrule", mailrule.navi_context(), True),
        manipulator=manipulator,
        object_id = mailrule_id,
        extra_context = dict(kunden_name=manipulator.original_object.kunde.name))

@staff_only
@check_readonly(allow_get=False)
def set_mailflag(request, mailbox_name):
    try:
        mailbox = models.Person.objects.get(user=mailbox_name)
        check_kunde(request.person, mailbox.kunde.name)
    except models.Mailrule.DoesNotExist:
        raise Http404
    except KundeNotActive, err:
        return err.response(request)
    if request.POST:
        mailbox.pwuse |= models.Person.get_pwuse_bitmask('mail')
        mailbox.save()
        return HttpResponseRedirect(navi_url('mailadmin.edit_mailbox', mailbox.navi_context(), True))
    else:
        manipulator = MailboxChangeManipulator(mailbox.id, request.person)
        manipulator["name"].readonly = manipulator["user"].readonly = True
        context = {'form': oldforms.FormWrapper(manipulator, manipulator.flatten_data(), {}),
                   'kunden_name': mailbox.kunde.name}
        template = loader.get_template("mailadmin/set_mailflag.html")
        return HttpResponse(template.render(RequestContext(request, context)))


class OpenXChangeForm(newforms.BaseForm):
    
    if settings.OPEN_XCHANGE_OPTIONS is not None:
        __options = [(models.OpenXChange.get_option_code(s), models.OPEN_XCHANGE_DISLAY.get(s, s)) for s in settings.OPEN_XCHANGE_OPTIONS]
        __fields = { False: newforms.ChoiceField([(u'', u'--')] + __options, required=False),
                    True:  newforms.ChoiceField(__options, required=True) }
        base_fields = []
    
    def __init__(self, kunden, *args, **kwargs):
        super(OpenXChangeForm, self).__init__(*args, **kwargs)
        self.fields = SortedDict([('kunde_%s' % k.name, self.__fields[k.hat_open_xchange_entschieden()]) for k in kunden])

    def clean(self):
        if not any(self.cleaned_data.itervalues()):
            if len(self.fields) > 1:
                raise newforms.ValidationError(_(u'Bitte mindestens bei einem Kunden eine Option auswählen.'))
            else:
                raise newforms.ValidationError(_(u'Bitte eine Option auswählen.'))
        return self.cleaned_data


class ConfirmationForm(newforms.Form):
    confirm = newforms.FrontBooleanField(label=u'''Der Bestellung und den Allgemeinen Geschäftsbedingungen stimme ich zu.''')
    signature = newforms.CharField(widget=newforms.HiddenInput)
    
    def clean_confirm(self):
        data = self.cleaned_data.get('confirm')
        if not data:
            raise newforms.ValidationError(_(u'Bitte bestätigen Sie Ihre Zustimmung.'))
        return


@service_flag_required
@check_readonly
def open_xchange(request, kunden_name=None):
    """
    zeigt die Postfaecher und Mailrules zu allen Kunden an,
    die von der eingeloggten person (request.person) verwaltet werden.
    """
    if not settings.OPEN_XCHANGE_OPTIONS:
        # open_xchange abgeschaltet.
        raise Http404
    if not request.person.is_staff():
        # Nicht-Mitarbeitern immer alle verwalteten Kunden anzeigen.
        kunden_name = None
    if kunden_name:
        kunden_suffix = ""
    else:
        kunden_suffix = ".ohnekunde"
    if 'open_xchange_options' in request.session:
        del request.session['open_xchange_options']
    try:
        kunden = check_kunde(request.person, kunden_name)
    except KundeNotActive, err:
        return err.response(request)
        
    if request.method == 'POST':
        form = OpenXChangeForm(kunden, request.POST)
        if form.is_valid():
            data = form.cleaned_data
            infos = []
            for kunde in kunden:
                mailbox_count = len(kunde.mailbox_set())
                option = data.get(u"kunde_%s" % kunde.name, u'')
                if option != u'':
                    option = models.OpenXChange.get_option_descr_obj(int(option))
                    infos.append(dict(kunde_id=kunde.id, kunden_name=kunde.name, mailbox_count=mailbox_count,
                                    option=option, option_display=unicode(models.OPEN_XCHANGE_DISLAY.get(option.bla, option.bla)),
                                    kosten=settings.OPEN_XCHANGE_COSTS[option.bla.lower()] * mailbox_count,
                                    ))
            request.session['open_xchange_options'] = infos
            return HttpResponseRedirect(navi_url('mailadmin.open_xchange_bestellung'+kunden_suffix, {'kunden_name': kunden_name}, True))
    else:
        objs = (qset.latest("beginn")
                for qset in (models.OpenXChange.objects.filter(kunde=k) for k in kunden)
                if qset.exists())
        data = dict.fromkeys(("kunde_%s" % k.name for k in kunden), u'')
        data.update(("kunde_%s" % obj.kunde.name, obj.option.descr) for obj in objs)
        form = OpenXChangeForm(kunden, initial=data)
    infos = []
    for kunde in kunden:
        mailbox_count = len(kunde.mailbox_set())
        infos.append(dict(field=form["kunde_%s" % kunde.name],
                            kunden_name = kunde.name,
                            mailbox_count=mailbox_count,
                            kosten=[settings.OPEN_XCHANGE_COSTS[option.lower()] * mailbox_count
                                    for option in settings.OPEN_XCHANGE_OPTIONS
                                    if option != u'off']
                    )    )
    template = loader.get_template("mailadmin/open_xchange_auswahl.html")
    content = template.render(RequestContext(
            request,
            {'form': form, 'infos': infos,
             'optionen': [models.OPEN_XCHANGE_DISLAY[s]
                          for s in settings.OPEN_XCHANGE_OPTIONS
                          if s!= u"off"],
             'kunden_name': kunden_name,
            }))
    return HttpResponse(content)

def _signature(data):
    pickled = pickle.dumps(data, pickle.HIGHEST_PROTOCOL)
    return md5.new(pickled).hexdigest()

@service_flag_required
@check_readonly
def open_xchange_bestellung(request, kunden_name=None):
    """
    zeigt die Postfaecher und Mailrules zu allen Kunden an,
    die von der eingeloggten person (request.person) verwaltet werden.
    """
    if not settings.OPEN_XCHANGE_OPTIONS:
        # open_xchange abgeschaltet.
        raise Http404
    if not request.person.is_staff():
        # Nicht-Mitarbeitern immer alle verwalteten Kunden anzeigen.
        kunden_name = None
    if kunden_name:
        kunden_suffix = ""
    else:
        kunden_suffix = ".ohnekunde"
    bad_session_url = navi_url("mailadmin.open_xchange"+kunden_suffix, {'kunden_name': kunden_name}, True)
    if 'open_xchange_options' not in request.session:
        logging.info("open_xchange_bestellung: %s abgewiesen, 'open_xchange_options' not in request.session" % request.person.user)
        return HttpResponseRedirect(bad_session_url)
    try:
        kunden = check_kunde(request.person, kunden_name)
    except KundeNotActive, err:
        return err.response(request)
    option_infos = request.session["open_xchange_options"]
    if request.method == 'POST':
        if _signature(option_infos) != request.POST.get('signature'):
            del request.session['open_xchange_options']
            logging.info("open_xchange_bestellung: %s abgewiesen, signature stimmt nicht." % request.person.user)
            return HttpResponseRedirect(bad_session_url)
        form = ConfirmationForm(request.POST)
        if form.is_valid():
            if 'open_xchange_log' not in request.session:
                logging.info("open_xchange_bestellung: %s abgewiesen, 'open_xchange_log' not in request.session" % request.person.user)
                return HttpResponseRedirect(bad_session_url)
            try:
                try:
                    log_obj = models.OpenXChangeLog.objects.get(id=request.session['open_xchange_log'])
                except models.OpenXChangeLog.DoesNotExist:
                    logging.info("open_xchange_bestellung: %s abgewiesen, models.OpenXChangeLog.DoesNotExist" % request.person.user)
                    return HttpResponseRedirect(bad_session_url)
            finally:
                del request.session['open_xchange_log']
            for info in option_infos:
                if info['option'] is not None:
                    models.OpenXChange.objects.create(
                        kunde=models.Kunde.objects.get(id=info['kunde_id']),
                        beginn=datetime.datetime.now(),
                        option=info['option'],
                        log=log_obj,
                        request=unicode(str(request), 'iso-8859-1'))
            request.user.message_set.create(message=_(u"Ihre Eingaben wurden gesichert. ")
                                                      + _(u"Es kann noch bis zu zehn Minuten dauern, bis die Änderung aktiv wird."))
            return HttpResponseRedirect(navi_url('mailadmin.open_xchange'+kunden_suffix, {'kunden_name': kunden_name}, True))
    else:
        form = ConfirmationForm(initial={'signature': _signature(option_infos)})
    template = loader.get_template("mailadmin/open_xchange_bestellung.html")
    content = template.render(RequestContext(
                request, {'form': form, 'infos': option_infos, 'kunden_name': kunden_name,
                          'gesamt': sum(info["kosten"] for info in option_infos)}))
    log_obj = models.OpenXChangeLog.objects.create(person=request.person, anzeige = unicode(content))
    request.session['open_xchange_log'] = log_obj.id
    return HttpResponse(content)

