# -*- encoding: utf-8 -*-

"""
Manipulators for the various searches
"""
from django import oldforms
from django.core import validators
from django.db import transaction
from django.db.models.query import Q
from django.utils.translation import gettext_lazy
from kundebunt.popkern import models
from kundebunt.popkern.fields import follow_only, ReadonlyTextField, PopQuerySet
from kundebunt.popkern.utils import domain_sort_key, EmailAddress
from kundebunt.popkern.datasources import KundenDataSource, DomainDataSource
from kundebunt.popkern.formfields import *

schnellsuche_hilfe = gettext_lazy("Der eingegebene Text wird als Teilstring gesucht. Jokerzeichen wie '*' oder '?' sind hier nicht moeglich.")

class MailboxQuickSearchForm(oldforms.Manipulator):
    def __init__(self, editor_person):
        self.kunden_name = None
        self.user_is_staff = editor_person.is_staff()
        self.kunden_src = KundenDataSource(editor_person)
        self.editor_person = editor_person
        self.fields = [
            KundenComboField(field_name="kunde",
                       person=editor_person,
                       data_source=self.kunden_src,
                       is_required=False,
                       choices_only=not self.user_is_staff),
            oldforms.TextField(field_name="search")
        ]
        self["search"].help_text = schnellsuche_hilfe

    def noquery(self, data):
        """Wenn True, dann findet keine Suche statt"""
        return "noquery" in data or 'quick' in data and not data.get("search")

    def search_results(self, data):
        if self.noquery(data):
            return []
        try:
            data_kunde = self['kunde'].html2python(data.get("kunde"))
        except ValueError:
            self.kunden_name = None
            return []
        data_search = data.get("search")
        if self.user_is_staff:
            kunden = models.Kunde.objects.all()
        else:
            kunden = self.editor_person.managed_kunden()
        self.kunden_name = None
        if data_kunde:
            kunden = kunden.filter(name=data_kunde)
            if kunden:
                self.kunden_name = kunden[0].name
        if self.user_is_staff and not data_kunde and data_search:
            # Hat keinen Sinn, über die zugeordneten Mailboxen zu gehen, staff sieht eh alle.
            all_mboxes = models.Person.objects.active().has_flag("pwuse", "mail")
        else:
            # Suche wird über die `kunden` zugeordneten Mailboxen eingeschränkt
            if self.user_is_staff and not data_kunde and not data_search:
                # Notbremse: Wenn ein Mitarbeiter gar nichts eingibt, zeigen wir
                # ihm nur die Daten die zu seinen assoziierten Kunden gehören.
                kunden = self.editor_person.managed_kunden()
            all_mboxes = PopQuerySet.list_union(models.Person, [kunde.mailbox_set() for kunde in kunden])
        if data_search and (data_search!="*"):
            by_quelle = all_mboxes.extra(where=["(person.user=mailrules.ziel and mailrules.quelle like %s)"],
                                tables=["mailrules"],
                                params=["%%%s%%" % data_search])
            # Folgendes ist nötig, weil die Django-ORM QuerySets zur selben Tabelle mit unterschiedlichen
            # Joins nicht richtig behandelt, sonst ginge dieses:
            #   mboxes = list((all_mboxes.filter(Q(user__icontains=data_search)|Q(name__icontains=data_search)) | by_quelle).distinct())
            mboxes = list(all_mboxes.filter(Q(user__icontains=data_search)|Q(name__icontains=data_search)).distinct())
            mboxes.extend([m for m in list(by_quelle.distinct()) if m not in mboxes])
            mboxes.sort(key=lambda mbox:mbox.user)
        else:
            mboxes = list(all_mboxes.select_related())
        mboxes.sort(key=lambda mbox:mbox.user)
        return mboxes



class MailruleQuickSearchForm(oldforms.Manipulator):
    def __init__(self, editor_person):
        self.user_is_staff = editor_person.is_staff()
        self.kunden_src = KundenDataSource(editor_person)
        self.domain_src = DomainDataSource(editor_person)
        self.kunden_name = None
        self.editor_person = editor_person
        if self.user_is_staff:
            # bei staff werden sonst Domains für den falschen Kunden generiert.
            use_javascript = False
        else:
            use_javascript = None
        self.fields = [
            KundenComboField(field_name="kunde",
                       person=editor_person,
                       data_source=self.kunden_src,
                       is_required=False, null_display="*",
                       choices_only=not self.user_is_staff),
            oldforms.TextField(field_name="search"),
            DomainComboField(field_name="domain_select",
                       data_source=self.domain_src,
                       is_required=False,
                       choices_only=not self.user_is_staff,
                       use_javascript=use_javascript),
        ]
        self["search"].help_text = schnellsuche_hilfe


    def noquery(self, data):
        """Wenn True, dann findet keine Suche statt"""
        return "noquery" in data or 'quick' in data and not data.get("search")
    
    def search_results(self, data):
        try:
            data_kunde = self['kunde'].html2python(data.get("kunde"))
        except ValueError:
            self.kunden_name = None
            return []
        if self.noquery(data):
            return []
        self.kunden_name = None
        if self.user_is_staff:
            kunden = models.Kunde.objects.all()
        else:
            kunden = self.editor_person.managed_kunden()
        if data_kunde:
            # Paranoia ... die Felder werden in den Such-Formularen
            # nicht extra validiert.
            kunden = kunden.filter(name=data_kunde)
            if kunden:
                self.kunden_name = kunden[0].name
        if self.user_is_staff and not data_kunde and data.get("search"):
            # Hat keinen Sinn, über die zugeordneten Kunden
            qset = models.Mailrule.objects.all()
        else:
            # Suche wird über die zugeordneten Kunden eingeschränkt
            if self.user_is_staff and not data_kunde and not data.get("search"):
                # Notbremse: Wenn ein Mitarbeiter gar nichts eingibt, zeigen wir
                # ihm nur die Daten die zu seinen assoziierten Kunden gehören.
                kunden = self.editor_person.managed_kunden()
            qset = models.Mailrule.objects.filter(kunde__in = [k.id for k in kunden])
        search = data.get("search")
        if search=="*":
            qset = qset.filter(quelle__isnull=True)
        elif search:
            qset = (  qset.filter(quelle__icontains=search)
                    | qset.filter(ziel__icontains=search))
        domain = data.get("domain_select")
        if domain:
            qset = qset.filter(quelle__iendswith=domain) | qset.filter(ziel__iendswith=domain)
        rules = list(qset.distinct())
        rules.sort(key=lambda rule: domain_sort_key(rule.quelle))
        return rules


class MailruleSearchForm(oldforms.Manipulator):
    def __init__(self, editor_person):
        self.kunden_src = KundenDataSource(editor_person)
        self.kunden_name = None
        self.user_is_staff = editor_person.is_staff()
        self.editor_person = editor_person
        self.fields = [
            KundenComboField(field_name="kunde",
                person=editor_person,
                data_source = self.kunden_src,
                is_required=False,
                choices_only=not self.user_is_staff),
            oldforms.SelectField(field_name="typ",
                              choices = [("","*")]
                                        + [ (descr.bla, models.MAILRULE_TYP_DISPLAY[descr.bla])
                                            for descr in models.Mailrule.objects.all_descr("typ")
                                            if descr.bla in models.MAILRULE_TYP_DISPLAY ]
                                        + [ ("-%s" % descr.bla, _("ausser %s") % models.MAILRULE_TYP_DISPLAY[descr.bla])
                                            for descr in models.Mailrule.objects.all_descr("typ")
                                            if descr.bla in models.MAILRULE_TYP_DISPLAY ]),
            oldforms.SelectField(field_name="field", choices = [("any","*"), ("quelle",_("Quelle")), ("ziel",_("Ziel"))]),
            oldforms.SelectField(field_name="localpart_mode",
                              choices = [("all",_("alle")), ("equals",_("gleich")), ("starts_with", _("beginnt mit")),
                                         ("contains", _("enthaelt"))]),
            oldforms.TextField(field_name="localpart"),
            oldforms.SelectField(field_name="domain_mode",
                              choices = [("all",_("alle")), ("equals",_("gleich")), ("starts_with", _("beginnt mit")),
                                         ("contains", _("enthaelt"))]),
            oldforms.TextField(field_name="domain_text"),
        ]

    def prepare(self, data):
        super(MailruleSearchForm, self).prepare(data)
        data.setdefault("localpart_mode","starts_with")
        data.setdefault("domain_mode","starts_with")

    def noquery(self, data):
        """Wenn True, dann findet keine Suche statt"""
        return "noquery" in data or 'quick' in data and not data.get("search")


    def search_results(self, data):
        def _fquery(field, condition, value):
            return { "%s__%s" % (field, condition): value }
        def _filter_succeeds(fn_list, addr_list):
            # für jede fn muss für mind. ein addr aus addr_list
            #   fn(addr) erfüllt sein.
            for fn in fn_list:
                for addr in addr_list:
                    if fn(addr):
                        break
                else:
                    return False
            return True
        # first query roughly the set we need. The conditions are not specific enough,
        # later we'll postprocess the result set. post_filter contains a list of functions
        # that do the postprocessing.
        if self.noquery(data):
            return []
        try:
            data_kunde = self['kunde'].html2python(data.get("kunde"))
            self.kunden_name = data_kunde
        except ValueError:
            self.kunden_name = None
            return []
        post_filter = []
        if data == {} or 'noquery' in data:
            return []
        if data.get("field")=="ziel":
            search_fields = ["ziel"]
        elif data.get("field")=="any":
            search_fields = ['quelle', 'ziel']
        else:
            search_fields = ["quelle"]
        search_is_restricted = (
               data.get("localpart") and data.get("localpart_mode") and data["localpart_mode"] != "all"
            or data.get("domain_text") and data.get("domain_mode") and data["domain_mode"] != "all" and data.get("domain_text")!="*")
        if self.user_is_staff:
            kunden = models.Kunde.objects.all()
        else:
            kunden = self.editor_person.managed_kunden()
        if data_kunde:
            # Paranoia ... die Felder werden in den Such-Formularen
            # nicht extra validiert.
            kunden = kunden.filter(name=data_kunde)
        elif self.user_is_staff and not search_is_restricted:
            # Notbremse: Wenn ein Mitarbeiter gar nichts eingibt, zeigen wir
            # ihm nur die Daten die zu seinen assoziierten Kunden gehören.
            kunden = self.editor_person.managed_kunden()
        rules = []
        for field in search_fields:
            if self.user_is_staff and not data_kunde and search_is_restricted:
                # Hat keinen Sinn, über die zugeordneten Kunden zu gehen,
                # staff hat eh Zugriff auf alle.
                qset = models.Mailrule.objects.all()
            else:
                # Suche wird über die zugeordneten Kunden eingeschränkt
                qset = models.Mailrule.objects.filter(kunde__in = [k.id for k in kunden])
            #if data.get("kunde"):
                #if data["kunde"] not in [k.id for k in self.kunden]:
                    ## Paranoia ...
                    #return []
                #qset = models.Mailrule.objects.filter(kunde=data["kunde"])
            #else:
                #qset = models.Mailrule.objects.filter(kunde__in = self.kunden_ids)
            if data.get("typ"):
                if data["typ"].startswith("-"):
                    qset = qset.descr_ne("typ", data["typ"][1:])
                else:
                    qset = qset.descr_eq("typ",data["typ"])
            if data.get("localpart") and data.get("localpart_mode") and data["localpart_mode"] != "all":
                localpart = data["localpart"].lower()
                localpart_mode = data["localpart_mode"]
                if localpart_mode in ["equals","starts_with"]:
                    qset = qset.filter(**_fquery(field,"istartswith", localpart))
                    if localpart_mode == "equals":
                        post_filter.append(lambda addr: getattr(addr, "localpart", addr)==localpart)
                    else:
                        post_filter.append(lambda addr: getattr(addr, "localpart", addr).startswith(localpart))
                else:
                    qset = qset.filter(**_fquery(field,"icontains",data["localpart"]))
                    post_filter.append(lambda addr: getattr(addr, "localpart", addr).find(localpart)!=-1)
            if data.get("domain_text") and data.get("domain_mode")  and data["domain_mode"]!= "all":
                domain_mode = data["domain_mode"]
                domain_text = data["domain_text"]
                if domain_text == "*":
                    qset = qset.filter(quelle__isnull=True)
                elif domain_mode == "equals":
                    qset = qset.filter(**_fquery(field, "iendswith", domain_text))
                    post_filter.append(lambda addr: getattr(addr, "domain", addr)==domain_text or getattr(addr, "domain", addr).endswith("." + domain_text))
                else:
                    qset = qset.filter(**_fquery(field, "icontains", domain_text))
                    if domain_mode == "starts_with":
                        post_filter.append(lambda addr: getattr(addr, "domain", addr).startswith(domain_text))
                    else:
                        post_filter.append(lambda addr: getattr(addr, "domain", addr).find(domain_text)!=-1)

            # Now match again and filter out the real matches
            if field=="ziel":
                get_addresses = lambda rule: rule.ziel_adressen()
            else:
                get_addresses = lambda rule: [rule.quell_adresse()]
            rules.extend([ rule for rule in qset if _filter_succeeds(post_filter, get_addresses(rule)) ])
        rules.sort(key=lambda rule: (domain_sort_key(rule.quelle), rule.id))
        # return without duplicates
        return ([rule for i, rule in enumerate(rules) if i==0 or rule.id!=rules[i-1].id])

class KundeQuickSearchForm(oldforms.Manipulator):

    def __init__(self, editor_person):
        self.fields = [
            KundenComboField(field_name="cust_search",
                person=editor_person,
                is_required=False,
                choices_only=not editor_person.is_staff()),
            ]

    def search_result(self, data):
        """Gibt den Kundennamen oder None zurück."""
        try:
            return self['cust_search'].html2python(data.get("cust_search"))
        except ValueError:
            return None
