# -*- encoding: utf-8 -*-

"""
Manipulators for editing and creating mailrules
"""
from django import oldforms
from django.core import validators
from django.db import transaction
from django.http import Http404
from django.utils.simplejson.encoder import JSONEncoder
from django.utils.safestring import mark_safe
from kundebunt.popkern import models
from kundebunt.popkern.async_dns import AsyncDNSResolver
from kundebunt.popkern.fields import follow_only, ReadonlyTextField, PopQuerySet
from kundebunt.popkern.utils import domain_sort_key, EmailAddress, assemble_address
from kundebunt.popkern.datasources import WildcardDomainDataSource
from kundebunt.popkern.formfields import *


class MailruleAddManipulator2(oldforms.Manipulator):
    """Spezialisierter Manipulator zum Anlegen/Editieren von Mailrules
    für einen bestimmten Kunden.
    """
    change = False
    ziel_label = dict.fromkeys(models.Mailrule.get_typ_group_list("to_host"), _("Hostname (FQDN)"))
    ziel_label.update(dict.fromkeys(models.Mailrule.get_typ_group_list("to_domain"), _("Domain")))
    quelle_hat_localpart = set(models.Mailrule.get_typ_group_list('@_allowed'))
    ziel_irrelevant = set(models.Mailrule.get_typ_group_list('target_irrelevant'))  # z.B. dev_null
    ziel_ist_meldung = set(models.Mailrule.get_typ_group_list('to_freetext'))       # z.B. reject, defer
    ziel_mehrfach = set(models.Mailrule.get_typ_group_list('multiple_targets'))     # z.B. virt

    def __init__(self, kunde, typ, editor_person, allowed_domains=None):
        super(MailruleAddManipulator2,self).__init__()
        self.kunde = kunde
        self.editor_person = editor_person
        self.user_is_staff = editor_person.is_staff()
        if self.user_is_staff:
            self.allowed_types = [t for t in models.Mailrule.objects.all_descr("typ")]
        else:
            self.allowed_types = [t for t in models.Mailrule.get_typ_descr_in_group('kunde')]
            if self.change and self.original_object.typ not in self.allowed_types:
                self.allowed_types.append(self.original_object.typ)
        self.typ_choices = [ (t.bla, models.MAILRULE_TYP_DISPLAY.get(t.bla, t.bla) )
                             for t in self.allowed_types ]
        self.rule_type = typ
        if allowed_domains == None:
            if self.user_is_staff:
                allowed_domains = kunde.domainkunde_set.active()
            else:
                allowed_domains = editor_person.managed_domainkunden()
            allowed_domains = list(allowed_domains)
            allowed_domains.sort(key=lambda dom: domain_sort_key(dom.domain))
        self.allowed_domains = allowed_domains
        resolver = AsyncDNSResolver()
        self.fields = [ReadonlyTextField("kunden_name"),
                       oldforms.SelectField("rule_type",
                                         choices = [ (t.bla, models.MAILRULE_TYP_DISPLAY.get(t.bla, t.bla) )
                                                     for t in self.allowed_types]),
                       LocalpartField("quelle_localpart"),
                       SubdomainField("quelle_subdomain"),
                       WildcardDomainComboField(field_name="quelle_domain",
                                        data_source=WildcardDomainDataSource(editor_person, kunde.id),
                                        choices_only=not self.user_is_staff,
                                        is_required=True
                                       ).add_advisors([checkSourceDomain(self.allowed_domains, self.user_is_staff,
                                                                         allow_wildcard=True),
                                                       hasResolvableDomain(
                                                         resolver=resolver,
                                                         subdomain_field_name='quelle_subdomain',
                                                         allow_wildcard=True)]),
                       EmailTargetCollectionField("virt_ziele", cols=80,
                                                 ).add_advisors([hasExistingEmailTargets(resolver=resolver)]),
                       oldforms.LargeTextField("meldung", cols=80, rows=5, validator_list=[hasNoDollar]),
                       FQDNField("ziel").add_advisors([hasResolvableDomain(resolver, ["mx", "a"])]),
                       oldforms.CheckboxField("force_create")]
        self["rule_type"].help_text = models.mailrule_typ_help()
        self["quelle_localpart"].help_text = models.quelle_localpart_help()

    def flatten_data(self):
        """returns a dict of form data (strings) for the corresponding model
        Setzt außerdem quelle_obsolete_domain.choices
        """
        if self.change:
            if self.original_object.quelle:
                addr = EmailAddress(self.original_object.quelle, True)
            else:
                addr = EmailAddress("*", True)
        else:
            addr = EmailAddress('', True)
        addr.check_domain(self.allowed_domains)
        new_data = {}
        new_data["kunden_name"] = self.kunde.name
        new_data["quelle_localpart"] = addr.localpart
        new_data["quelle_subdomain"] = addr.subdomain
        new_data["quelle_domain"] = addr.parent_domain
        new_data["form_is_readonly"] = ""
        new_data["rule_type"] = self.rule_type
        if self.change:
            typ_code = models.Mailrule.get_typ_code(self.rule_type)
            if typ_code in self.ziel_mehrfach:
                new_data["virt_ziele"] = "\n".join([str(zaddr) for zaddr in self.original_object.ziel_adressen()])
            elif typ_code in self.ziel_ist_meldung:
                new_data["meldung"] = self.original_object.ziel
            else:
                descr = models.Mailrule.get_typ_descr(self.rule_type)
                if not descr.descr in self.ziel_irrelevant:
                    new_data["ziel"] = self.original_object.ziel
        if not self.is_editable(addr):
            self.make_form_readonly()
            new_data["form_is_readonly"] = 'True'
        return new_data

    def jscript_data(self):
        encoder = JSONEncoder()
        return {
            "quelle_hat_localpart": mark_safe(encoder.encode([t.descr in self.quelle_hat_localpart
                                                             for t in self.allowed_types])),
            "ziel_hat_virt_ziele": mark_safe(encoder.encode([t.bla == 'virt'
                                                             for t in self.allowed_types])),
            "ziel_irrelevant": mark_safe(encoder.encode([t.descr in self.ziel_irrelevant
                                                         for t in self.allowed_types])),
            "ziel_label": mark_safe(encoder.encode([self.ziel_label.get(t.descr, '')
                                                         for t in self.allowed_types])),
            "ziel_ist_meldung": mark_safe(encoder.encode([t.descr in self.ziel_ist_meldung
                                                          for t in self.allowed_types])),
        }

    def make_form_readonly(self):
        for field in self.fields:
            field.readonly = True

    def prepare(self, data):
        rule_type = data.get('rule_type', '')
        try:
            descr = models.Mailrule.get_typ_descr(rule_type)
            data["rule_type_descr"] = descr
        except models.Descr.DoesNotExist:
            raise Http404
        if not descr.descr in self.quelle_hat_localpart:
            data['quelle_localpart'] = ''
        elif data.get('quelle_localpart') == '*':
            data['quelle_localpart'] = ''
        if descr.descr in self.ziel_irrelevant:
            data['ziel'] = data['virt_ziele'] = data['meldung'] = ''
        else:
            if descr.descr not in self.ziel_mehrfach:
                data['virt_ziele'] = ''
            if descr.descr not in self.ziel_ist_meldung:
                data['meldung'] = ''
            if (descr.descr in self.ziel_mehrfach
                  or descr.descr in self.ziel_ist_meldung):
                data['ziel'] = ''
        data.setdefault('quelle_localpart')
        data.setdefault('quelle_subdomain')
        data.setdefault('quelle_domain')

    def validate_rule_type(self, field_data, all_data):
        return isValidMailruleType(self.user_is_staff)(field_data)

    @always_test
    def validate_quelle_localpart(self, field_data, all_data):
        if all_data["rule_type_descr"].descr in self.quelle_hat_localpart:
            self.validate_quelle(field_data, all_data)

    @always_test
    def validate_quelle_domain(self, field_data, all_data):
        if not all_data["rule_type_descr"].descr in self.quelle_hat_localpart:
            self.validate_quelle(field_data, all_data)

    def validate_quelle(self, field_data, data):
        if not self.is_editable():
            raise validators.ValidationError(_('Weiterleitungsregeln fuer "%(quelle)s" koennen Sie nicht veraendern.')
                                               % {'quelle': field_data})
        if data["quelle_domain"]=="*":
            if data["quelle_localpart"] or data["quelle_subdomain"]:
                raise validators.ValidationError(_('Die Domain "*" bedeutet, dass diese Regel auf alle E-Mails angewandt wird, auf die keine andere Regel passt. Sie ist daher nicht mit weiteren Angaben kombinierbar.'))
            other_rule = models.Mailrule.objects.filter(quelle__isnull=True, kunde__id=self.kunde.id)
            if self.change:
                other_rule = other_rule.exclude(id=self.obj_key)
            if other_rule.exists():
                raise validators.ValidationError(_('Eine Weiterleitungsregel fuer "%(quelle)s" besteht bereits.') % {'quelle': "*"})

        quelle = assemble_address(data["quelle_localpart"], data["quelle_subdomain"], data["quelle_domain"])
        other_rule = models.Mailrule.objects.filter(quelle=quelle)
        if self.change:
            other_rule = other_rule.exclude(pk=self.obj_key)
        if other_rule.exists():
            raise validators.ValidationError(_('Eine Weiterleitungsregel fuer "%(quelle)s" besteht bereits.')
                                               % {'quelle': quelle})

    @always_test
    def validate_virt_ziele(self, field_data, all_data):
        if all_data["rule_type"] in self.ziel_mehrfach and not field_data:
            raise validators.ValidationError(_('This field is required.'))

    @always_test
    def validate_ziel(self, field_data, all_data):
        descr = all_data["rule_type_descr"]
        if (not field_data
              and not descr.descr in self.ziel_irrelevant
              and not descr.descr in self.ziel_mehrfach
              and not descr.descr in self.ziel_ist_meldung):
            raise validators.ValidationError(_('This field is required.'))

    #def get_validation_errors(self, data):
        #if not self.is_editable():
            #return {'quelle_domain': _('Weiterleitungsregeln zu "%(quelle)s" können Sie nicht verändern.')
                                       #% {'quelle': self.original_object.quelle}}
        #return super(MailruleAddManipulator2, self).get_validation_errors(data)

    def save(self, data):
        if data['rule_type_descr'].descr in self.ziel_irrelevant:
            ziel = None
        elif data["virt_ziele"]:
            ziel = ",".join([line.strip() for line in data["virt_ziele"].split("\n")])
        elif data["meldung"]:
            ziel = data["meldung"]
        else:
            ziel = data["ziel"]
        if data["quelle_domain"]=="*":
            quelle = None
        else:
            quelle=assemble_address(data["quelle_localpart"], data["quelle_subdomain"], data["quelle_domain"])
        params = dict(kunde = self.kunde,
                      typ=data["rule_type_descr"],
                      quelle=quelle,
                      ziel=ziel)
        if self.change:
            params["id"] = self.obj_key
        new_obj = models.Mailrule(**params)
        new_obj.save()
        self.editor_person.auth_user().message_set.create(message=_("Es kann aber noch bis zu 10 Minuten dauern, bis die Aenderung aktiv wird."))
        return new_obj

    def is_editable(self, addr=None):
        """New mailrules are always editable.
        Rewritten in MailruleChangeManipulator.
        """
        return True


class MailruleChangeManipulator2(MailruleAddManipulator2):
    change = True

    def __init__(self, obj_key, editor_person, new_rule_type=None, allowed_domains=None, obj=None, kunde_cache = None):
        if obj == None:
            self.obj_key = obj_key
            self.original_object = models.Mailrule.objects.get(pk=obj_key)
        else:
            self.obj_key = obj.id
            self.original_object = obj
        # Meistens ist der kunde derselbe wie der in kunde_cache übergebene (aber nicht immer):
        if kunde_cache and kunde_cache.id == self.original_object.kunde_id:
            kunde = kunde_cache
        else:
            kunde = self.original_object.kunde
        if new_rule_type and new_rule_type != self.original_object.typ:
            typ = new_rule_type
            self.changing_type = True
        else:
            typ = self.original_object.typ.bla
            self.changing_type = False
        super(MailruleChangeManipulator2, self).__init__(kunde, typ, editor_person, allowed_domains)

    def is_editable(self, addr=None):
        """check whether, based on either `addr` as parsed `quelle` or the original object,
        a user can be allowed to edit/delete this mailrule
        """
        if self.editor_person.is_staff():
            return True
        if addr==None:
            addr = EmailAddress(self.original_object.quelle)
        if not addr.is_checked():
            addr.check_domain(self.allowed_domains)
        return addr.foreign_domain == None

