# -*- encoding: utf-8 -*-

"""
Manipulators for editing and creating mailboxes
"""

import os
from itertools import izip, chain, repeat, count
from django import oldforms
from django.core import validators
from django.db import transaction
from django.utils.datastructures import DotExpandedDict
from django.utils.translation import gettext, gettext_lazy, ngettext, ugettext_lazy, ugettext, ugettext as _
from django.utils.safestring import mark_safe
from django.utils.html import conditional_escape
from kundebunt.popkern import models
from kundebunt.popkern.async_dns import AsyncDNSResolver
from kundebunt.popkern.fields import follow_only, ReadonlyTextField, PopQuerySet
from kundebunt.popkern.utils import domain_sort_key, EmailAddress, assemble_address, any
from kundebunt.popkern.formfields import *
from kundebunt.popkern.navigation import navi_url
from kundebunt.popkern.datasources import WildcardDomainDataSource

_missing_sender_email = ugettext_lazy(u"""Es ist noch keine Absenderadresse fuer das Webmail-Interface konfiguriert.

Wenn Sie das Webmail-Interface nutzen moechten, tragen Sie bitte die Absenderadresse manuell ein, indem Sie sich mit den Zugangsdaten auf <a href="https://mail.noris.net">https://mail.noris.net</a> anmelden und sie dann unter Optionen / "Persoenliche Informationen" / E-Mail-Adresse eingeben.""")

_deviating_sender_email = ugettext_lazy(u"""<p>Wenn Sie das Webmail-Interface nutzen:</p>

<p>Dieses Postfach hat die ungültige Absender- und Antwortadresse '%(email)s':
Es fehlt die Zuordnung der Adresse ans Postfach, Antworten können also nicht empfangen werden.</p>

<p>Um diesem Postfach eine gültige Absender- und Antwortadresse zuzuordnen, legen
Sie bitte</p>

<ul><li>eine Weiterleitungsregel für diese Adresse auf das Postfach an, z.B. indem Sie die Adresse unten eintragen,</li>
<li><em>oder</em> ändern Sie die Absenderadresse unter <a href="https://mail.noris.net">https://mail.noris.net</a>,
indem Sie diese unter folgendem Menüpunkt eintragen:<br />
Optionen -> Persönliche Informationen -> E-Mail-Adresse</li>
<li>Zum Deaktivieren dieser Warnung geben Sie bitte dem <a href="mailto:support@noris.net">Support</a>
Bescheid, dass er noch die Absendeadresse in unserer Datenbank anpasst.</li>
</ul>
""")

def _get_automatic_password():
    """Gibt ein per pwgen erzeugtes Passwort zurück.
    """
    fp = os.popen("/usr/bin/pwgen -Bnc", "r")
    line = fp.readline()
    fp.close()
    assert(len(line)>7)
    return line[:-1]

def get_new_mailbox_name(kunde_id):
    """Sucht den Namen für eine neue Mailbox heraus
    Regel: Die ersten 3 Buchstaben des Kunden + laufende Nummer
    FIXME: So ist das eine race condition ... das geht während einer Transaktion nicht
    """
    def mailbox_exists(candidate):
        return (models.Person.objects.filter(user=candidate).count() > 0
                or models.Mailrule.objects.filter(quelle=candidate).count() > 0)
    kunde = models.Kunde.objects.filter(id=kunde_id).get()
    prefix = kunde.name[:3]
    try:
        isValidUserName(prefix)
    except validators.ValidationError:
        if kunde.hauptperson and kunde.hauptperson.user:
            prefix = kunde.hauptperson.user[:3]
        else:
            prefix = 'pf'
    for i in count(1):
        candidate = "%s%d" % (prefix,i)
        if not mailbox_exists(candidate):
            return candidate
    raise Exception, "Wir haben das Ende einer unendlichen Schleife erreicht. (1:2^100 und fallend ...)"


class MailboxAddManipulator(oldforms.Manipulator):
    """Spezialisierter Manipulator zum Anlegen/Editieren von Mailboxen
    für einen bestimmten Kunden.

    Eine Mailbox ist im Prinzip eine Person, allerdings werden nur
    `kunde`, `name`, `pass` verwendet. `pwuse` muss das Flag `mail` enthalten.
    Zu der `Person` gehört jeweils noch (exakt) eine `Mailrule` vom typ `virt`,
    die die E-Mail-Adresse definiert, unter der die Mailbox erreichbar ist.
    Diese Mailrule wird ebenfalls von diesem Manipulator behandelt.
    Der domain part in `Mailrule.quelle` muss einer Domain des Kunden entsprechen.
    Die Verknüpfung zwischen beiden Tabellen ist `Mailrule.ziel` = `Person.user`.

    Obacht: Diese Klasse ist auch Oberklasse von MailboxChangeManipulator; `self.change`
    differenziert.

    Verbesserungsvorschlag fürs Datenmodell: Domain und localpart der E-Mail-Adressen in den Mailrules
    sollten getrennt werden, wobei die Domain dann ein foreign key in domainkunde ist. (mir)
    """
    change = False
    typ_virt = models.Mailrule.get_typ_descr("virt")

    # FIXME: statt `kunde` muss hier `kunden` übergeben werden.
    change = False

    def __init__(self, kunde, editor_person, num_extra=1):
        super(MailboxAddManipulator,self).__init__()
        resolver = AsyncDNSResolver()
        if editor_person.is_staff():
            self.allowed_domains = kunde.domainkunde_set.active()
        else:
            self.allowed_domains = editor_person.managed_domainkunden()
        self.kunde = kunde
        self.editor_person = editor_person
        self.fields = [
            ReadonlyTextField("kunden_name"),
            oldforms.TextField("user", max_length=32),
            oldforms.TextField("name", max_length=255, is_required=not self.change or self.original_object.name),
            oldforms.PasswordField("pass_field", max_length=255),
            PasswordConfirmationField("password_confirmal", "pass_field"),
            oldforms.CheckboxField("force_create"),
            oldforms.CheckboxField("auto_password"),
            ]
        if self.change:
            mailrules =  list(self.kunde.mailrule_set.filter(ziel=self.original_object.user).descr_eq("typ","virt"))
            mailrules.sort(key=lambda rule: (domain_sort_key(rule.quelle)))
            self.rel_mailrules = RelatedMailrules(mailrules, self.kunde, editor_person, num_extra)
        else:
            self["pass_field"].validator_list.append(
                validators.RequiredIfOtherFieldNotGiven(
                    "auto_password",
                    _('Bitte ein Passwort angeben oder "Automatische Passwortvergabe" ankreuzen.')))
            self.rel_mailrules = RelatedMailrules([], self.kunde, editor_person, num_extra)
        if editor_person.is_staff():
            self["user"].is_required = self.change
            self["user"].validator_list.append(isValidUserName)
        else:
            self["user"].readonly = True

        mailrules_count = len(self.rel_mailrules.mailrules)
        for i in range(mailrules_count + num_extra):
            prefix = "mailrule.%d." % i
            is_new = i >= mailrules_count
            self.fields.extend([
                LocalpartField(prefix + "quelle_localpart"),
                SubdomainField(prefix + "quelle_subdomain"),
                WildcardDomainComboField(field_name=prefix + "quelle_domain",
                                 data_source = WildcardDomainDataSource(editor_person, kunde.id),
                                 choices_only=not editor_person.is_staff(),
                                 is_required=not is_new,
                                 validator_list=[
                                    validators.RequiredIfOtherFieldGiven(
                                        prefix + "quelle_localpart",
                                        _('Wenn eine E-Mail-Adresse angegeben ist, bitte auch eine Domain angeben.'))]
                                ).add_advisors([checkSourceDomain(self.allowed_domains, editor_person.is_staff(), allow_wildcard=True),
                                                hasResolvableDomain(
                                                    subdomain_field_name=prefix+'quelle_subdomain',
                                                    allow_wildcard=True,
                                                    resolver=resolver)]),
                oldforms.CheckboxField(prefix+"check_remove", readonly=is_new),
                oldforms.HiddenField(field_name=prefix+"quelle")
                ])

    def flatten_data(self):
        """returns a dict of form data (strings) for the corresponding model
        """
        return {'kunden_name': self.kunde.name}

    def get_advisories(self, data):
        advisories = {}
        # Der Inhalt von pass_field ist nicht in `data` enthalten, weil
        # das Passwort ja nicht angezeigt werden soll. Daher können die
        # normalen Advisories nicht verwendet werden, und wir müssen
        # hier alles explizit aufrufen.
        pass_field = data.get("pass_field")
        if not pass_field and self.change:
            pass_field = self.original_object.pass_field
        if pass_field:
            if len(pass_field) < 6:
                advisories["pass_field"] = [_("Das Passwort ist recht kurz.")]
            charsetValidator = hasOnlyLatin1Characters('')
            try:
                charsetValidator(pass_field)
            except validators.ValidationError:
                advisories.setdefault('pass_field', []).append(
                    mark_safe(ugettext_lazy(
                        u"Einige Zeichen des Passworts liegen außerhalb des üblichen Zeichensatzes (iso-latin-1). "
                        u"Es kann nicht garantiert werden, dass der Zugriff mit allen Schnittstellen und Clients möglich ist. "
                        u'Deswegen raten wir, in Passwörtern nur Buchstaben, Ziffern und die Zeichen "<tt>!?=+#,&*/%.()[]{}_$§@;</tt>" zu verwenden.'
                        )))
        elif self.change:
            advisories["pass_field"] = [_("Dieses Postfach hat ein leeres Passwort.")]

        if self.change:
            if not self.composite_rules() and len(self.rel_mailrules.mailrules) == 0:
                advisories['mailrules'] = [_("Es sind keine E-Mail-Adressen angegeben. Wenn dieses Postfach nicht ueber andere Regeln angesprochen wird, kann es keine E-Mail empfangen.")]
            if not self.original_object.email:
                advisories['mailrules'] = advisories.get('mailrules',[]) + [mark_safe(_missing_sender_email)]
            else:
                domains = self.kunde.domainkunde_set.active()
                email = self.original_object.email
                if not any((rule.matches_email(email, domains) for rule in chain(self.composite_rules(), self.rel_mailrules.mailrules))):
                    advisories['mailrules'] = advisories.get('mailrules',[]) + [mark_safe(_deviating_sender_email % {'email': email})]
        return advisories

    def save(self, data):
        user = data.get("user", None)
        if not self.editor_person.is_staff() or not user:
            user = get_new_mailbox_name(self.kunde.id)
        rules = self.save_mailrules(user=user, data=data)
        if len(rules) > 0 and rules[0].quelle and "@" in rules[0].quelle:
            email = rules[0].quelle
        else:
            email = None
        
        params = dict(
            kunde=self.kunde,
            user=user,
            pwuse=models.Person.get_pwuse_bitmask("mail"),
            name=data.get("name"),
            pass_field=data.get("pass_field"),
            email=email,
            )
        if data.get("auto_password"):
            params["pass_field"] = _get_automatic_password()
        new_mailbox = models.Person(**params)
        new_mailbox.save()
        self.editor_person.auth_user().message_set.create(message=_("Es kann aber noch bis zu 10 Minuten dauern, bis die Aenderung aktiv wird."))
        if data.get("auto_password"):
            self.set_mailbox_hint(new_mailbox, rules)
        return new_mailbox

    def get_related_objects(self):
        return [self.rel_mailrules]

    def get_validation_errors(self ,data):
        benutzte_quellen = set()
        errors = super(MailboxAddManipulator, self).get_validation_errors(data)
        rel_data = DotExpandedDict(data)['mailrule']
        for (i, rule_data, rule) in izip(count(),
                                         (rel_data[str(i)] for i in range(len(rel_data))),
                                         chain(self.rel_mailrules.mailrules, repeat(None))):
            if rule_data.get("check_remove"):
                # Von super().get_validation_errors gesetzte Fehlermeldungen entfernen, da diese Regel ohnehin entfernt wird.
                for shortname in rule_data.iterkeys():
                    fullname = "mailrule.%d.%s" % (i, shortname)
                    if fullname in errors:
                        del errors[fullname]
            else:
                errors.update(self.get_rule_validation_errors("mailrule.%d" % i, rule, rule_data, benutzte_quellen))
        return errors

    def get_rule_validation_errors(self, prefix, rule, rule_data, benutzte_quellen):
        if rule_data.get("quelle_localpart") == '*':
            rule_data["quelle_localpart"] = ''
        if rule_data["quelle_domain"]=="*":
            if rule_data.get("quelle_localpart") or rule_data.get("quelle_subdomain"):
                return {"%s.quelle" % prefix: [_('Die Domain "*" bedeutet, dass diese Regel auf alle E-Mails angewandt wird, auf die keine andere Regel passt. Sie ist daher nicht mit weiteren Angaben kombinierbar.')]}
            else:
                other_rule = models.Mailrule.objects.filter(quelle__isnull=True, kunde__id=self.kunde.id)
                if rule:
                    other_rule = other_rule.exclude(id=rule.id)
                if other_rule.exists():
                    return {"%s.quelle" % prefix:
                            [_('Eine Weiterleitungsregel fuer "%(quelle)s" besteht bereits.')
                            % {'quelle': "*"}]}
        else:
            quelle=assemble_address(rule_data.get("quelle_localpart"),
                                    rule_data.get("quelle_subdomain"),
                                    rule_data.get("quelle_domain"))
            if not rule or rule.quelle != quelle or rule_data.get("check_remove"):
                addr = EmailAddress(quelle, True)
                addr.check_domain(self.allowed_domains)
                if not self.editor_person.is_staff() and addr.foreign_domain:
                    if rule:
                        msg = _('Die Domain "%(domain)s" ist Ihnen nicht zuwiesen, deswegen koennen Sie keine keine E-Mail-Adressen aus dieser Domain umleiten.')
                    else:
                        msg = _('Die Domain "%(domain)s" ist Ihnen nicht zuwiesen, deswegen koennen Sie diese Adresse nicht veraendern oder loeschen. Bitte wenden Sie sich fuer diese Aenderung an den Support.')
                    return {"%s.quelle" % prefix: [msg % {'domain': addr.foreign_domain}]}
            if quelle and not rule_data.get("check_remove"):
                if quelle in benutzte_quellen:
                    return {"%s.quelle" % prefix:
                            [_('Die Adresse "%(quelle)s" wird bereits in einer vorangeganenen Regel umgeleitet.')
                             % {'quelle': quelle}]}
                else:
                    benutzte_quellen.add(quelle)
            other_rule = models.Mailrule.objects.filter(quelle=quelle)
            if rule:
                other_rule = other_rule.exclude(id=rule.id)
            if other_rule.exists():
                return {"%s.quelle" % prefix:
                        [_('Eine Weiterleitungsregel fuer "%(quelle)s" besteht bereits.')
                         % {'quelle': quelle}]}
        return {}

    def validate_user(self, field_data, all_data):
        if self.editor_person.is_staff():
            other_persons = models.Person.objects.filter(user=field_data)
            if self.change:
                other_persons = other_persons.exclude(id=self.original_object.id)
            if other_persons.exists():
                other_person = other_persons.get()
                if other_person.kunde_id != self.kunde.id:
                    raise validators.ValidationError(_("Der Benutzername '%(user)s' ist fuer einen anderen Kunden belegt.") % all_data)
                elif self.change or other_person.has_pwuse_flag("mail"):
                    raise validators.ValidationError(_("Ein Postfach mit dem Benutzernamen '%(user)s' besteht bereits.") % all_data)
                else:
                    raise validators.CriticalValidationError(
                        mark_safe(_("Es gibt bereits einen Personeneintrag fuer den Benutzernamen '%(user)s', allerdings ohne Mailflag. "        "Entweder nehmen Sie einen anderen Benutzernamen, oder Sie "
                          "<a href=\"%(set_mailflag_url)s\">setzen das Mailflag furr diesen Personeneintrag</a>"
                         ) % {'user': conditional_escape(field_data),
                              'set_mailflag_url': navi_url('mailadmin.set_mailflag', other_person.navi_context(), True)}))

    def validate_auto_password(self, field_data, all_data):
        if field_data and all_data.get('pass_field'):
            raise validators.ValidationError(
                _(u"Bitte entweder ein Passwort eingeben oder das Passwort automatisch generieren lassen. Beides gleichzeitig ist nicht möglich."))

    def save_mailrules(self, user, data):
        """Speichert die Änderungen in den related mailrules.
        Gibt eine Liste der (neuen) mailrules als [Mailrule, ...] zurück.
        """
        rel_data = DotExpandedDict(data)['mailrule']
        rules = []
        # go through all mailrule rows, put data in rule_data
        # and the corresponding mailrule object into rule, filling in None when
        # we run out of existing mailrules.
        for rule_data, rule in izip((rel_data[str(i)] for i in range(len(rel_data))),
                                    chain(self.rel_mailrules.mailrules, repeat(None))):
            if rule_data.get("quelle_localpart") == "*":
                rule_data["quelle_localpart"] = ''
            if rule_data["quelle_domain"] == "*":
                quelle = None
            else:
                quelle=assemble_address(rule_data.get("quelle_localpart"),
                                        rule_data.get("quelle_subdomain"),
                                        rule_data.get("quelle_domain"))
            if rule:
                if rule_data.get("check_remove"):
                    rule.delete()
                else:
                    rule.kunde = self.kunde
                    rule.quelle = quelle
                    rule.ziel = user
                    rule.save()
                    rules.append(rule)
            elif rule_data.get("quelle_domain") and not rule_data.get("check_remove"):
                params = dict(
                    kunde=self.kunde,
                    quelle=quelle,
                    ziel=user,
                    typ=self.typ_virt,
                    )
                new_rule = models.Mailrule(**params)
                new_rule.save()
                rules.append(new_rule)
        return rules

    def composite_rules(self):
        return []

    def set_mailbox_hint(self, mailbox, rules):
        """Setzt das Attribut `mailbox_hint.
        Dieses wird dann in der View an die Session übergeben und dann in der
        folgenden Ansicht des Postfachs angezeigt.
        """
        email = mailbox.email
        hinweis_dict = {
            'user': mailbox.user,
            'pass_field': mailbox.pass_field,
            'name': mailbox.name,
            'email': email,
            }
        if not email:
            hinweis_dict["email"] = _(u"- keine -")
        weitere_emails = [rule.quelle for rule in rules if rule.quelle and rule.quelle != email]
        if weitere_emails:
            hinweis_dict["weitere_emails"] = (
                _(u"\nWeitere E-Mail-Adressen: ")
                + u"\n                         ".join(weitere_emails))
        else:
            hinweis_dict["weitere_emails"] = ""
        hinweis = _(u"""Zugangsdaten für Ihr Postfach:

Server (POP(3)/WebMail): mail.noris.net
Benutzername:            %(user)s
Passwort:                %(pass_field)s
Realname:                %(name)s
primäre E-Mail-Adresse:  %(email)s
%(weitere_emails)s""") % hinweis_dict
        self.new_mailbox_hint = hinweis


class MailboxChangeManipulator(MailboxAddManipulator):
    change = True
    def __init__(self, obj_key, editor_person):
        self.obj_key = obj_key   # needed by superclasses
        self.original_object = models.Person.objects.filter(id=obj_key).select_related().get()
        self.editor_person = editor_person
        kunde = self.original_object.kunde
        super(MailboxChangeManipulator,self).__init__(kunde, editor_person)

    def do_html2python(self, data):
        """
        Convert the data from HTML data types to Python datatypes, changing the
        object in place. This happens after validation but before storage. This
        must happen after validation because html2python functions aren't
        expected to deal with invalid input.
        """
        super(MailboxChangeManipulator, self).do_html2python(data)
        for i in range(len(self.rel_mailrules.mailrules) + 1):
            fieldname = 'mailrule.%d.quelle_subdomain' % i
            if data.get(fieldname):
                data["mailrules_have_subdomains"] = True
                break

    def flatten_data(self):
        data = dict(
            kunden_name=self.kunde.name,
            user=self.original_object.user,
            name=self.original_object.name,
            )
        for i, rule in enumerate(self.rel_mailrules.mailrules):
            prefix = 'mailrule.%d.' % i
            if rule.quelle==None:
                addr = EmailAddress("*", True)
            else:
                addr = EmailAddress(rule.quelle, domain_required=True)
            addr.check_domain(self.allowed_domains)
            data[prefix + "quelle_localpart"] = addr.localpart
            data[prefix + "quelle_subdomain"] = addr.subdomain
            data[prefix + "quelle_domain"] = addr.parent_domain
            if addr.subdomain:
                data["mailrules_have_subdomains"] = True
            if not self.editor_person.is_staff() and addr.foreign_domain:
                # mailrule is not editable
                self[prefix + "quelle_localpart"].readonly = True
                self[prefix + "quelle_subdomain"].readonly = True
                self[prefix + "quelle_domain"].readonly = True
                self[prefix + "check_remove"].readonly = True
        return data

    def save(self, data):
        if self.editor_person.is_staff():
            user = data.get("user")
            self.original_object.user = user
        else:
            user = self.original_object.user
        self.original_object.name = data.get("name")
        if data.get("auto_password"):
            pass_field = _get_automatic_password()
        else:
            pass_field = data.get("pass_field")
        if pass_field:
            self.original_object.pass_field = pass_field
        self.original_object.save()

        rules = self.save_mailrules(user=user, data=data)
        self.editor_person.auth_user().message_set.create(message=_("Es kann aber noch bis zu 10 Minuten dauern, bis die Aenderung aktiv wird."))
        if data.get("auto_password"):
            self.set_mailbox_hint(self.original_object, rules)
        return self.original_object

    def composite_rules(self):
        if not hasattr(self, "_composite_rules"):
            self._composite_rules = list(self.original_object.composite_rules())
        return self._composite_rules


class RelatedMailrules(object):
    """
    Notwendige Zwischenschicht - stellt die mit einer
    Mailbox verbundenen Mailrules für die Django Forms
    zur Verfügung.

    Normalerweise übernimmt das db.models.related.RelatedObject,
    aber die Art der Zuordnung von Mailrules zu Person entzieht sich
    der reinen relationalen Lehre, da ja mailrules.ziel nicht wirklich
    ein ForeignKey zu person.user ist. Auch weichen die Felder des
    MailboxManipulators vom Datenmodell ab (was aber nicht entscheidend ist).

    RelatedMailrules passt sich ansonsten sehr dem an, was von db.models.related.RelatedObject
    benutzt wird (duck typing)

    FIXME: readonly???

    """
    class DummyField(object):
        """
        Tut so, als ob es ein Datenbank-Feld wär,
        und verweist bei get_manipulator_field_names()
        auf die gewünschten Felder.
        """
        def __init__(self, field_names):
            self.field_names = field_names

        def get_manipulator_field_names(self, dummy):
            return [self.field_names]

    def __init__(self, mailrules, kunde, editor_person, num_extra):
        self.mailrules = mailrules
        self.related_fields = ["quelle_localpart", "quelle_subdomain", "quelle_domain", "check_remove", "quelle"]
        self.name = 'mailrules'
        self.opts = models.Mailrule._meta
        self.num_extra = num_extra


    def extract_data(self, data):
        """Werte der Datenfelder für die Mailrules.
        Man kann problemlos alles zurückgeben.
        """
        return data

    def get_list(self, original_mailbox):
        """Gibt die Mailrules + zusätzlich Dummies für leere Zeilen zurück"""
        return self.mailrules + [None] * self.num_extra

    def editable_fields(self):
        """Hier muss man Plazebos für Datenbank-Felder konstruieren, für
        alle benutzten Felder (nicht nur die editerbaren)."""
        return [self.DummyField(name)
                for name in self.related_fields]

