# -*- encoding: utf-8 -*-
from kundebunt.kunde_auth import LOGIN_URL, REDIRECT_FIELD_NAME, http_auth
from kundebunt.popkern import models
from django.http import HttpResponseRedirect
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.conf import settings
from django.core import urlresolvers
from urlparse import urlsplit

def _redirect_or_forbidden(request, login_url, forbidden):
    """Handles missing access rights or missing logins.

    If the site is configured to use http (basic) authentication, return an authentication challenge.
    Else, redirect to the "access forbidden" page if `forbidden` is True, or the user has already logged in.
    Otherwise, redirect to the login page.
    """
    if settings.AUTHENTICATION_METHOD == "http":
        return http_auth.basic_challenge()
    query = "%s=%s;%s" % (REDIRECT_FIELD_NAME,request.path, request.META["QUERY_STRING"])
    if forbidden or request.person:
        response = render_to_response('kunde_auth/forbidden.html',
            context_instance=RequestContext(request,
                {'login_url': login_url,
                'login_url2': '%s?%s' % (login_url, query)}))
        response.status_code = 403 # forbidden
        return response
    else:
        return HttpResponseRedirect('%s?%s' % (login_url, query))

def check_readonly(*dargs, **dkwargs):
    """Sperrt Schreibzugriff für Accounts mit dem User-Flag `readonly`.

    Als Schreibzugriff wird dabei alles gewertet, was mit POST übermittelt wird. Alternativ kann man den Parameter `allow_get` übergeben, der bestimmt, ob GET-Requests erlaubt sind oder nicht (Default ist `True`.)
    """
    def _dec(view_func):
        def _check(request, *args, **kwargs):
            if (request.method != "GET" or not allow_get) and request.person.has_pwuse_flag("readonly"):
                response = render_to_response('kunde_auth/readonly.html',
                    context_instance=RequestContext(request))
                response.status_code = 403 # forbidden
                return response
            else:
                return view_func(request, *args, **kwargs)
        _check.__name__ = view_func.__name__
        _check.__module__ = view_func.__module__
        return _check
    if "allow_get" in dkwargs:
        allow_get = dkwargs["allow_get"]
        return _dec
    else:
        allow_get = True
        return _dec(*dargs)




def user_passes_test(test_func, login_url=LOGIN_URL, forbidden=False):
    """
    Decorator for views that checks that the user passes the given test,
    redirecting to the log-in page if necessary. The test should be a callable
    that takes the user object and returns True if the user passes.

    If `forbidden` is True or the user has already logged in,
    it will return an "access forbidden" page with status 403
    instead of the redirect. Use it for web services and for pages with restricted access.
    """
    def _dec(view_func):
        def _checklogin(request, *args, **kwargs):
            if test_func(request.user):
                return view_func(request, *args, **kwargs)
            else:
                return _redirect_or_forbidden(request, login_url, forbidden)
        _checklogin.__name__ = view_func.__name__
        _checklogin.__module__ = view_func.__module__
        return _checklogin
    return _dec

login_required = user_passes_test(lambda u: not u.is_anonymous())
login_required.__doc__ = (
    """
    Decorator for views that checks that the user is logged in, redirecting
    to the log-in page if necessary.
    """
    )


def person_passes_test(test_func, login_url=LOGIN_URL, forbidden=False):
    """
    Decorator for views that checks that the user passes the given test,
    redirecting to the log-in page if necessary. The test should be a callable
    that takes the user object and returns True if the user passes.

    If `forbidden` is True or the user has already logged in (but the test failed),
    it will return an "access forbidden" page with status 403
    instead of the redirect. Use it for web services and for pages with restricted access.
    """
    def _dec(view_func):
        def _checklogin(request, *args, **kwargs):
            if test_func(request.person):
                return view_func(request, *args, **kwargs)
            else:
                return _redirect_or_forbidden(request, login_url, forbidden)
        _checklogin.__name__ = view_func.__name__
        _checklogin.__module__ = view_func.__module__
        return _checklogin
    return _dec

def pwuse_required(flagname):
    """
    Decorator for views that checks whether the logged in person has a given flag set in pwuse,
    redirecting to the log-in page if necessary
    """
    return person_passes_test(lambda p: p != None and p.has_pwuse_flag(flagname))

def person_has_service_association(person):
    """
    Gibt zurück ob die Person mindestens eine Assoziation in kundemail mit Dienst 'service' hat.

    person: models.Person-Objekt. None ist auch erlaubt (dann wird aber False zurückgegeben).
    """
    return person_has_association(person, ['service'])

def person_has_association(person, dienst_namen):
    if person==None:
        return False
    else:
        return models.Kunde.objects.managed_by(person.id, dienst_namen).exists()

def person_can_log_in(person):
    if person==None:
        return False
    else:
        return person_has_association(person, ['service', 'rtweb'])

with_service_association = service_flag_required = person_passes_test(person_has_service_association)
with_service_association_webservice = person_passes_test(person_has_service_association, forbidden=True)

staff_only = person_passes_test(lambda person: person != None and person.is_staff() and person_has_service_association(person))
staff_only_webservice = person_passes_test(
        lambda person: person != None and person.is_staff() and person_has_service_association(person),
        forbidden=True)
