# -*- encoding: utf-8 -*-

"""Dieses Modul enthält die Views zur Hardware-Behandlung.
"""
import datetime
from itertools import chain, groupby
from math import ceil

from django import oldforms
from django.http import HttpResponse, Http404, HttpResponseRedirect, QueryDict
from django.template import loader, RequestContext
from django.shortcuts import render_to_response
from django.views.generic.create_update import update_object, create_object, delete_object
from django.conf import settings
from django.utils.encoding import smart_str
from django.utils.http import urlencode
from django.db.models.query import Q

from kundebunt.popkern import models
from kundebunt.popkern.fields import PopQuerySet
from kundebunt.popkern.navigation import navi_url, navi_url_pattern
from kundebunt.popkern.utils import domain_sort_key, partition_dict, time_to_str
from kundebunt.kunde_auth.decorators import service_flag_required, staff_only, check_readonly
from kundebunt.kunde.forms import *
from kundebunt.hardware.views import hw_search
from kundebunt.popkern.utils import any, all
from kundebunt.history import get_kunde_history

_prio_colors = { 'A-Kunde': "red", 'B-Kunde': 'yellow', 'C-Kunde': 'green'}

@staff_only
def kunden_suche(request):
    #if not request.person.is_staff():
        #request.user.message_set.create(message=_('Diese Funktion steht vorerst nur fuer noris-network-Mitarbeiter zur Verfuegung.'))
        #return HttpResponseRedirect(navi_url('mailadmin', {}, True))
    manipulator = KundenSucheForm()
    if request.GET:
        data = request.GET
        treffer_listen = manipulator.search_results(data)
        if sum(tl.count() for tl in treffer_listen) > 0:
            all_treffer_it = chain(* (tl.treffer() for tl in treffer_listen))
            treffer = all_treffer_it.next()
            tkid = treffer.kunde().id
            if all(t.kunde().id == tkid for t in all_treffer_it):
                # Alle Treffer beziehen sich auf denselben Kunden!
                return HttpResponseRedirect(navi_url("kunde.overview", treffer.kunde().navi_context(), True))
    else:
        data = manipulator.flatten_data()
        treffer_listen = []
    return render_to_response(
        "kunde/kunden_suche.html",
        context_instance=RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, data, {}),
            "treffer_listen": treffer_listen,
            "keine_treffer": sum([treffer_liste.count() for treffer_liste in treffer_listen]) == 0,
            "mit_suche": request.GET,
            }))


def _by_log(x):
    return x["log_id"]

@staff_only
def overview(request, kunden_name):
    if kunden_name is None:
        kunde_id = request.person.kunde_id
        kunde = request.person.kunde
    try:
        kunde = models.Kunde.objects.get(name=kunden_name)
        kunde_id = kunde.id
    except models.Kunde.DoesNotExist:
        return HttpResponseRedirect(navi_url("kunde.kunden_suche", {}, True) + "?" + urlencode({'search': kunden_name}))
    unterkunden = list(kunde.unterkunde.active())
    if request.GET:
        mit_unterkunden = "mit_unterkunden" in request.GET
    else:
        mit_unterkunden = True
    manipulator = KundeOverviewForm(kunde_id)
    kunden = [kunde_id]
    mit_unterkunden_value = ''
    if mit_unterkunden:
        kunden.extend(k.id for k in unterkunden)
        if len(kunden) > 1:
            mit_unterkunden_value = 'on'

    # tickets
    visible_queue_ids = models.Queue_acl.objects.filter(person=request.person).has_flag("acls","display").valuelist("queue")
    anzahl_tickets = (
        models.Ticket.objects.filter(kunde__in=kunden, queue__in=visible_queue_ids)
                             .descr_eq('status','open')
                             .extra(where=['ticket.id=ticket.ticket'])
                             .count())
    if anzahl_tickets > 0:
        ticket_query = QueryDict('', mutable=True)
        try:
            default_query = QueryDict(models.Rt_defaults.objects.get(person=request.person).query, mutable=True)
            for param in ['q_sort', 'q_fields', 'fup', 'q_limit', 'refresh', 'fup', 'nst']:
                if param in default_query:
                    ticket_query[param] = default_query[param]
        except models.Rt_defaults.DoesNotExist:
            pass
        ticket_query['q_status'] = 'open'
        ticket_query['q_queue_all'] = 'true'
        ticket_query['q_owner_all'] = 'true'
        if mit_unterkunden and len(kunden) > 1:
            ticket_query['q_kunde'] = ','.join(smart_str(k.name) for k in models.Kunde.objects.filter(id__in=kunden))
        else:
            ticket_query['q_kunde'] = kunden_name
        ticket_query = ticket_query.urlencode()
    else:
        ticket_query = None

    # personen
    personen_spalten, personen_sort, personen = PersonenForm(kunden, "dienste", None).results(
        {'mit_unterkunden': mit_unterkunden})
    anzahl_personen = len(personen)
    personen = [p for p in personen if "contact" in p.spalten_set or hasattr(p, "css_class")]
    personen_unvollstaendig = len(personen) != anzahl_personen or anzahl_personen > 10
    

    # accounts
    accounts_spalten, accounts_sort, accounts = PersonenForm(kunden, "flags", None).results(
        {'mit_unterkunden': mit_unterkunden})
    accounts = sorted(accounts, key=lambda account:(-sum((1 for x in account.spalten if x is not False)), account.user, account,id))
    accounts_unvollstaendig = len(accounts) > 10
    
    # domains
    hide_domain_status = [d.descr for d in chain(models.Domainkunde.get_status_descr_in_group('ignore'),
                                                 models.Domainkunde.get_status_descr_in_group('hide'))]
    domains = list(models.Domainkunde.objects.filter(kunde__in=kunden).active().extra(
        select={'rulecount': u"""(select count(*) from mailrules m where m.kunde in %s
                                 and m.quelle regexp concat('([@.]|^)',domainkunde.domain, '$'))"""
                             % u"(%s)" % u",".join(unicode(k) for k in kunden)},
        ).exclude(status__in=hide_domain_status))
    domains.sort(key=lambda d:(-d.rulecount, d.domain, d.id))
    domains_unvollstaendig = len(domains) > 10

    # hardware
    hardware = list(models.Hardware.objects.filter(kunde__in=kunden).active().select_related(follow_fields=["ip", "standort", ["rack", "rz"]]))
    hardware.sort(key=lambda hw: (hw.ip_id and 1 or 2,
                                  hw.hardware_id and 1 or 2,
                                  hw.rack_id and -(hw.he or 0) or 1,
                                  hw.hardware_id,
                                  hw.id))
    hardware_unvollstaendig = len(hardware) > 10

    # ips
    ips = []
    ip_qset = models.Ipkunde.objects.filter(kunde__in=kunden).active().filter(ip6__startswith=models.ipv4_prefix).order_by('-bits', 'id')
    anzahl_ips = ip_qset.count()
    for ip in ip_qset.iterator():
        if not any(((ip in other_ip) for other_ip in ips)):
            ips.append(ip)
            if len(ips) >= 11:
                break
    if len(ips) >= 11:
        ips = ips[:10]
    ips_unvollstaendig = len(ips) != anzahl_ips
    anzahl_ips = ip_qset.count()

    # history
    if settings.ADD_CUSTOMER_HISTORY:
        db_kunde = models.DbTabelle.objects.get(name='kunde')
        history = list(get_kunde_history(kunden, datetime.datetime.now(), datetime.timedelta(1), 11))
    else:
        history = []

    return render_to_response(
        "kunde/overview.html",
        context_instance=RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, {'mit_unterkunden': mit_unterkunden}, {}),
             "kunden_name": manipulator.kunden_name(),
             "ueberkunde": manipulator.ueberkunde(),
             "hat_unterkunden": manipulator.hat_unterkunden(),
             "unterkunden": unterkunden,
             "abgelaufen": manipulator.abgelaufen(),
             "ap_technik": kunde.ap_technik,
             "ap_vertrieb": kunde.ap_vertrieb,
             "prio": kunde.kprio and kunde.kprio.bla,
             "prio_class": kunde.kprio and "prio_%s" % chr(kunde.kprio.descr),
             "kunde_id": kunde_id,

             "anzahl_tickets": anzahl_tickets,
             "ticket_query": ticket_query,
             "personen": personen,
             "anzahl_personen": anzahl_personen,
             "personen_unvollstaendig": personen_unvollstaendig,
             'accounts': accounts,
             'accounts_unvollstaendig': accounts_unvollstaendig,
             'domains': domains,
             'domains_unvollstaendig': domains_unvollstaendig,
             'hardware': hardware,
             'hardware_unvollstaendig': hardware_unvollstaendig,
             'ips': ips,
             'ips_unvollstaendig': ips_unvollstaendig,
             "anzahl_ips": anzahl_ips,
             "mit_unterkunden_value": mit_unterkunden_value,
             "history": history,
             "has_monitoring": kunde.has_flags_flag("alarm2hotline"),
            })
        )

@staff_only
def overview_by_id(request, kunde_id):
    try:
        kunde = models.Kunde.objects.get(pk=kunde_id)
    except models.Kunde.DoesNotExist:
        raise Http404
    return HttpResponseRedirect(redirect_to=navi_url("kunde.overview", kunde.navi_context(), True))

@staff_only
def unterkunden(request, kunden_name):
    kunden = models.Kunde.objects.active().filter(kunde__name__exact=kunden_name).order_by("name")
    return render_to_response(
        "kunde/unterkunden.html",
        context_instance=RequestContext(request,
            {"kunden_name": kunden_name,
             "kunden": kunden,
            })
        )

@staff_only
def personen(request, kunden_name, filter_spalte=None, modus="dienste"):
    try:
        kunde = models.Kunde.objects.get(name=kunden_name)
    except models.Kunde.DoesNotExist:
        raise Http404
    kunden = [kunde.id]
    kunden.extend(k.id for k in kunde.unterkunde.active())
    manipulator = PersonenForm(kunden, modus, filter_spalte)
    if request.GET:
        data = request.GET
    else:
        data = manipulator.flatten_data()
    spalten, sortier_spalte, personen = manipulator.results(data)
    if modus=="dienste":
        base_url = navi_url("kunde.personen", kunde.navi_context(), True)
        other_url = navi_url("kunde.accounts", kunde.navi_context(), True)
        titel = _("Personen zu")
        template = "kunde/personen.html"
    else:
        base_url = navi_url("kunde.accounts", kunde.navi_context(), True)
        other_url = navi_url("kunde.personen", kunde.navi_context(), True)
        titel = _("Benutzerkonten zu")
        template = "kunde/accounts.html"
    return render_to_response(
        template,
        context_instance=RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, data, {}),
             "filter_spalte": filter_spalte,
             "sortier_spalte": sortier_spalte,
             "spalten": spalten,
             "personen": personen,
             "kunden_name": kunden_name,
             "icon_base": "/kundebunt/static/icons/dienst",
             "base_url": base_url,
             "other_url": other_url,
             "titel": titel,
             "modus": modus,
             "hat_unterkunden": len(kunden) > 1,
            })
        )

@staff_only
def accounts(request, kunden_name, filter_spalte=None):
    return personen(request, kunden_name, filter_spalte, modus="accounts")

@staff_only
def person_redirect(request, person_id_or_username):
    try:
        if person_id_or_username.isdigit():
            person = models.Person.objects.get(id=person_id_or_username)
        else:
            person = models.Person.objects.get(user=person_id_or_username)
    except models.Person.DoesNotExist:
        raise Http404
    return HttpResponseRedirect(navi_url('kunde.person', person.navi_context(), True))

@staff_only
@check_readonly
def person(request, kunden_name, person_id):
    try:
        manipulator = PersonForm(person_id, request.person)
    except models.Person.DoesNotExist:
        raise Http404
    return update_object(request,
        models.Person,
        template_name="kunde/person.html",
        post_save_redirect=navi_url("kunde.person", manipulator.original_object.navi_context(), True),
        manipulator=manipulator,
        object_id = person_id,
        extra_context = dict(id=person_id)
        )

@staff_only
@check_readonly
def create_person(request, kunden_name=None):
    try:
        if kunden_name is None:
            kunde = None
        else:
            kunde = models.Kunde.objects.get(name=kunden_name)
    except models.Kunde.DoesNotExist:
        raise Http404
    manipulator = PersonAddForm(kunde, request.person)
    return create_object(
        request,
        models.Person,
        template_name="kunde/person.html",
        manipulator=manipulator
    )

@staff_only
def domains(request, kunden_name):
    mit_unterkunden = True
    form = MitUnterkundenForm(request.GET or None)
    if form.is_valid():
        mit_unterkunden = form.cleaned_data['mit_unterkunden']
    kunden_namen = [kunden_name]
    unterkunden_namen = list(models.Kunde.objects.active().filter(kunde__name__exact=kunden_name).valuelist("name"))
    if mit_unterkunden:
        kunden_namen.extend(unterkunden_namen)
    domains = list(models.Domainkunde.objects.active().filter(kunde__name__in=kunden_namen).select_related(follow_fields=(("nserver",),)))
    domains.sort(key=lambda dom:domain_sort_key(dom.domain))
    return render_to_response(
        "kunde/domains.html",
        context_instance=RequestContext(request,
            {"form": form,
             "kunden_name": kunden_name,
             "domains": domains,
             "hat_unterkunden": len(unterkunden_namen) > 0,
            })
        )

@staff_only
def ips(request, kunden_name):
    mit_unterkunden = True
    form = MitUnterkundenForm(request.GET or None)
    if form.is_valid():
        mit_unterkunden = form.cleaned_data['mit_unterkunden']
    kunden_namen = [kunden_name]
    unterkunden_namen = list(models.Kunde.objects.active().filter(kunde__name__exact=kunden_name).valuelist("name"))
    if mit_unterkunden:
        kunden_namen.extend(unterkunden_namen)
    ips = models.Ipkunde.objects.active().filter(kunde__name__in=kunden_namen).filter(ip6__startswith=models.ipv4_prefix).order_by("ip6", "-bits")
    return render_to_response(
        "kunde/ips.html",
        context_instance=RequestContext(request,
            {"form": form,
            "kunden_name": kunden_name,
             "ips": ips,
             "hat_unterkunden": len(unterkunden_namen) > 0,
            })
        )

@staff_only
def hardware(request, kunden_name):
    return hw_search(request, in_panel=True, kunden_name=kunden_name)

def _parse_kunden(s):
    return [int(x) for x in s.split(u",")]

