# -*- coding: UTF-8 -*-

import datetime, re 
from itertools import groupby, chain, islice

from django import oldforms, newforms
from django.conf import settings
from django.core import validators
from django.db.models.query import Q
from django.utils.encoding import smart_str
from django.utils.safestring import mark_safe
from django.utils.html import conditional_escape
from django.utils.translation import ugettext
from django.contrib.auth.models import User

from kundebunt.popkern import models
from kundebunt.popkern.datasources import KundenDataSource, LaenderDataSource
from kundebunt.popkern.utils import time_to_str, glob_to_like, glob_to_regex, any
from kundebunt.popkern.navigation import navi_url
from kundebunt.popkern.formfields import *

__all__ = ['KundeOverviewForm', 'PersonenForm', 'PersonForm', 'PersonAddForm', 'KundenSucheForm', 'MitUnterkundenForm']

_domain_id_re = re.compile(ur'^domain#(\d+)$')
_rechnung_nr_re = re.compile(ur'^R(\d+)')

MAX_TREFFER = 100

class KundeOverviewForm(oldforms.Manipulator):
    def __init__(self, kunde_id):
        self.kunde = models.Kunde.objects.get(pk=kunde_id)
        self.fields = [
            oldforms.CheckboxField("mit_unterkunden"),
        ]

    def ueberkunde(self):
        """Gibt das model des übergeordneten Kunden zurück (oder None)"""
        try:
            return self.kunde.kunde
        except AttributeError:
            return None

    def hat_unterkunden(self):
        """Gibt zurück, ob dieser Kunde Unterkunden hat."""
        return models.Kunde.objects.active().filter(kunde__exact=self.kunde.id).exists()

    def kunden_name(self):
        return self.kunde.name

    def abgelaufen(self):
        return time_to_str(self.kunde.abgelaufen(), '%Y-%m-%d')

    def flatten_data(self):
        return {"mit_unterkunden": True}

class PersonenForm(oldforms.Manipulator):

    important_dienste = (set((x.descr for x in models.Kundemail.objects.all_descr("dienst")))
                         - set((x.descr for x in models.Kundemail.get_dienst_descr_in_group("hide"))))


    def __init__(self, kunden_ids, modus, filter_spalte):
        self.kunden_ids = kunden_ids
        self.fields = [SearchField("suche"),
                       oldforms.CheckboxField("mit_unterkunden"),
                      ]
        self.modus = modus
        self.filter_spalte = filter_spalte

    def flatten_data(self):
        return { 'suche': '*', 'mit_unterkunden': True }

    def results(self, data):
        #kunden = models.Kunde.objects.filter(id__in=self.kunden_ids)
        #hauptperson_ids = set(kunde.hauptperson_id for kunde in kunden if kunde.hauptperson_id is not None)
        #billc_ids = set(kunde.billc_id for kunde in kunden if kunde.billc_id is not None)
        if data.get('mit_unterkunden', False):
            kunden_ids = self.kunden_ids
        else:
            kunden_ids = self.kunden_ids[:1]
        hauptkunde =  models.Kunde.objects.get(id=kunden_ids[0])
        qset = models.Person.objects
        nachfilter = lambda person: True
        if data.get("suche", "*") not in ("*", "", None):
            suche = data["suche"]
            if suche != "-":
                qset = qset.default_query(suche)
                suche_re = glob_to_regex(suche)
                nachfilter = lambda person: person and (
                    person.user and suche_re.match(person.user)
                    or person.name and models.Person.vergleiche_namen(suche, person.name))
            else:
                qset = qset.filter(Q(name__isnull=True)|Q(user__isnull=True))
        personen = {}
        anzeige_spalten = set()
        if self.modus == "dienste":
            if hauptkunde.hauptperson_id is None:
                hauptperson_ids = set()
            else:
                hauptperson_ids = set([hauptkunde.hauptperson_id])
            if hauptkunde.billc_id is None:
                billc_ids = set()
            else:
                billc_ids = set([hauptkunde.billc_id])
            sortier_spalte = data.get("order", "name")
            hide_dienste = list(models.Kundemail.get_dienst_descr_in_group("hide"))
            hide_dienst_codes = [d.descr for d in hide_dienste]
            hide_spalten = [d.bla for d in hide_dienste]
            important_flags = (set((x.bla for x in models.Person.objects.all_descr("pwuse")))
                               - set((x.bla for x in models.Person.get_pwuse_descr_in_group("hide"))))
            qset = (qset.active().filter(kundemail__kunde__in=kunden_ids).distinct().select_related(follow_fields=["kunde"])
                        .extra(select={'dienst': 'person__kundemail.dienst'}))
            for person in qset:
                if nachfilter(person):
                    if person.dienst not in hide_dienst_codes:
                        spalte = models.Kundemail.get_dienst_descr_obj(person.dienst).bla
                        anzeige_spalten.add(spalte)
                        t_person = personen.get(person.id)
                        if t_person:
                            t_person.spalten_set.add(spalte)
                        else:
                            person.spalten_set = set([spalte])
                            personen[person.id] = person
            for hauptperson_id in hauptperson_ids:
                hauptperson = personen.get(hauptperson_id)
                if not hauptperson:
                    hauptperson = models.Person.objects.get(id=hauptperson_id)
                    if nachfilter(hauptperson):
                        hauptperson.spalten_set = set()
                        personen[hauptperson_id] = hauptperson
                hauptperson.css_class = "hauptperson"
            for billc_id in billc_ids:
                billc = personen.get(billc_id)
                if not billc:
                    billc = models.Person.objects.get(id=billc_id)
                    if nachfilter(billc):
                        billc.spalten_set = set()
                        personen[billc_id] = billc
                billc.css_class = "billc"
        else:
            # account-Modus
            hauptperson_ids = billc_ids = set()
            sortier_spalte = data.get("order", "user")
            hide_spalten = set((d.bla for d in models.Person.get_pwuse_descr_in_group("hide")))
            personen_mit_diensten = set(
                    models.Kundemail.objects.filter(
                        dienst__in=self.important_dienste,
                        kunde__in=kunden_ids).valuelist("person"))
            qset = qset.active().filter(kunde__in=kunden_ids).distinct().select_related(follow_fields=("kunde",))
            for person in qset:
                if nachfilter(person):
                    spalten = set((descr.bla for descr in person.pwuse_set())) - hide_spalten
                    if spalten:
                        person.spalten_set = spalten
                        personen[person.id] = person
                        anzeige_spalten.update(spalten)
        anzeige_spalten = sorted(anzeige_spalten - set(hide_spalten))
        if self.modus == "dienste":
            anzeige_spalten.append("flags")
        else:
            anzeige_spalten.append("dienste")

        treffer = []
        spezial_ids = hauptperson_ids
        spezial_ids.union(billc_ids)
        for person in personen.itervalues():
            if not self.filter_spalte or self.filter_spalte in person.spalten_set or (self.modus=="dienste" and person.id in spezial_ids):
                if self.modus=="dienste":
                    if person.kunde_id in kunden_ids and any(flag.bla in important_flags for flag in person.pwuse_set()):
                        person.spalten_set.add("flags")
                else:
                    if person.id in personen_mit_diensten:
                        person.spalten_set.add("dienste")
                treffer.append(person)

        def sort_by_name(p):
            if p.id in hauptperson_ids:
                group = 0
            elif p.id in billc_ids:
                group = 2
            else:
                group = 4
            name = p.name
            if name and name[:5] in ("Frau ", "Herr "):
                namen = name.split()
                return (group, " ".join(namen[-1:] + namen[1:-1]).lower(), p.id)
            elif name:
                return (group, name.lower(), p.id)
            else:
                return (group + 1, p.user or '', p.id)
        def sort_by_user(p):
            if p.id in hauptperson_ids:
                group = 0
            elif p.id in billc_ids:
                group = 2
            else:
                group = 4
            if p.user:
                return (group, p.user.lower())
            else:
                return (group + 1, p.id)
        def sort_by_email(p):
            if p.id in hauptperson_ids:
                group = 0
            elif p.id in billc_ids:
                group = 2
            else:
                group = 4
            if p.email:
                return (group, p.email.lower())
            else:
                return (group + 1, p.id)

        sort_key_dict = {"name": sort_by_name, "user": sort_by_user, "email": sort_by_email}
        treffer = sorted(treffer, key=sort_key_dict.get(sortier_spalte, sort_by_name))
        for person in treffer:
            person.spalten = [(spalte in person.spalten_set and spalte)
                              for spalte in anzeige_spalten]
        return anzeige_spalten, sortier_spalte, treffer


class PersonForm(oldforms.Manipulator):
    change = True
    def __init__(self, obj_key, editor_person):
        self.obj_key = obj_key
        if self.change:
            self.original_object = models.Person.objects.get(pk=obj_key)
        super(PersonForm, self).__init__()
        self.fields = [
            KundenComboField(person=editor_person, field_name="kunde", data_source=KundenDataSource(editor_person), is_required=True),
            oldforms.TextField("name", max_length=255),
            oldforms.TextField("abt", max_length=255),
            oldforms.LargeTextField("strasse", max_length=250, rows=2),
            oldforms.TextField("plz", max_length=10),
            oldforms.TextField("ort", max_length=100,
                               validator_list=[validators.RequiredIfOtherFieldsGiven(
                                ("strasse", "plz", "land"),
                                _(u"Wenn eine Straße, eine PLZ oder ein Land eingegeben wird, bitte auch den Ort angeben."))]),
            ComboField("land", data_source=LaenderDataSource(), min_query_length=0),
            oldforms.TextField("fon", max_length=255),
            oldforms.TextField("fax", max_length=255),
            oldforms.TextField("pager", max_length=255),
            EmailField("email", max_length=255, validator_list=[validators.isValidEmail, hasNoDollar]),
            LocalpartField("user", max_length=32, validator_list=[isValidUserName, hasNoDollar], readonly=True),
            oldforms.TextField("id", readonly=True),
            oldforms.TextField("homedir", max_length=255, readonly=True),
            oldforms.TextField("uremip", readonly=True),
            oldforms.TextField("pwuse", readonly=True),
            oldforms.TextField("timestamp", readonly=True),
            oldforms.TextField("zusatzinfo", max_length=255),
            oldforms.TextField("suchbegriff", max_length=255),
            oldforms.TextField("job", max_length=255),
            oldforms.TextField("infotext", max_length=255),
            oldforms.TextField("ausweisnr", max_length=255),
            oldforms.DateField("geburtstag")
        ]

    def flatten_data(self):
        obj = self.original_object
        return dict(
            kunde=obj.kunde.name,
            name=obj.name,
            abt=obj.abt,
            strasse=obj.adresse and obj.adresse.strasse,
            plz=obj.adresse and obj.adresse.plz,
            ort=obj.adresse and obj.adresse.ort,
            land=obj.adresse and obj.adresse.land and obj.adresse.land.name,
            fon=obj.fon,
            fax=obj.fax,
            pager=obj.pager,
            email=obj.email,
            user=obj.user,
            id=obj.id,
            homedir=obj.homedir,
            uremip=obj.uremip_id and ("#%s: %s" % (obj.uremip_id, obj.uremip_id and obj.uremip.ipaddr_num_display())),
            pwuse=",".join([d.bla for d in obj.pwuse_set()]),
            zusatzinfo=obj.zusatzinfo,
            suchbegriff=obj.suchbegriff,
            job=obj.job,
            infotext=obj.infotext,
            ausweisnr=obj.ausweisnr,
            geburtstag=obj.gebjahr and datetime.date(year=obj.gebjahr, month=obj.gebtag // 100, day=obj.gebtag % 100) or None,
            timestamp=time_to_str(obj.timestamp))
    
    def validate_suchbegriff(self, field_data, all_data):
        other_persons = models.Person.objects.filter(suchbegriff=field_data)
        if self.change:
            other_persons = other_persons.exclude(id=self.original_object.id)
        if other_persons.exists():
            other_person = other_persons.get()
            raise validators.ValidationError(_(u"Der Suchbegriff '%(suchbegriff)s' ist bereits für eine andere Person belegt.") % all_data)


    def save_address(self, adr, data):
        adr_keys = ("strasse", "plz", "ort", "land")
        if any(v for (k,v) in data.items() if k in adr_keys):
            if data["land"]:
                try:
                    land = models.Land.objects.get(name=data["land"])
                except models.Land.DoesNotExist:
                    land = models.Land.objects.get(kurz=data["land"])
            else:
                land = models.Land.objects.get(name="Deutschland")
            if adr and adr.strasse==data["strasse"] and adr.plz==data["plz"] and adr.ort==data["ort"] and adr.land_id==land.id:
                # Adresse wurde nicht verändert, beibehalten.
                return adr
            if adr and not models.Person.objects.filter(adresse__id=adr.id).exclude(pk=self.obj_key).exists():
                # Die Adresse wird noch nicht benutzt und kann geändert werden.
                adr.strasse = data["strasse"]
                adr.plz = data["plz"]
                adr.ort = data["ort"]
                adr.land = land
                adr.save()
                return adr
                
            # Neue Adresse anlegen
            params = dict(((smart_str(k),v) for (k,v) in data.items() if k in adr_keys))
            params["land"] = land
            adr = models.Adresse(**params)
            adr.save()
            return adr
        else:
            return None
        
    def save(self, data):
        person = self.original_object
        old_adr = person.adresse
        adr = self.save_address(old_adr, data)
        person.kunde = models.Kunde.objects.get(name=data["kunde"])
        person.name = data["name"]
        person.abt = data["abt"] or None
        person.fon = data["fon"] or None
        person.fax = data["fax"] or None
        person.pager = data["pager"] or None
        person.email = data["email"] or None
        person.zusatzinfo = data["zusatzinfo"] or None
        person.suchbegriff = data["suchbegriff"] or None
        person.job = data["job"] or None
        person.infotext = data["infotext"] or None
        person.ausweisnr = data["ausweisnr"] or None
        person.adresse = adr
        if data["geburtstag"]:
            person.gebjahr = data["geburtstag"].year
            person.gebtag = data["geburtstag"].month * 100 + data["geburtstag"].day
        else:
            person.gebjahr = person.gebtag = None
        person.save()
        if old_adr and not adr:
            if not models.Person.objects.filter(adresse=old_adr).exclude(id=person.id).exists():
                old_adr.delete()
        return person


class PersonAddForm(PersonForm):
    change = False

    def __init__(self, kunde, editor_person):
        self.kunde = kunde
        super(PersonAddForm, self).__init__(None, editor_person)

    def flatten_data(self):
        if self.kunde is None:
            kunden_name = None
        else:
            kunden_name = self.kunde.name
        return dict(
            kunde=kunden_name
        )

    def save(self, data):
        if data.get("kunde"):
            kunde = models.Kunde.objects.get(name=data["kunde"])
        else:
            kunde = self.kunde
        if data["geburtstag"]:
            gebjahr = data["geburtstag"].year
            gebtag = data["geburtstag"].month * 100 + data["geburtstag"].day
        else:
            gebjahr = gebtag = None
        person = models.Person.objects.create(
            kunde = kunde,
            name = data["name"],
            adresse = self.save_address(None, data),
            gebjahr = gebjahr,
            gebtag = gebtag,
            abt = data["abt"] or None,
            fon = data["fon"] or None,
            fax = data["fax"] or None,
            pager = data["pager"] or None,
            email = data["email"] or None,
            zusatzinfo = data["zusatzinfo"] or None,
            suchbegriff = data["suchbegriff"] or None,
            job = data["job"] or None,
            infotext = data["infotext"] or None,
            ausweisnr = data["ausweisnr"] or None,
            pwuse = 0,
            )
        models.Kundemail.objects.create(
            kunde = kunde,
            person = person,
            dienst = models.Kundemail.get_dienst_descr("contact"),
            dringend = 0
        )
        return person


class KundenSucheForm(oldforms.Manipulator):

    class TrefferListe(object):
        def __init__(self, h2, th, qset, filter, treffer_class):
            self.h2 = h2
            self.th = th
            self.qset = qset
            self.treffer_class = treffer_class
            self.filter = filter
            treffer = list(islice((x for x in self.qset if filter==None or filter(x)), MAX_TREFFER + 1))
            self.ist_gekuerzt = len(treffer) > MAX_TREFFER
            if self.ist_gekuerzt:
                treffer = treffer[:MAX_TREFFER]
            self._treffer = treffer
                

        def __repr__(self):
            return '<TrefferListe(%s): %r' % (self.treffer_class, self._treffer)

        def treffer(self):
            return (self.treffer_class(x) for x in self._treffer)

        def count(self):
            return len(self._treffer)

        #def __iter__(self):
            #return self.treffer().__iter__()

    class KundenNameTreffer(object):
        def __init__(self, kunde):
            self._kunde = kunde

        def kunde(self):
            return self._kunde

        def suchfeld(self):
            return " "

    class KundenNrTreffer(KundenNameTreffer):
        def suchfeld(self):
            return self._kunde.id

    class DomainIdTreffer(object):
        def __init__(self, domain):
            self._domain = domain

        def kunde(self):
            return self._domain.kunde

        def suchfeld(self):
            return self._domain.domain

    class RechnungsNrTreffer(object):
        def __init__(self, rechnung):
            self._rechnung = rechnung

        def kunde(self):
            return self._rechnung.kunde

        def suchfeld(self):
            return self._rechnung.rnr

    class PersonenUserTreffer(object):
        def __init__(self, person):
            self._person = person

        def kunde(self):
            return self._person.kunde

        def suchfeld(self):
            return self._person.user

        def url(self):
            return navi_url('kunde.person', self._person.navi_context(), True)

    class PersonenNameTreffer(PersonenUserTreffer):
        def suchfeld(self):
           return self._person.name

    class PersonenEmailTreffer(PersonenUserTreffer):
        def suchfeld(self):
            return self._person.email

    class DomainTreffer(object):
        def __init__(self, domainkunde):
            self._domainkunde = domainkunde

        def kunde(self):
            return self._domainkunde.kunde

        def suchfeld(self):
            return self._domainkunde.domain

    class KundenAliasTreffer(object):
        def __init__(self, uucpkunde):
            self._uucpkunde = uucpkunde

        def kunde(self):
            return self._uucpkunde.kunde

        def suchfeld(self):
            return self._uucpkunde.name

    class IPAddressTreffer(object):
        def __init__(self, ipkunde):
            self._ipkunde = ipkunde

        def kunde(self):
            return self._ipkunde.kunde

        def suchfeld(self):
            return self._ipkunde.ipaddr_num_display()

        def url(self):
            hw = len(self._ipkunde.hardware_set.active())
            if len(hw==1):
                return navi_url("hardware.edit", hw[0].navi_context(), True)
            else:
                return None

    class IPHostnameTreffer(IPAddressTreffer):
        def suchfeld(self):
            return self._ipkunde.name

    class MailruleTreffer(object):
        def __init__(self, mailrule):
            self._mailrule = mailrule

        def kunde(self):
            return self._mailrule.kunde

        def suchfeld(self):
            return self._mailrule.quelle

        def url(self):
            return navi_url("mailadmin.edit_mailrule", self._mailrule.navi_context(), True)

    class HardwareIdTreffer(object):
        def __init__(self, hw):
            self._hw = hw

        def kunde(self):
            return self._hw.kunde

        def suchfeld(self):
            return self._hw.hardware_id

        def url(self):
            return navi_url("hardware.hw_edit", self._hw.navi_context(), True)

    class NicHandleTreffer(object):
        def __init__(self, nic):
            self._nic = nic

        def kunde(self):
            return self._nic.person.kunde

        def suchfeld(self):
            return self._nic.handle
            
    def __init__(self):
        self.fields = [
            SearchField(field_name="search", is_required=True),
            ]

    def flatten_data(self):
        return {'search': '*'}

    def search_results(self, data):
        """Gibt ein Tupel zurück: (kunden: [models.Kunde], by_person: [models.Kunde])"""
        kategorien = []
        TrefferListe = self.TrefferListe
        if data.get("search"):
            search = data.get("search", "*")
            globbed = glob_to_like(search)
            kunden = models.Kunde.objects.active()
            personen = models.Person.objects.active()
            if search.isdigit():
                kategorien.append(
                    TrefferListe(h2=_("Treffer in der Kunden-Nr."),
                                 th=_("Kunden-Nr."),
                                 qset = kunden.filter(id__exact=search).order_by("id"),
                                 filter=None,
                                 treffer_class = self.KundenNrTreffer))
            elif _domain_id_re.match(search):
                kategorien.append(
                    TrefferListe(h2=_("Treffer in der Domain-ID"),
                                th=_("Domain-ID"),
                                qset=models.Domainkunde.objects.active().filter(id=match.group(1)),
                                filter=None,
                                treffer_class=self.DomainIdTreffer
                ))
            elif _rechnung_nr_re.match(search):
                kategorien.append(
                    TrefferListe(h2=_("Treffer in der Rechnungs-Nummer"),
                                 th=_("Rechnungs-Nummer"),
                                 qset=models.Rechnung.objects.active().filter(rnr=match.group(1)),
                                 filter=None,
                                 treffer_class=self.RechnungsNrTreffer))
            else:
                kategorien.append(
                    TrefferListe(h2=_("Treffer im Kunden-Namen"),
                                    th=" ",
                                    qset = kunden.filter(name__like=globbed),
                                    filter=None,
                                    treffer_class = self.KundenNameTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer im Kunden-Alias"),
                                    th=_("Alias"),
                                    qset = models.Uucpkunde.objects.active().filter(name__like=globbed).order_by("name", "kunde"),
                                    filter=None,
                                    treffer_class = self.KundenAliasTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in Domains"),
                                    th=_("Domain"),
                                    qset = (models.Domainkunde.objects.active()
                                        .filter(domain__like=globbed)
                                        .order_by('domain')),
                                    filter=None,
                                    treffer_class = self.DomainTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in IP-Adressen"),
                                    th=_("IP-Adresse"),
                                    qset = models.Ipkunde.objects.active().filter(models.Ipkunde.q_ip_glob(search)).order_by("ip6", "id"),
                                    filter=None,
                                    treffer_class = self.IPAddressTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in IP-Hostnames"),
                                    th=_("Hostname"),
                                    qset = models.Ipkunde.objects.active().filter(name__like=globbed).order_by("ip6", "id"),
                                    filter=None,
                                    treffer_class = self.IPHostnameTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in Mailweiterleitungsregeln"),
                                    th=_("E-Mail (Quelle)"),
                                    qset = models.Mailrule.objects.active().filter(quelle__like=globbed).order_by("quelle", "id"),
                                    filter=None,
                                    treffer_class = self.MailruleTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer im Personen-Benutzernamen"),
                                    th=_("Benutzername"),
                                    qset = personen.filter(user__like=globbed),
                                    filter=None,
                                    treffer_class = self.PersonenUserTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in der Personen-E-Mail-Adresse"),
                                    th=_("E-Mail"),
                                    qset = personen.filter(email__like=globbed).order_by("email", "id"),
                                    filter=None,
                                    treffer_class = self.PersonenEmailTreffer))
                if search.startswith("!"):
                    now = datetime.datetime.now()
                    kategorien.append(
                        TrefferListe(h2=_("Treffer in deaktivierten Domains"),
                                     th=_("Domain"),
                                     qset = (models.Domainkunde.objects
                                        .filter(domain__like=globbed[1:], ende__gt=0, ende__lt=now)
                                        .order_by('domain', '-beginn')),
                                     filter=OnlyFirstById(),
                                     treffer_class=self.DomainTreffer))
                    kategorien.append(
                        TrefferListe(h2=_("Treffer in deaktivierten IP-Adressen"),
                                     th=_("IP-Adresse"),
                                     qset = (models.Ipkunde.objects.active()
                                             .filter(models.Ipkunde.q_ip_glob(search[1:]), ende__gt=0, ende__lt=now)
                                             .order_by("ip6", "-beginn", "id")),
                                     filter=OnlyFirstById(),
                                    treffer_class = self.IPAddressTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer in der Hardware-ID"),
                                 th=_("Hardware-ID"),
                                 qset = models.Hardware.objects.active().filter(hardware_id__like=globbed).order_by("hardware_id"),
                                 filter=None,
                                 treffer_class=self.HardwareIdTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer im NIC-Handle"),
                                 th=_("Handle"),
                                 qset=models.Nic.objects.filter(handle__like=globbed).order_by("handle", "id"),
                                 filter=None,
                                 treffer_class=self.NicHandleTreffer))
                kategorien.append(
                    TrefferListe(h2=_("Treffer im Personen-Namen"),
                                    th=_("Personen-Name"),
                                    qset = personen.filter(name__like=models.Person.name_like(globbed)).order_by("name", "id"),
                                    filter = lambda person: models.Person.vergleiche_namen(search, person.name),
                                    treffer_class = self.PersonenNameTreffer))
        return kategorien

class OnlyFirstById(object):
    def __init__(self):
        self.previous_id = None

    def __call__(self, obj):
        if obj.id != self.previous_id or self.previous_id is None:
            self.previous_id = obj.id
            return True
        else:
            return False
        
        
class MitUnterkundenForm(newforms.Form):
    mit_unterkunden = newforms.BooleanField(label=ugettext(u'Unterkunden einbeziehen'), initial=True, required=False)
