# -*- encoding: utf-8 -*-

"""Dieses Modul enthält die Views zur Hardware-Behandlung.
"""
from itertools import chain

from django import oldforms
from django.http import HttpResponse, Http404, HttpResponseRedirect
from django.template import loader, RequestContext, Context
from django.shortcuts import render_to_response
from django.views.generic.create_update import update_object, create_object, delete_object
from django.conf import settings

from kundebunt.popkern import models
from kundebunt.popkern.fields import PopQuerySet
from kundebunt.popkern.navigation import navi_url, navi_url_pattern
from kundebunt.popkern.utils import domain_sort_key, partition_dict
from kundebunt.kunde_auth.decorators import service_flag_required, staff_only, staff_only_webservice, check_readonly
from kundebunt.hardware.forms import *

@staff_only
def hw_search(request, in_panel=False, kunden_name=None):
    manipulator = HWSearchForm(request.person, kunden_name)
    if request.GET or request.POST:
        data = request.REQUEST
    else:
        data = manipulator.flatten_data()
    return render_to_response(
        "hardware/hw_search.html",
        context_instance=RequestContext(request,
            {"form": oldforms.FormWrapper(manipulator, data, {}),
            "in_panel": in_panel,
            }))

@staff_only
def hw_edit_by_hardware_id(request, hw_id):
    try:
        hw = models.Hardware.objects.get(hardware_id=hw_id)
        return HttpResponseRedirect(navi_url('hardware.hw_edit', hw.navi_context(), True))
    except models.Hardware.DoesNotExist:
        raise Http404
        
@staff_only
@check_readonly
def hw_edit(request, id, in_panel=False):
    try:
        manipulator=HWChangeForm(id, request.person, in_panel)
    except models.Hardware.DoesNotExist:
        raise Http404
    if in_panel:
        template_name = "hw_edit_panel"
        copy_url = "-javascript-"
    else:
        template_name = "hw_edit"
        copy_url = navi_url("hardware.hw_copy", manipulator.original_object.navi_context(), True)
    obj = manipulator.original_object
    return update_object(
        request,
        models.Hardware,
        object_id=id,
        template_name="hardware/%s.html" % template_name,
        post_save_redirect=navi_url("hardware.%s" % template_name, {"id": id}, True),
        manipulator=manipulator,
        extra_context={
            'enthaltene_hardware': obj.enthaltene_hardware.select_related(),
            'id_prefix': "hw%s_" % id,
            'id': id,
            'rack_id': obj.rack_id,
            'in_panel': in_panel,
            'copy_url': copy_url,
            'wartungsvertraege': obj.wartungsvertraege_fuer_anzeige()
            }
        )

@staff_only_webservice
@check_readonly
def hw_edit_panel(request, id):
    return hw_edit(request, id, True)

@staff_only
@check_readonly(allow_get=False)
def hw_create(request, in_panel=False):
    if in_panel:
        template_name = "hw_create_panel"
        post_save_redirect = navi_url_pattern("hardware.hw_edit_panel", RequestContext(request))
    else:
        template_name = "hw_edit"
        post_save_redirect = navi_url_pattern("hardware.hw_edit", RequestContext(request))
    manipulator=HWAddForm(request.person, in_panel)
    return create_object(
        request,
        models.Hardware,
        template_name="hardware/%s.html" % template_name,
        manipulator=manipulator,
        extra_context={
            'id_prefix': "hwnew_",
            'in_panel': in_panel,
            },
        post_save_redirect=post_save_redirect,
        )

@staff_only_webservice
@check_readonly(allow_get=False)
def hw_create_panel(request):
    return hw_create(request, True)

@staff_only
@check_readonly(allow_get=False)
def hw_copy(request, id, in_panel=False):
    if in_panel:
        template_name = "hw_create_panel"
        post_save_redirect = navi_url_pattern("hardware.hw_edit_panel", RequestContext(request))
    else:
        template_name = "hw_edit"
        post_save_redirect = navi_url_pattern("hardware.hw_edit", RequestContext(request))
    manipulator=HWCopyForm(id, request.person, in_panel)
    return create_object(
        request,
        models.Hardware,
        template_name="hardware/%s.html" % template_name,
        manipulator=manipulator,
        extra_context={
            'id_prefix': "hwnew_",
            'copying': True,
            'in_panel': in_panel,
            },
        post_save_redirect=post_save_redirect,
        )

@staff_only_webservice
@check_readonly(allow_get=False)
def hw_copy_panel(request, id):
    return hw_copy(request, id, True)

@staff_only
@check_readonly
def hw_wartungsvertrag_edit(request, id, wv_id, in_panel=False):
    manipulator = HWWartungsvertragChangeForm(wv_id, request.person, in_panel)
    if in_panel:
        template_name = "hardware/hw_wartungsvertrag_panel.html"
        copy_url = '-javascript-'
        post_save_redirect = navi_url(
                "hardware.hw_wartungsvertrag_edit_panel",
                manipulator.original_object.navi_context(),
                True)
    else:
        template_name = "hardware/hw_wartungsvertrag.html"
        copy_url = navi_url("hardware.hw_wartungsvertrag_copy", manipulator.original_object.navi_context(), True)
        post_save_redirect = navi_url(
                "hardware.hw_wartungsvertrag_edit",
                manipulator.original_object.navi_context(),
                True)
    return update_object(
        request,
        models.WartungsvertragHardware,
        object_id=wv_id,
        template_name=template_name,
        post_save_redirect=post_save_redirect,
        manipulator=manipulator,
        extra_context=dict(
            hardware=manipulator.hardware_obj,
            ansprechpartner=manipulator.original_object.wartungsvertrag.ansprechpartner,
            in_panel=in_panel,
            copy_url=copy_url,
            id_prefix='wv%s_' % wv_id,
            )
        )


@staff_only_webservice
@check_readonly
def hw_wartungsvertrag_edit_panel(request, id, wv_id):
    return hw_wartungsvertrag_edit(request, id, wv_id, True)

@staff_only
@check_readonly(allow_get=False)
def hw_wartungsvertrag_create(request, id, in_panel=False):
    if in_panel:
        template_name = "hardware/hw_wartungsvertrag_panel.html"
        post_save_redirect = navi_url_pattern(
                "hardware.hw_wartungsvertrag_edit_panel",
                Context({'id': id}),
                True,
                pk_context="wv_id")
    else:
        template_name = "hardware/hw_wartungsvertrag.html"
        post_save_redirect = navi_url_pattern(
                "hardware.hw_wartungsvertrag_edit",
                Context({'id': id}),
                True,
                pk_context="wv_id")
    manipulator = HWWartungsvertragAddForm(id, request.person, in_panel)
    return create_object(
        request,
        models.WartungsvertragHardware,
        template_name=template_name,
        post_save_redirect=post_save_redirect,
        manipulator=manipulator,
        extra_context=dict(
            hardware=manipulator.hardware_obj,
            in_panel=in_panel,
            id_prefix='wvnew%s_' % id,
            )
        )

@staff_only_webservice
@check_readonly(allow_get=False)
def hw_wartungsvertrag_create_panel(request, id):
    return hw_wartungsvertrag_create(request, id, True)

@staff_only
@check_readonly(allow_get=False)
def hw_wartungsvertrag_copy(request, id, wv_id, in_panel=False):
    if in_panel:
        template_name = "hw_wartungsvertrag_panel"
        post_save_redirect = navi_url_pattern(
                "hardware.hw_wartungsvertrag_edit_panel",
                Context({'id': id}),
                True,
                pk_context="wv_id")
    else:
        template_name = "hw_wartungsvertrag"
        post_save_redirect = navi_url_pattern(
                "hardware.hw_wartungsvertrag_edit",
                Context({'id': id}),
                True,
                pk_context="wv_id")
    manipulator=HWWartungsvertragCopyForm(wv_id, request.person, in_panel)
    return create_object(
        request,
        models.WartungsvertragHardware,
        template_name="hardware/%s.html" % template_name,
        manipulator=manipulator,
        extra_context={
            'id_prefix': "hwnew_",
            'copying': True,
            'in_panel': in_panel,
            'hardware': manipulator.hardware_obj,
            'id_prefix': 'wvnew%s_' % id,
            },
        post_save_redirect=post_save_redirect,
        )


@staff_only_webservice
@check_readonly(allow_get=False)
def hw_wartungsvertrag_copy_panel(request, id, wv_id):
    return hw_wartungsvertrag_copy(request, id, wv_id, True)
