# -*- coding: UTF-8 -*-

import datetime, re
from smtplib import SMTPException
from decimal import Decimal, InvalidOperation
from copy import copy

from django import oldforms
from django.core import validators
from django.utils.safestring import mark_safe
from django.utils.html import conditional_escape
from django.contrib.auth.models import User
from django.oldforms import SaveRejected
from django.conf import settings
from django.template import loader
from django.template.context import Context
from django.utils.translation import ugettext_lazy, ugettext as _

from kundebunt.popkern import models
from kundebunt.popkern.navigation import navi_url
from kundebunt.popkern.formfields import *
from kundebunt.popkern.utils import send_mail, time_to_str, smart_repr, whitespace
from kundebunt.popkern.datasources import *

__all__ = ['HWSearchForm', 'HWChangeForm', 'HWAddForm', 'HWCopyForm', 'HWWartungsvertragChangeForm',
           'HWWartungsvertragAddForm', 'HWWartungsvertragCopyForm']

class HWSearchForm(oldforms.Manipulator):
    def __init__(self, editor_person, kunden_name=None):
        self.kunden_name = kunden_name
        self.user_is_staff = editor_person.is_staff()
        self.editor_person = editor_person
        kunden_src = KundenDataSource(editor_person)
        self.fields =  [
            SearchField('name'),
            SearchField('hardware_id'),
            oldforms.TextField('ip'),
            SearchField('ivnr'),
            SearchField('seriennr'),
            DescriptorQueryField('typ', models.Hardware, is_required=False),
            DescriptorQueryField('klasse', models.Hardware, is_required=False),
            DescriptorQueryField('lieferant', models.Hardware, is_required=False),
            DescriptorQueryField('status', models.Hardware, is_required=False),
            SearchField('info'),
            oldforms.SelectField('aktiv', choices = [('nur_aktiv', _(u'nur aktive Hardware')), ('egal', _(u'egal')), ], ),
            KundenSearchField(editor_person, "kunde",  is_required=False),
            oldforms.CheckboxField("mit_unterkunden", is_required=False),
            PersonSearchField('standort', is_required=False),
            PersonSearchField('verantwortlich', is_required=False),
            PersonSearchField('eigentuemer', is_required=False),
            SearchField('rack', is_required=False),
            ComboField(field_name='rz',
                       data_source=RZDataSource(),
                       is_required=False, null_display="------",
                       choices_only=True),
            oldforms.CheckboxField('more'),
            ]
        self["ip"].help_text=mark_safe(_(
u"""Hier kann nach FQDN oder nach IP-Adresse gefiltert werden.<br />

Neben der Eingabe einer kompletten IP-Adresse ist auch die Angabe eines Subnetzes im Format 'Netz'/'Bits', also. z.B. '10.10.0.0/22'.<br />

Wenn nach FQDN gefiltert werden soll, kann man die Jokerzeichen '<tt>*</tt>' und '<tt>?</tt>' verwenden. '<tt>?</tt>' steht fuer genau ein beliebiges Zeichen, '<tt>*</tt>' fuer beliebig viele davon. Wenn man das Feld leer laesst, wird es bei der Suche ignoriert. Hier sind ein paar Beispiele fuer sinnvolle Suchkriterien:<br />

    <span class="dd"><tt>*abc*</tt></span>
    <span class="dt">Sucht nach Eintraegen, in denen 'abc' als Teilstring irgendwo vorkommt, z.B. 'XXXabcXXX'.</span>

    <span class="dd"><tt>abc*</tt></span>
    <span class="dt">Sucht nach Eintraegen, die mit 'abc' beginnen, z.B. 'abcXXX'.</span>

    <span class="dd"><tt>*abc</tt></span>
    <span class="dt">Sucht nach Eintraegen, die auf 'abc' enden, z.B. 'XXXabc'.</span>

Ein Sonderfall ist das '<tt>-</tt>' (alleine im Eingabefeld). Damit kann man nach leeren bzw. fehlenden Eintraegen suchen. Das ist allerdings nicht bei allen Eingabefeldern sinnvoll.<br />
"""))


    def noquery(self, data):
        """Wenn True, dann findet keine Suche statt"""
        return not data or "noquery" in data or 'quick' in data and not any((data.get(field.fieldname) for field in self.fields))

    def flatten_data(self):
        data = dict([(f.field_name,'*') for f in self.fields if f.field_name != 'more'])   # alles "*"
        data['mit_unterkunden'] = True
        if self.kunden_name:
            try:
                kunde = models.Kunde.objects.get(name=self.kunden_name)
                data["kunde"] = kunde.id
            except models.Kunde.DoesNotExist:
                data["kunde"] = ''
        return data

    def search_results(self, data):
        if self.noquery(data):
            return []
        data_source = HardwareDataSource(self.editor_person, data)
        return data_source.query_set()

class HWAddForm(oldforms.Manipulator):
    """Spezialisierter Manipulator zum Anlegen/Editieren von Hardware
    """
    change = False
    _readonly_fields = ['rack', 'rz', 'id']
    _enthalten_in_re = re.compile(r"^\s*(?:#\s*)?(\d+)(\s.*)?$")

    def __init__(self, editor_person, in_panel):
        super(HWAddForm, self).__init__()
        self.editor_person = editor_person
        self.in_panel = in_panel
        self.user_is_staff = editor_person.is_staff()
        kunden_src = KundenDataSource(editor_person)
        self.ipkunde_src = IpkundeDataSource()
        self.fields =  [
            oldforms.IntegerField('id'),
            oldforms.TextField('name', is_required=True),
            oldforms.TextField('hardware_id', validator_list=[
                    validators.MatchesRegularExpression(r'^D\d{6}$', ugettext_lazy(
                        u'Falsches Format! Eine Hardware-ID beginnt mit einem D, gefolgt von exakt 6 Ziffern.'))]),
            ComboField(field_name="ip",
                       data_source=self.ipkunde_src,
                       is_required=False,
                       choices_only=False),
            oldforms.TextField('ivnr', validator_list=[
                    validators.MatchesRegularExpression(r'^\d{6,7}$', ugettext_lazy(
                        u'Falsches Format! Eine Inventarnummer besteht aus 6 oder 7 Ziffern.'))]),
            oldforms.LargeTextField('seriennr', rows=2),
            DescriptorField('typ', models.Hardware, is_required=False),
            DescriptorField('klasse', models.Hardware, is_required=False),
            ComboField('lieferant', DescriptorDataSource(models.Hardware, "lieferant"),
                       is_required=False, min_query_length=0),
            DescriptorField('status', models.Hardware, is_required=False),
            oldforms.LargeTextField('info', rows=4),
            oldforms.SelectField('aktiv', choices = [('nur_aktiv', _(u'nur aktive Hardware')), ('egal', _(u'egal')), ], ),
            KundenComboField(field_name="kunde",
                       person=editor_person,
                       is_required=True,
                       choices_only=True),
            PersonComboField(field_name='verantwortlich',
                       is_required=False, null_display="-",
                       choices_only=True),
            PersonComboField(field_name='standort',
                       is_required=False, null_display="-",
                       choices_only=True),
            PersonComboField(field_name='eigentuemer',
                       is_required=False, null_display="-",
                       choices_only=True),
            ComboField(field_name='rack',
                       data_source=RackDataSource(),
                       is_required=False, null_display="*",
                       choices_only=True),
            ComboField(field_name='rz',
                       data_source=RZDataSource(),
                       is_required=False, null_display="-",
                       choices_only=True),
            oldforms.LargeTextField('enthalten_in', rows=4),
            ]
        if not self.change:
            self.fields.extend([
                oldforms.DateField("datum_bestellung"),
                oldforms.DateField("datum_lieferung"),
                oldforms.IntegerField("ticket", is_required=True),
                oldforms.TextField("artikelbezeichnung"),
                oldforms.TextField("kostenstelle"),
                oldforms.TextField("ek_waehrung", is_required=True),
                MonetaryField("originalpreis", validator_list=[
                    validators.RequiredIfOtherFieldsGiven(['rabatt', 'ek_pop', 'ek_kunde'],
                                                          _(u'Bitte auch den Originalpreis angeben.'))]),
                MonetaryField("rabatt"),
                MonetaryField("ek_pop"),
                MonetaryField("ek_kunde"),
                MonetaryField("vk_kunde"),
                oldforms.LargeTextField("buch_notiz", rows=5),
            ])

        for fieldname in self._readonly_fields:
            self[fieldname].readonly = True
        if self.change and self.original_object.rack_id:
            self["standort"].readonly = True
            self["enthalten_in"].readonly = True
        else:
            self['enthalten_in'].help_text = _(u"Zum Aendern dieses Feldes nur die Datenbank-ID der enthaltenden Hardware eingeben.")
            if self.change and self.in_panel and self.original_object.enthalten_in:
                self['enthalten_in'].help_text = self['enthalten_in'].help_text + _(u" Ein Doppelklick im Eingabefeld führt zur übergeordneten Hardware.")
        if self.change:
            prefix = "hw%s" % self.obj_key
        else:
            prefix = "new"
        for field in self.fields:
            field.id_prefix = prefix

    def flatten_data(self):
        """returns a dict of form data (strings) for the corresponding model
        """
        return {'datum_lieferung': datetime.date.today().isoformat(), 'ek_waehrung': 'EUR'}

    def make_form_readonly(self):
        for field in self.fields:
            field.readonly = True

    def do_html2python(self, data):
        """
        Convert the data from HTML data types to Python datatypes, changing the
        object in place. This happens after validation but before storage. This
        must happen after validation because html2python functions aren't
        expected to deal with invalid input.
        """
        super(HWAddForm, self).do_html2python(data)
        for fieldname in ["typ", "klasse", "status"]:
            v = data.get(fieldname, None)
            if v != None:
                try:
                    data[fieldname] = int(v)
                except ValueError:
                    data[fieldname] = None
            else:
                data[fieldname] = None

    def save(self, data):
        if self.change:
            obj = copy(self.original_object)
            hatte_standort = obj.hat_standort()
        else:
            obj = models.Hardware()
            obj.beginn = datetime.datetime.now()
            obj.flags = 0
            hatte_standort = False
        obj.name = data["name"] or None
        obj.hardware_id = data["hardware_id"] or None
        # ip ändern geht noch nicht, das ist relativ komplex!
        #try:
            #obj.ip = data["ip"] and models.Ipkunde.objects.get(ipaddr=models.Ipkunde.aton(data["ip"]))
        #except models.Ipkunde.DoesNotExist:
            #obj.ip = models.Ipkunde.objects.create() ???
        obj.ivnr = data["ivnr"] or None
        if data["seriennr"]:
            obj.seriennr = whitespace(data["seriennr"]).strip()
        else:
            obj.seriennr = None
        obj.typ_id = data["typ"]
        obj.klasse_id = data["klasse"]
        obj.lieferant_id = models.Hardware.get_lieferant_code(data["lieferant"])
        obj.status_id = data["status"]
        obj.kunde = data["kunde"] and models.Kunde.objects.get(name__iexact=data["kunde"])
        obj.eigentuemer = data["eigentuemer"]
        obj.verantwortlich = data["verantwortlich"]
        obj.standort =  data["standort"]
        obj.info = whitespace(data["info"])
        if data.get('enthalten_in'):
            match = self._enthalten_in_re.match(data['enthalten_in'])
            enthalten_in = match.group(1)
        else:
            enthalten_in = None
        obj.enthalten_in_id = enthalten_in
        if data.get("ip", None):
            obj.ip = models.Ipkunde.objects.filter_name_or_ip(data["ip"]).active().get()
        else:
            obj.ip = None
        if obj.standort_id and obj.rack_id:
            User.objects.get(username=self.editor_person.user).message_set.create(
                message=_(
                    'Diese Hardware war in %(rz_name)s, Rack %(rack_name)s eingebaut. '
                    'Durch die Zuweisung eines Standorts hier wurde sie dort automagisch "ausgebaut".'
                )
                % {'rz_name': obj.rack.rz.name,
                   'rack_name': obj.rack.name}
                )
            obj.rack_id = None
        elif hatte_standort and not obj.hat_standort():
            obj.standort = self.editor_person
            User.objects.get(username=self.editor_person.user).message_set.create(
                message=_(
                    u'Die Hardware braucht wieder einen Standort, deswegen habe ich %s als Standort eingetragen.'
                    % smart_repr(self.editor_person.user)))
        # RT #340798: Vererbung von Verantwortlichkeit für Hardware
        if not obj.verantwortlich_id and obj.enthalten_in_id and obj.enthalten_in.verantwortlich_id:
            if not self.change or str(self.original_object.enthalten_in_id) != str(obj.enthalten_in_id):
                obj.verantwortlich_id = obj.enthalten_in.verantwortlich_id
                User.objects.get(username=self.editor_person.user).message_set.create(
                    message=_(u'Der Verantwortliche wurde automatisch von der enthaltenden Hardware uebernommen.'))
        if not self.change:
            self.send_mail(obj, data)
        obj.save()
        return obj

    def send_mail(self, obj, data):
        subject = "Lieferung: " + " | ".join(
             s
             for s in [
                data.get("lieferant"),
                data.get("ticket") and "#%d" % data["ticket"],
                data.get("datum_bestellung") and str(data.get("datum_bestellung", "")),
                data.get("artikelbezeichnung")]
             if s)
        context = Context(data)
        context.push()
        context['person'] = self.editor_person
        context['obj'] = obj
        ek_gesamt = (Decimal((data.get('originalpreis') or '0').replace(",", "."))
                     - Decimal((data.get('rabatt') or '0').replace(",", ".")))
        if ek_gesamt:
            context['ek_gesamt'] = moneyfmt(ek_gesamt, sep='', dp=',')
        else:
            context['ek_gesamt'] = ''
        sender = self.editor_person.email or ("%s@noris.net" % self.editor_person.user)
        ticket_mail = settings.TICKET_MAIL % data
        recipients = [s % data for s in settings.NEW_HW_MAIL]
        recipients.append(ticket_mail)
        body = loader.render_to_string("hardware/hw_buchhaltung_email.txt", context_instance=context)
        try:
            send_mail(subject, body, sender, recipients, fail_silently=False,
                      headers={'Reply-To': ticket_mail})
        except SMTPException:
            raise SaveRejected({'datum_bestellung': [_(
                u'Das Senden der E-Mail an die Buchhaltung schlug fehl. '
                u'Bitte etwas spaeter nochmal versuchen und eventuell die '
                u'Entwicklung kontaktieren sowie dieses Fenster offen lassen. ')]})
        User.objects.get(username=self.editor_person.user).message_set.create(
                message=_(u'Die E-Mail mit Daten fuer die Buchhaltung wurde erfolgreich versandt '
                          u'(To: %(to)s, Subject: %(subject)s)') % dict(to=u", ".join(recipients), subject=subject))

    def get_hw_url(self, hw):
        """Die URL entweder zur Anzeige im Tab-Panel, oder für ein separates Fenster
        """
        if self.in_panel:
            return u'javascript:ResultGridLoader.showDetails(%s, %s)' % (hw.id, smart_repr(hw.name))
        else:
            return hw.get_absolute_url()


    def validate_hardware_id(self, field_data, data):
        """hardware_id ist eindeutig"""
        qset = models.Hardware.objects.filter(hardware_id__iexact=field_data)
        if self.change:
            qset = qset.exclude(id__exact=self.obj_key)
        if qset:
            hw = qset[0]
            raise validators.ValidationError(mark_safe(
                _(u'Die hardware_id ist bereits fuer <a href="%s">diese Hardware</a> vergeben.')
                % self.get_hw_url(hw)))

    def validate_ip(self, field_data, data):
        """
        Zunächst muss die IP entweder als numerische IP oder als Rechnername schon in
        ipkunde eingetragen sein. Zusätzlich darf die IP nicht bereits an eine andere Hardware
        vergeben sein.
        """
        #try:
            #validators.isValidIPAddress4(field_data, data)
            ## ok - es ist eine numerische IP
            #ipkunde_ids = models.Ipkunde.objects.active().filter(ipaddr__exact=models.Ipkunde.aton(field_data))
        #except validators.ValidationError:
            ## es muss wohl ein FQDN sein.
            #ipkunde_ids = self.ipkunde_src.query_set().filter(name__iexact=field_data)

        ipkunde_ids = models.Ipkunde.objects.filter_name_or_ip(field_data).active().valuelist("id")
        if not ipkunde_ids:
            raise validators.ValidationError(
                _(u'"%s" ist nicht in der Datenbank eingetragen.'
                  u'Bitte entweder eine vorhandene numerische IP-Adresse (z.B. "1.2.3.4") oder einen '
                  u'in die Datenbank eingetragenen FQDN eingeben. '
                 )
                 % field_data
                )
        hw_qset = models.Hardware.objects.filter(ip__in=ipkunde_ids)
        if self.change:
            hw_qset = hw_qset.exclude(id__exact=self.obj_key)
        if hw_qset:
            raise validators.ValidationError(mark_safe(
                _(u'Die IP "%(ip)s" ist bereits an <a href="%(hwlink)s">diese Hardware</a> vergeben')
                % { 'ip': conditional_escape(field_data),
                    'hwlink': self.get_hw_url(hw_qset[0]) }
                ))


    def validate_ivnr(self, field_data, data):
        """ivnr ist eindeutig"""
        qset = models.Hardware.objects.filter(ivnr__iexact=field_data)
        if self.change:
            qset = qset.exclude(id__exact=self.obj_key)
        if qset:
            hw = qset[0]
            raise validators.ValidationError(mark_safe(
                _(u'Die Inventar-Nr. ist bereits fuer <a href="%s">diese Hardware</a> vergeben.')
                % self.get_hw_url(hw)))

    @always_test
    def validate_originalpreis(self, field_data, data):
        """ek_gesamt = ek_kunde + ek_pop"""
        def _dec(fieldname):
            s = data.get(fieldname)
            if s:
                return Decimal(s.replace(",", "."))
            else:
                return Decimal(0)
        try:
            if ((data.get("ek_pop") or data.get("ek_kunde"))
                    and not _dec("originalpreis") - _dec("rabatt") == _dec("ek_kunde") + _dec("ek_pop")):
                raise validators.ValidationError(
                    _(u'Mit den Einkaufspreisen stimmt etwas nicht. '
                    'Es muss gelten: Originalpreis - Rabatt = EK POP + EK Kunde'))
        except InvalidOperation:
            return

    def validate_standort(self, field_data, data):
        if self.change and self.original_object.rack_id:
            raise validators.ValidationError(
                _(u'Die Hardware ist bereits in einem Rack eingebaut, sie kann nicht gleichzeitig einen eigenen Standort haben.'))

    def validate_enthalten_in(self, field_data, data):
        if data.get('standort'):
            raise validators.ValidationError(
                _(u'Die Hardware ist hat bereits einen Standort, sie kann nicht gleichzeitig '
                  u'in anderer Hardware eingebaut sein.'))
        if self.change and self.original_object.rack_id:
            raise validators.ValidationError(
                _(u'Die Hardware ist bereits in einem Rack eingebaut, sie kann nicht gleichzeitig '
                  u'in anderer Hardware eingebaut sein.'))
        match = self._enthalten_in_re.match(field_data)
        if not match:
            raise validators.ValidationError(_(u'Bitte hier eine (numerische) ID eingeben.'))
        id = int(match.group(1))
        try:
            hw = models.Hardware.objects.get(id=id)
        except models.Hardware.DoesNotExist:
            raise validators.ValidationError(_(u'Eine Hardware mit der ID %d gibt es nicht.') % id)
        # ist enthalten_in selbstbezüglich?
        if self.change:
            seen_ids = [int(self.obj_key)]
            while hw:
                if hw.id in seen_ids:
                    raise validators.ValidationError(
                        _(u'Das kann ja wohl nicht sein: Hier ist Hardware in sich selber eingebaut.'))
                seen_ids.append(hw.id)
                hw = hw.enthalten_in

    def get_advisories(self, data):
        advisories = {}
        if data.get("seriennr"):
            # Seriennr. ist im Normalfall eindeutig
            qset = models.Hardware.objects.filter(seriennr__iexact=data["seriennr"])
            if self.change:
                qset = qset.exclude(id__exact=self.obj_key)
            if qset:
                hw = qset[0]
                advisories['seriennr'] = [mark_safe(
                    _(u'Die gleiche Serien-Nr. gibt es bei <a href="%s">dieser Hardware</a>. Kann natuerlich Zufall sein ...')
                    % self.get_hw_url(hw))]
        if self.change and not data.get("standort") and not data.get("rack") and not data.get("enthalten_in"):
            advisories['standort'] = [_(u'Die Hardware hat noch keinen Standort. Das wird der Buchhaltung nicht gefallen.')]
        return advisories


class HWChangeForm(HWAddForm):
    change = True

    def __init__(self, obj_key, editor_person, in_panel):
        self.obj_key = obj_key
        self.original_object = models.Hardware.objects.get(pk=obj_key)
        super(HWChangeForm, self).__init__(editor_person, in_panel)

    def flatten_data(self):
        obj = self.original_object
        return dict(
            id = obj.id,
            name = obj.name,
            hardware_id = obj.hardware_id,
            ip = obj.ip and obj.ip.ipaddr_display(),
            ivnr = obj.ivnr,
            seriennr = obj.seriennr,
            typ = obj.typ_id,
            klasse = obj.klasse_id,
            lieferant = obj.get_lieferant_display(),
            status = obj.status_id,
            info = obj.info,
            kunde = obj.kunde.name,
            eigentuemer = obj.eigentuemer,
            verantwortlich = obj.verantwortlich,
            standort = obj.standort,
            rack = obj.rack and obj.rack.name,
            rz = obj.rack and obj.rack.rz.name,
            enthalten_in = obj.enthalten_in and str(obj.enthalten_in),
            )

class HWCopyForm(HWChangeForm):
    change=False

    def flatten_data(self):
        data = super(HWCopyForm, self).flatten_data()
        data["seriennr"] = data['hardware_id'] = data['ivnr'] = data['ip'] = data['id'] = ''
        data['ek_waehrung'] = 'EUR'
        data['datum_lieferung'] = datetime.date.today().isoformat()
        return data

class HWWartungsvertragAddForm(oldforms.Manipulator):
    """Spezialisierter Manipulator zum Anlegen/Editieren von Hardware
    """
    change = False
    _readonly_fields = ['rack', 'rz', 'id']
    _enthalten_in_re = re.compile(r"^\s*(?:#\s*)?(\d+)(\s.*)?$")

    def __init__(self, hardware_id, editor_person, in_panel):
        super(HWWartungsvertragAddForm, self).__init__()
        self.hardware_obj = models.Hardware.objects.get(pk=hardware_id)
        self.editor_person = editor_person
        self.in_panel = in_panel
        self.user_is_staff = editor_person.is_staff()
        self.fields =  [
            WartungsvertragComboField(
                    field_name='wartungsvertrag',
                    is_required=True,
                    choices_only=True),
            oldforms.DateField('beginn', is_required=True,
                validator_list=[isValidUnixDateTime(isValidUnixDateTime.parse_date)]),
            oldforms.DateField('ende',
                validator_list=[isValidUnixDateTime(isValidUnixDateTime.parse_date)])
            ]
        if self.change:
            self["wartungsvertrag"].readonly = True
            self.fields.extend([
                oldforms.TextField("ansprechpartner_name", readonly=True),
                oldforms.TextField("ansprechpartner_fon", readonly=True),
                oldforms.TextField("ansprechpartner_email", readonly=True),
                oldforms.TextField("sla", readonly=True),
                oldforms.LargeTextField("beschreibung", readonly=True, rows=15)
                ])
            prefix = "wv%s_" % self.obj_key
        else:
            prefix = "wvnew%s_" % hardware_id
        for field in self.fields:
            field.id_prefix = prefix


    def flatten_data(self):
        """returns a dict of form data (strings) for the corresponding model
        """
        return {'beginn': datetime.date.today()}

    def validate_ende(self, field_data, data):
        if data.get('beginn') and data["beginn"] > data["ende"]:
            raise validators.ValidationError(_("Das uebliche Konzept von 'Beginn' und 'Ende' sieht vor, dass der Beginn vor dem Ende liegt. Sind hier vielleicht versehentlich Beginn und Ende vertauscht?"))

    def save(self, data):
        if self.change:
            obj = self.original_object
        else:
            obj = models.WartungsvertragHardware()
            obj.hardware = self.hardware_obj
            obj.wartungsvertrag = data.get('wartungsvertrag')
        obj.beginn = data.get('beginn')
        obj.ende = data.get('ende')
        obj.save()
        return obj

class HWWartungsvertragChangeForm(HWWartungsvertragAddForm):
    change = True

    def __init__(self, obj_key, editor_person, in_panel):
        self.obj_key = obj_key
        self.original_object = models.WartungsvertragHardware.objects.get(pk=obj_key)
        super(HWWartungsvertragChangeForm, self).__init__(self.original_object.hardware_id, editor_person, in_panel)

    def flatten_data(self):
        obj = self.original_object
        wartungsvertrag = obj.wartungsvertrag
        person = wartungsvertrag.ansprechpartner
        return dict(
            wartungsvertrag=wartungsvertrag,
            beginn = time_to_str(obj.beginn),
            ende = time_to_str(obj.ende),
            ansprechpartner_name = person and person.name,
            ansprechpartner_fon = person and person.fon,
            ansprechpartner_email = person and person.email,
            sla = wartungsvertrag.sla,
            beschreibung = wartungsvertrag.beschreibung,
            )

class HWWartungsvertragCopyForm(HWWartungsvertragChangeForm):
    change=False

    def flatten_data(self):
        data = super(HWWartungsvertragCopyForm, self).flatten_data()
        return data
