# -*- encoding: UTF-8 -*-

from kundebunt.kunde_auth.decorators import staff_only, check_readonly

from datetime import date
from django.conf import settings
from django.core.urlresolvers import reverse
from django.db import connection
from django.http import Http404
from django.utils.translation import ugettext as _

from django.http import HttpResponseNotFound, HttpResponse, HttpResponseForbidden, HttpResponseRedirect
from django.template import loader, RequestContext
from kundebunt.acct.forms import AcctForm, monatsanfang
from kundebunt.popkern import models
from kundebunt.popkern.updatelog import log_update, DELETE

def _acct_index(ticket_nr, hash, seq):
    """Sucht die Daten für "weitere Accounting-Einträge" zusammen und
    gibt diese Liste zurück. Kann dann als Kontext-Variable "acct_list"
    für das Template "part_acct_index.html" verwendet werden.
    """
    _monatsanfang = monatsanfang()
    cursor = connection.cursor()
    cursor.execute(
        "SELECT acct.hash, acct.seq, acct.jjmm, acct.tt, acct.dienst, acct.bytes, "
        " assoc.info, dienst.info as dienst "
        "FROM acct JOIN rt_billing rtb on acct.hash=rtb.hash and acct.seq=rtb.seq "
        "LEFT JOIN acctassoc assoc on assoc.hash=acct.hash and assoc.seq=acct.seq "
        "JOIN dienst on dienst.id=acct.dienst "
        "WHERE rtb.ticket=%d" % ticket_nr
    )
    cols = [x[0] for x in cursor.description]
    acct_list = [
        x for x in
            sorted(
                (dict(zip(cols, row)) for row in cursor.fetchall()),
                key=lambda x:(x["jjmm"],x["tt"],x["seq"],x["hash"]))
        if x["hash"] != hash or x["seq"] != seq
    ]
    for x in acct_list:
        x["datum"] = date(x["jjmm"] // 100, x["jjmm"] % 100, x["tt"])
        x["arbeitszeit"] = "%01d:%02d" % (x["bytes"]/3600, (x["bytes"]/60) % 60)
        x["edit_url"] = reverse(edit_acct, args=(x["hash"], x["seq"]))
        x["is_editable"] = x["datum"] >= _monatsanfang
    return acct_list

def _ticket_url(ticket_nr):
    """Ermittelt die URL von AgentTicketZoom zu einer ticket_nr (integer)
    """
    return settings.TICKET_URL % ticket_nr

@staff_only
def edit_acct(request, hash, seq):
    try:
        hash = int(hash)
        seq = int(seq)
        acct = models.Acct.objects.get(hash=hash, seq=seq)
        rtb = models.RtBilling.objects.get(hash=hash, seq=seq)
        acctassoc = models.Acctassoc.objects.get(hash=hash, seq=seq)
    except ValueError:
        raise Http404
    except models.Acct.DoesNotExist:
        raise Http404
    except models.RtBilling.DoesNotExist:
        raise Http404  # acct ohne Tickets hier nicht bearbeiten
    except models.Acctassoc.DoesNotExist:
        acctassoc = models.Acctassoc.objects.create(
            hash=hash, seq=seq, info='')
    if request.method == 'POST':
        if not acct.is_editable():
            return HttpResponseForbidden('Buchungsmonat ist bereits abgeschlossen.')
        if request.person.has_pwuse_flag("readonly"):
            return HttpResponseForbidden('readonly flag fuer Person gesetzt.')
        form = AcctForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            datum = data["datum"]
            acct.jjmm = datum.year * 100 + datum.month
            acct.tt = datum.day
            acct.dienst = models.Dienst.objects.get(id=data["dienst"])
            acct.bytes = data["arbeitszeit"]
            acct.save()
            acctassoc.info = data["text"]
            acctassoc.acctinfo_id = None
            acctassoc.save()
            return HttpResponseRedirect(request.path)
    else:
        if acctassoc.info:
            text = acctassoc.info
        elif acctassoc.acctinfo:
            text = acctassoc.acctinfo.bla
        else:
            text = u''
        form = AcctForm(initial=dict(
            kunde = acct.kunde.name,
            datum = date(acct.jjmm // 100, acct.jjmm % 100, acct.tt),
            dienst = acct.dienst.id,
            text = text,
            arbeitszeit = acct.bytes))
    template = loader.get_template("acct/edit.html")
    content = template.render(RequestContext(
            request,
            {'form': form,
             'object': acct,
             'readonly': not acct.is_editable() or request.person.has_pwuse_flag("readonly"),
             'acct_list': _acct_index(rtb.ticket, hash, seq),
             'ticket': rtb.ticket,
             'ticket_url': _ticket_url(rtb.ticket),
             'kunde': acct.kunde,
            }))
    return HttpResponse(content)


@staff_only
@check_readonly
def add_acct(request, ticket_nr, kunden_name=None):
    try:
        if kunden_name is None:
            kunden_name = request.GET["kunde"]
        kunde = models.Kunde.objects.get(name=kunden_name)
        ticket_nr = int(ticket_nr)
    except (models.Kunde.DoesNotExist, KeyError), err:
        return HttpResponseForbidden("wrong or missing GET parameter: kunde")
    except ValueError:
        return HttpResponseForbidden("ticket_nr not int")
    if request.method == 'POST':
        form = AcctForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            datum = data["datum"]
            acct = models.Acct.objects.create(
                kunde = kunde,
                jjmm = datum.year * 100 + datum.month,
                tt = datum.day,
                dienst = models.Dienst.objects.get(id=data["dienst"]),
                bytes = data["arbeitszeit"],
                pakete = 1,
                quelle = models.Acct.get_quelle_descr(settings.KUNDE_KONFIG["WDESCR"]),
                dest = models.Acct.get_dest_descr(settings.KUNDE_KONFIG["DEFAULTKLASSE"]),
            )
            rtb = models.RtBilling.objects.create(
                hash = acct.hash,
                seq = acct.seq,
                ticket = ticket_nr,
            )
            acctassoc = models.Acctassoc.objects.create(
                hash = acct.hash,
                seq = acct.seq,
                info = data["text"],
            )
            return HttpResponseRedirect(
                reverse(edit_acct, args=(acct.hash, acct.seq)))
    else:
        form = AcctForm(initial=dict(
            kunde=kunde.name,
            text=request.GET.get("text", ""),
            datum=date.today()
        ))
    template = loader.get_template("acct/add.html")
    content = template.render(RequestContext(
            request,
            {'form': form,
             'acct_list': _acct_index(ticket_nr, None, None),
             'ticket': ticket_nr,
             'ticket_url': _ticket_url(ticket_nr),
            }))
    return HttpResponse(content)

@staff_only
@check_readonly
def delete_acct(request, hash, seq):
    try:
        hash = int(hash)
        seq = int(seq)
        acct = models.Acct.objects.get(hash=hash, seq=seq)
        rtb = models.RtBilling.objects.get(hash=hash, seq=seq)
        acctassoc = models.Acctassoc.objects.get(hash=hash, seq=seq)
    except ValueError:
        raise Http404
    except models.Acct.DoesNotExist:
        raise Http404
    except models.RtBilling.DoesNotExist:
        raise Http404  # acct ohne Tickets hier nicht bearbeiten
    if request.method == 'POST':
        if not acct.is_editable():
            return HttpResponseForbidden('Buchungsmonat ist bereits abgeschlossen.')
        if request.person.has_pwuse_flag("readonly"):
            return HttpResponseForbidden('readonly flag fuer Person gesetzt.')
        form = AcctForm(request.POST)
        if form.is_valid():
            acctassoc.delete()
            rtb.delete()
            acct.delete()
            return HttpResponseRedirect(reverse(add_acct, args=(rtb.ticket, acct.kunde.name)))
    else:
        if not acct.is_editable():
            return HttpResponseRedirect(reverse(edit_acct, args=(hash,seq)))
        if acctassoc.info:
            text = acctassoc.info
        elif acctassoc.acctinfo:
            text = acctassoc.acctinfo.bla
        else:
            text = u''
        form = AcctForm(initial=dict(
            kunde=acct.kunde.name,
            datum = date(acct.jjmm // 100, acct.jjmm % 100, acct.tt),
            dienst = acct.dienst.id,
            text = text,
            arbeitszeit = acct.bytes))
    template = loader.get_template("acct/delete.html")
    content = template.render(RequestContext(
            request,
            {'form': form,
             'object': acct,
             'readonly': not acct.is_editable() or request.person.has_pwuse_flag("readonly"),
             'acct_list': _acct_index(rtb.ticket, hash, seq),
             'ticket': rtb.ticket,
             'ticket_url': _ticket_url(rtb.ticket),
             'kunde': acct.kunde,
            }))
    return HttpResponse(content)
