#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use lib '/opt/noris-otrs-reports';

use NorisOtrsReport::Framework 
    qw(:data :convert generate);
use NorisOtrsReport::QuelleProcessor;
use NorisOtrsReport::TemplateRenderer;
use NorisOtrsReport::SimpleXMLRenderer;
use NorisOtrsReport::DebugProcessor;
use DateTime::Duration;
use DateTime::Format::Strptime;
use Date::Manip;
use Data::Dump qw(pp);
use Getopt::Long qw(GetOptions :config no_ignore_case);
use Pod::Usage qw(pod2usage);

{
    my $DateParser = new DateTime::Format::Strptime(
            pattern => '%Y%m%d%T', 
            locale => 'de_DE', 
            time_zone => 'local',
            on_error => 'croak');
    my ($opt_von, $opt_bis, $opt_debug, $opt_woche, $opt_tag, $opt_format, @opt_queues, $opt_opcall, $opt_raw, $opt_load, $opt_kb, $opt_help);
        
    GetOptions(
        'debug|d'        => \$opt_debug, # no 'evaluate' step
        'von=s'          => \$opt_von,
        'bis=s'          => \$opt_bis,
        'woche|w'        => \$opt_woche,
        'tag|t'          => \$opt_tag,
        'format|f=s'     => \$opt_format, # html, xml, json
        'queue|q=s'      => \@opt_queues,
        'opcall'         => \$opt_opcall,
        'raw'            => \$opt_raw,
        'load=s'         => \$opt_load,
        'knowledge'      => \$opt_kb,
        'help|h'         => \$opt_help,
    ) or pod2usage( -verbose => 0, -exitval => 2 );
    
    pod2usage( -verbose => 2, -exitval => 0 ) if $opt_help;
    
    die "Please use either --knowledge or --opcall, not both at the same time"
        if $opt_kb && $opt_opcall;
    my @TicketNumbers = @ARGV;
    my ($Von, $Bis, $Duration);
   
    $Duration = DateTime::Duration->new(weeks => 1) if $opt_woche;
    $Duration = DateTime::Duration->new(days => 1) if $opt_tag;
    $Duration = DateTime::Duration->new(seconds => 0) if $opt_opcall || $opt_kb;
    if (defined $Duration && ! $opt_von && ! $opt_bis) {
        $opt_bis = "now";
    }
    if ($opt_von) {
        $Von = $DateParser->parse_datetime(ParseDate($opt_von)) 
                or die "Cannot parse --von $opt_von";
        $Bis = $Von + $Duration if defined $Duration;
    }
    if ($opt_bis) {
        $Bis = $DateParser->parse_datetime(ParseDate($opt_bis)) 
                or die "Cannot parse --bis $opt_bis";
        $Von = $Bis - $Duration if defined $Duration;
    }
    if (! $opt_opcall && ! $opt_kb && ! defined $Von && ! defined $Bis && !@TicketNumbers) {
        die "Insufficient date specification!" unless defined $Von and defined $Bis;
    }
    
    @opt_queues = (
            'sd::quelle::incident',
            'sd::quelle::problem',
        ) unless @opt_queues;
    
    my $report_type = 
        $opt_opcall ? 'opcall'
        : $opt_kb ? 'knowledge'
        : 'daily';
    
    # Renderer
    if (! $opt_format) {
        if ($opt_debug) {
            $opt_format = 'json';
        }
        elsif ($opt_raw) {
            $opt_format = 'xml';
        }
        else {
            $opt_format = 'html';
        }
    }

    my $Renderer;
    my $ExistingTickets = {};
    if ($opt_format eq 'html') {
        if ($opt_opcall) {
            $Renderer = NorisOtrsReport::TemplateRenderer->new(
                TemplatePath => 'quelle/opcall.html',
                AdditionalData => { 
                        Titel => 'OPC-Liste vom '
                                . $Bis->strftime('%F %H:%M'),
                        ExistingTickets => $ExistingTickets,
                }
            )
        }
        elsif ($opt_kb) {
            $Renderer = NorisOtrsReport::TemplateRenderer->new(
                TemplatePath => 'quelle/opcall.html',
                AdditionalData => { 
                        Titel => 'Knowledge-Base-Liste vom '
                                . $Bis->strftime('%F %H:%M'),
                        ExistingTickets => $ExistingTickets,
                }
            )
        }
        else {
            my $Titel;
            if (@TicketNumbers) {
                $Titel = "Bericht für die Tickets " 
                    . join ", ", (map "#$_", @TicketNumbers);
            }
            else {
                $Titel = 'Bericht vom '
                            . $Von->strftime('%F %H:%M')
                            . ' bis '
                            . $Bis->strftime('%F %H:%M');
            }
            $Renderer = NorisOtrsReport::TemplateRenderer->new(
                TemplatePath => 'quelle/daily.html',
                AdditionalData => { 
                        Titel => $Titel,
                        ExistingTickets => $ExistingTickets,
                }
            )
        }
    }
    elsif ($opt_format eq 'json') {
        $Renderer = NorisOtrsReport::JSONRenderer->new();
    }
    elsif ($opt_format eq 'xml') {
        $Renderer = NorisOtrsReport::SimpleXMLRenderer->new(
                AdditionalData => {
                        'report-start' => { date => convert_epoch_to_date_hash($Von->epoch)},
                        'report-end' => { date => convert_epoch_to_date_hash($Bis->epoch)},
                        'type' => $report_type,
                        version => 1,
                        subversion => 0,
                        created => {date => convert_epoch_to_date_hash(time())},
                    },
                SimpleXMLAttributes => {
                        NoAttr => 0,
                        RootName => 'ticketlist', 
                        KeyAttr => { customtext => 'key', customdate => 'key'}, 
                        ContentKey =>'value', 
#                         ValueAttr => {'created' => 'date'},
                        NoSort => 0,
                        GroupTags => {'state-history'=>'state-event', 'history' => 'item'},
                        XMLDecl => 1,
                    },
            );
    }
    else {
        die "format '$opt_format' not implemented";
    }
    
    my $Processors;
    if ($opt_debug) {
        $Processors = [NorisOtrsReport::DebugProcessor->new(
                Renderer=> $Renderer
            )];
    }
    else {
        $Processors =     $Processors = [NorisOtrsReport::QuelleProcessor->new(
                Renderer => $Renderer,
                Typ => $opt_raw ? 'raw' : $report_type,
                ExistingTickets => $ExistingTickets,
                Start => $Von->epoch(),
            )];
    }

    my $Search;
    if (@TicketNumbers) {
        $Search = [ map { TicketNumber => $_ }, @TicketNumbers ];
    }
    elsif ($opt_opcall) {
        $Search = [
                { 
                    Queues =>  \@opt_queues,
                    TicketCreateTimeOlderDate => $Bis->strftime('%F %T'),
                    TicketFreeKey12 => 'Op-Call',
                    TicketFreeText12 => 'ja',
                }
            ];
    }
    elsif ($opt_kb) {
        $Search = [
                {
                    Queues => \@opt_queues,
                    TicketCreateTimeOlderDate => $Bis->strftime('%F %T'),
                    TicketFreeKey5 => 'Knowledge-Base',
                    TicketFreeText5 => 'required',
                }
            ];
    }
    else {
        $Search = [ 
                {
                    Queues =>  \@opt_queues,
                    TicketCloseTimeNewerDate => $Von->strftime('%F %T'),
                    TicketCreateTimeOlderDate => $Bis->strftime('%F %T'),
                    StateType => ['closed'],
                },
                {
                    Queues =>  \@opt_queues,
                    StateType => ['open', 'new', 'pending auto', 'pending over', 'seen', 'hidden'],
                    TicketCreateTimeOlderDate => $Bis->strftime('%F %T'),
                }
            ];
    }
    
    my $Collectors;
    if ($opt_load) {
        $Collectors = undef;
        $Search = undef;
    }
    else {
        $Collectors = [
                \&data_otrs_ticket,
                \&data_customtext, 
                \&data_customdate,
                \&data_state_history,
                \&data_linked_tickets,
                make_data_articles(
                        qw(note-external-error note-external-reason 
                        note-external-actions note-external-opcall 
                        note-external-progression ),
                    ),
            ];
    }
    generate (
        UserID => 1,
        Search => $Search,
        LoadFixture => $opt_load,
        Conversions => $opt_debug ? [] : [
                [ Created     => \&convert_isodate_to_date_hash, 'created' ],
                [ RealTillTimeNotUsed => \&convert_epoch_to_date_hash, 'pending-until' ],
                [ CreateTime  => \&convert_isodate_to_date_hash, 'created' ],
            ],
        Collectors => $Collectors,
        Processors => $Processors,
    );
}

=head1 quelle-report

quelle-report - Erstellen von Berichten über Tickets für den Kunden quelle / quelle-contact

=head1 SYNOPSIS

quelle-contact [options]

 Optionen:
        
        --debug, -d      Zum Debuggen, es werden nur die Rohdaten geholt, ohne "eval"
        --opcall         Einen OP-Call-Bericht erstellen ("opcall" auf "ja")
        --knowledge      Einen Knowledge-Base-Bericht erstellen ("knowlege-base" auf  
                         "required")
        --von DATUM      Beginn des Berichts-Zeitraums setzen
        --bis DATUM      Ende des Berichts-Zeitraums setzen
        --woche          Berichtszeitraum ist eine Woche. Wenn weder --von noch --bis
                         angegeben ist, ist es die Woche bis gerade eben.
        --tag            Berichtszeitraum ist ein Tag. Wenn weder --von noch --bis
                         angegeben ist, sind es die 24 h bis gerade eben.
        --format, -f     Setzt das Ausgabeformat. Möglich ist html, xml oder json.
        --queue, -q      Welche Queues durchsucht werden. Optional.
        --raw            Der Kunde wollte auch "Rohdaten" geliefert bekommen. --raw
                         macht das.
        --load DATEI     Lädt Tickets aus einer Fixture-Datei und erstellt dafür einen
                         Bericht. Zum Testen gedacht. Fixture-Dateien kann man einfach
                         mut "--debug --format=json" erstellen.
        --help, -h       Anzeige dieser Gebrauchsanleitung

=cut
